function init_get_surface_exposure_ages()

% This script computes exposure ages for Kay Peak ridge samples and saves a
% data file for later use. Note: for C-14 data, computes ages 5 times to
% generate 68% and 95% confidence intervals from non-normally-distributed
% uncertainties on C-14 concentration. 
%
% Greg Balco
% Berkeley Geochronology Center
% Feb, 2022
% 
% Accompanies a paper entitled 'REVERSIBLE ICE SHEET THINNING IN THE
% AMUNDSEN SEA EMBAYMENT DURING THE LATE HOLOCENE,' by Balco, Brown,
% Nichols, Venturelli, and others. 
%
% This code is provided solely for review of the accompanying paper. It
% is not licensed for other use or distribution. 

% Be-10 data from Kay Peak from the ICE-D database as of 20220128:
disp('Ingesting sample/Be-10 data');
input_string_10 = ['JJ2015-KAY-101 -75.22058 -110.95982 334 ant  3.5 2.70 0.9993 0 2016;' ...
    'JJ2015-KAY-101 Be-10 quartz 1.812e+05 4.326e+03 NIST_27900;' ...
    'JJ2015-KAY-107 -75.21674 -110.96292 160 ant  3.2 2.70 0.9961 0 2016;' ...
    'JJ2015-KAY-107 Be-10 quartz 7.843e+04 2.286e+03 NIST_27900;' ...
    'JJ2015-KAY-108 -75.21687 -110.96308 167 ant  4.3 2.70 0.9942 0 2016;' ...
    'JJ2015-KAY-108 Be-10 quartz 1.308e+05 3.006e+03 NIST_27900;' ...
    'JS2010-R9-3-2 -75.22083 -110.95990 338 ant  3.1 2.70 0.9996 0 2010;' ...
    'JS2010-R9-3-2 Be-10 quartz 7.501e+04 2.250e+03 07KNSTD;' ...
    'JS2010-R9-3-5 -75.22062 -110.95950 338 ant  3.8 2.70 0.9993 0 2010;' ...
    'JS2010-R9-3-5 Be-10 quartz 1.867e+05 8.401e+03 07KNSTD;' ...
    'JS2010-R9-3-8 -75.22047 -110.96117 326 ant  2.7 2.70 0.9422 0 2010;' ...
    'JS2010-R9-3-8 Be-10 quartz 8.366e+04 2.761e+03 07KNSTD;' ...
    'JS2010-R9-3-10 -75.22055 -110.96105 324 ant  2.6 2.70 0.9767 0 2010;' ...
    'JS2010-R9-3-10 Be-10 quartz 8.630e+04 2.848e+03 07KNSTD;' ...
    'JS2010-R9-3-11 -75.22048 -110.96082 327 ant  2.7 2.70 0.9413 0 2010;' ...
    'JS2010-R9-3-11 Be-10 quartz 1.064e+05 2.767e+03 07KNSTD;'];

% Also Kay Cone:

input_cone = ['JJ2015-CIN-101 -75.21943 -111.02317 239 ant  4.3 2.65 0.9998 0 2015;' ...
'JJ2015-CIN-101 Be-10 quartz 4.986e+04 1.384e+03 07KNSTD;' ...
'JJ2015-CIN-102 -75.21943 -111.02316 239 ant  3.1 2.65 0.9989 0 2015;' ...
'JJ2015-CIN-102 Be-10 quartz 5.093e+04 1.377e+03 07KNSTD;' ...
'JJ2015-CIN-103 -75.21941 -111.02237 238 ant  2.9 2.65 0.9998 0 2015;' ...
'JJ2015-CIN-103 Be-10 quartz 4.569e+04 1.893e+03 07KNSTD;' ...
'JJ2015-CIN-104 -75.21933 -111.02158 233 ant  3.8 2.65 0.9954 0 2015;' ...
'JJ2015-CIN-104 Be-10 quartz 4.690e+04 1.321e+03 07KNSTD;' ...
'JJ2015-CIN-105 -75.21925 -111.02053 229 ant  3.1 2.65 0.9996 0 2015;' ...
'JJ2015-CIN-105 Be-10 quartz 4.781e+04 2.168e+03 07KNSTD;' ...
'JJ2015-CIN-106 -75.21925 -111.02053 229 ant  4.2 2.65 0.9997 0 2015;' ...
'JJ2015-CIN-106 Be-10 quartz 4.565e+04 1.935e+03 07KNSTD;' ...
'JJ2015-CIN-107 -75.21903 -111.01974 225 ant  3.6 2.65 0.9994 0 2015;' ...
'JJ2015-CIN-107 Be-10 quartz 4.645e+04 1.262e+03 07KNSTD;' ...
'JJ2015-CIN-108 -75.21652 -111.01973 181 ant  3.3 2.65 0.9994 0 2015;' ...
'JJ2015-CIN-108 Be-10 quartz 3.986e+04 1.353e+03 07KNSTD;' ...
'JJ2015-CIN-109 -75.21636 -111.01992 178 ant  5.7 2.65 0.9995 0 2015;' ...
'JJ2015-CIN-109 Be-10 quartz 3.912e+04 1.503e+03 07KNSTD;' ...
'JJ2015-CIN-110 -75.21636 -111.01992 178 ant  5.1 2.65 0.9995 0 2015;' ...
'JJ2015-CIN-110 Be-10 quartz 4.089e+04 1.194e+03 07KNSTD;' ...
'JJ2015-CIN-111 -75.21632 -111.01885 180 ant  3.3 2.65 0.9992 0 2015;' ...
'JJ2015-CIN-111 Be-10 quartz 4.233e+04 1.275e+03 07KNSTD;' ...
'JJ2015-CIN-112 -75.21627 -111.01796 179 ant  2.8 2.65 0.9994 0 2015;' ...
'JJ2015-CIN-112 Be-10 quartz 4.276e+04 1.245e+03 07KNSTD;'];

% Add Kay Cone data
input_string_10 = [input_string_10 input_cone];

% Parse this to extract sample elevations

t10.sample_name = {};
t10.elv_m = [];
t10.t_St = [];
t10.dtint_St = [];
t10.t_LSDn = [];
t10.dtint_LSDn = [];

split_input_string_10 = split(input_string_10,';');
for a = 1:length(split_input_string_10)
    thisline = split_input_string_10{a};
    if contains(thisline,'Be-10') || isempty(thisline)
        % Not a sample data line, do nothing
    else
        splitline = split(thisline);
        t10.sample_name{end+1} = splitline{1};
        t10.elv_m(end+1) = str2num(splitline{4});
    end
end

% Send this to the online exposure age calculator and get some results. 
disp('Getting Be-10 exposure ages from server');
% URL of service
calc_url = 'http://hess.ess.washington.edu/cgi-bin/matweb';

% Script to run
calc_mfile = 'age_input_v3';

% Use webread with other inputs as name,value pairs
t10_XML = webread(calc_url,'mlmfile',calc_mfile,'text_block',input_string_10, ...
    'reportType','XML','resultType','long','plotFlag','no','summary','no');

% Hokey string matching to extract ages from XML
t10_St_strings = regexp(t10_XML,['<t10quartz_St>(.*?)</t10quartz_St>'],'tokens');
t10_dtint_St_strings = regexp(t10_XML,['<delt10quartz_int_St>(.*?)</delt10quartz_int_St>'],'tokens');
t10_LSDn_strings = regexp(t10_XML,['<t10quartz_LSDn>(.*?)</t10quartz_LSDn>'],'tokens');
t10_dtint_LSDn_strings = regexp(t10_XML,['<delt10quartz_int_LSDn>(.*?)</delt10quartz_int_LSDn>'],'tokens');
t10_samplename_strings = regexp(t10_XML,['<sample_name>(.*?)</sample_name>'],'tokens');

for a = 1:length(t10_samplename_strings)
    sindex = find(strcmp(t10_samplename_strings{a},t10.sample_name));
    t10.t_St(sindex) = str2num(t10_St_strings{a}{1});
    t10.dtint_St(sindex) = str2num(t10_dtint_St_strings{a}{1});
    t10.t_LSDn(sindex) = str2num(t10_LSDn_strings{a}{1});
    t10.t_dtint_LSDn(sindex) = str2num(t10_dtint_LSDn_strings{a}{1});
end

% Make more intelligible name
surface_t10 = t10; 

%% Now we need to do the same thing with the C-14 data, repeatedly. 

% Here are the sample data lines from the ICE-D database as of 20220128

disp('Ingesting sample data');
t14_text_lines = {'JJ2015-KAY-101 -75.220579 -110.959817 334 ant 3.47 2.7 0.9993 0 2016 ;'
    'JJ2015-KAY-102 -75.220678 -110.959501 331 ant 5 2.7 0.7218 0 2016 ;'
    'JJ2015-KAY-103 -75.219622 -110.95909 301.818999 ant 1.9999 2.7 0.9975 0 2016 ;'
    'JJ2015-KAY-104 -75.225232 -110.965218 358.744999 ant 5 2.7 1 0 2016 ;'
    'JJ2015-KAY-105 -75.216509 -110.962679 149.831999 ant 1.9999 2.7 0.9933 0 2016 ;'
    'JJ2015-KAY-106 -75.21655 -110.96267 150 ant 5 2.7 0.9852 2016 ;'
    'JJ2015-KAY-107 -75.216743 -110.962919 160.011999 ant 3.21 2.7 0.9961 0 2016 ;'
    'JJ2015-KAY-108 -75.216867 -110.963084 166.619999 ant 4.27 2.7 0.9942 0 2016 ;'
    'JJ2015-KAY-109 -75.216969 -110.963212 170.414999 ant 1.9999 2.7 0.9972 0 2016 ;'
    'JJ2015-KAY-110 -75.216759 -110.962986 161.362999 ant 1.9999 2.7 0.9967 0 2016 ;'
    'JJ2015-KAY-111 -75.216605 -110.962824 153.645999 ant 1.9999 2.7 0.9941 0 2016 ;'
    'JJ2015-KAY-112 -75.216412 -110.962668 145.878999 ant 5 2.7 0.9974 0 2016 ;'
    'JS2010-R9-3-2 -75.22083 -110.9599 338 ant 3.15 2.7 0.9996 0 2010 ;'
    'JS2010-R9-3-5 -75.22062 -110.9595 338 ant 3.8 2.7 0.9993 0 2010 ;'
    'JS2010-R9-3-7 -75.22072 -110.95913 334 ant 5 2.7 0.994 0 2010 ;'
    'JS2010-R9-3-8 -75.22047 -110.96117 326 ant 2.74 2.7 0.9422 0 2010 ;'
    'JS2010-R9-3-10 -75.22055 -110.96105 324 ant 2.58 2.7 0.9767 0 2010 ;'
    'JS2010-R9-3-11 -75.22048 -110.96082 327 ant 2.69 2.7 0.9413 0 2010 ;'
    '19-KP-001-BR -75.21636667 -110.9624333 140 ant 8.2 2.7 0.9964 0 2020 ;'
    '19-KP-002-BR -75.21651667 -110.9626667 147 ant 8.2 2.7 0.9964 0 2020 ;'
    '19-KP-003-BR -75.21663333 -110.9629333 151 ant 8.2 2.7 0.9964 0 2020 ;'
    '19-KP-004-BR -75.21626 -110.96298 161 ant 5.6 2.7 0.9951 0 2020 ;'
    '19-KP-005-BR -75.21674 -110.96289 165 ant 7 2.7 0.9951 0 2020 ;'
    '19-KP-006-BR -75.21695 -110.96312 181 ant 5.3 2.7 0.9951 0 2020 ;'
    '19-KP-007-BR -75.21798 -110.96249 209 ant 3.4 2.7 0.9933 0 2020 ;'
    '19-KP-008-BR -75.21815 -110.96219 223 ant 6.4 2.7 0.9933 0 2020 ;'
    '19-KP-009-BR -75.21854 -110.96127 236 ant 6.1 2.7 0.993 0 2020 ;'
    '19-KP-010-BR -75.21881 -110.96068 248 ant 3.2 2.7 0.9936 0 2020 ;'
    '19-KP-011-BR -75.21927 -110.95924 281 ant 4.9 2.7 0.9936 0 2020 ;'
    '19-KP-012-BR -75.22079 -110.9593 338 ant 5.9 2.7 0.9996 0 2020 ;'
    'JJ2015-CIN-102 -75.21943 -111.02316 239 ant 3.1 2.65 0.9989 0 2015;'
    'JJ2015-CIN-108 -75.21652 -111.01973 181 ant 3.3 2.65 0.9994 0 2015;'
    'JJ2015-CIN-112 -75.21627 -111.01796 179 ant 2.8 2.65 0.9994 0 2015;'};

% Load available C-14 concentrations
disp('Loading C-14 concentrations');
load data_C14

% Match up sample names, elevations, and exposure-age data. 
for a = 1:length(d14_surface.sample_names)
    t14.sample_name{a} = d14_surface.sample_names{a};
    sample_line = t14_text_lines{find(contains(t14_text_lines,d14_surface.sample_names{a}))};
    % Also we need the elevation
    split_line = split(sample_line);
    t14.elv_m(a) = str2num(split_line{4});
    % Assemble calculator input text. Note: calculate 5 exposure ages to
    % match confidence bounds
    this_input_text = sample_line;
    for b = 1:5
        this_input_text = [this_input_text d14_surface.sample_names{a} ' C-14 quartz ' sprintf('%0.0f',d14_surface.Natomsg_bounds(a,b)) ' 1000;'];
    end
    disp(['Sending data for ' t14.sample_name{a} ' to server']); 
    current_XML = webread(calc_url,'mlmfile',calc_mfile,'text_block',this_input_text,'reportType','XML','resultType','long','plotFlag','no','summary','no');
    t14_St_strings = regexp(current_XML,['<t14quartz_St>(.*?)</t14quartz_St>'],'tokens');
    t14_LSDn_strings = regexp(current_XML,['<t14quartz_LSDn>(.*?)</t14quartz_LSDn>'],'tokens');
    t14.t_St_bounds(a,:) = [str2num(t14_St_strings{1}{1}) str2num(t14_St_strings{2}{1}) str2num(t14_St_strings{3}{1}) str2num(t14_St_strings{4}{1}) str2num(t14_St_strings{5}{1})];
    t14.t_LSDn_bounds(a,:) = [str2num(t14_LSDn_strings{1}{1}) str2num(t14_LSDn_strings{2}{1}) str2num(t14_LSDn_strings{3}{1}) str2num(t14_LSDn_strings{4}{1}) str2num(t14_LSDn_strings{5}{1})];
end

disp('Saving...');
save data_exposure_ages t10 t14




