function [ijResult,pcfStack, tempSt, searchSt] = patternMatchExample(I1,I2,ijList,tsize,ssize,method)
% PATTERNMATCHEXAMPLE
% matches displacement and its precision between two different images, at
% given locations (given by a list), with a kernel of a certain size (tsize) 
% and with a given search range (ssize).  
%
% INPUT:
% I1    - 2D matrix with template image
% I2    - 2D matrix with search image, (same size as I1)
% ij    - nx2 array with coordinates of interest
% tsize - integer of the size(radius) of the template in pixels
% ssize - integer of the search extent in the other image
%
% OUTPUT:
% ijResult  - nx9 array with results, giving in each line:
%            1) intI - integer displacement in collumn direction
%            2) infJ - integer displacement in row direction
%            3) dI - sub-pixel displacement in collumn direction
%            4) dJ - sub-pixel displacement in row direction
%            5) val - correlation value
%            6) snr - signal-to-noise value
%            7) sigI - sigma in row direction
%            8) sigJ - sigma in collumn direction
%            9) rho - orientation of Gaussian
%           10) fac - normalization factor
% pcfStack  - kxkxn 3D array with raw correlation values of the whole domain
% tempSt    - lxlxn 3D array with templates of image I1
% searchSt  - mxmxn 3D array with search domains of image I2
%
% B.Altena - UiO & UU

%if nargin<6, method = 'simple'; end
if nargin<6, method = 'affine'; end

fprintf(['using normalized cross-correlation\n']);
fprintf(['also modelling the correlation peak through a Gaussian function\n']);
fprintf(['template-range is ' num2str(tsize) ', thus pixel area is ' num2str(tsize.*2+1) ' by ' num2str(tsize.*2+1) '.\n'])
fprintf(['search-range is ' num2str(ssize) ', thus pixel area is ' num2str(ssize.*2+1) ' by ' num2str(ssize.*2+1) '.\n'])

[m1,n1,~] = size(I1);
iI = round(ijList(:,1)); jI = round(ijList(:,2));
ijOut = iI<1 | iI >m1 | jI<1 | jI >n1;
if any(ijOut), fprintf('some seed points are outside the imagery. \n'); end
ijOut = (iI-ssize)<1 | (iI+ssize) >m1 | (jI-ssize)<1 | (jI+ssize) >n1;
if any(ijOut), fprintf('some search windows are outside the imagery. \n'); end

ijResult = zeros(length(ijOut),10); 
% [intI intJ dI dJ val snr sigI sigJ rho fac; ... ]

I1 = padarray(I1,[ssize ssize],0);
I2 = padarray(I2,[ssize ssize],0);
iI = iI+ssize; jI = jI+ssize;

ijSBox = [iI-ssize iI+ssize jI-ssize jI+ssize];
ijTBox = [iI-tsize iI+tsize jI-tsize jI+tsize];

% for sub-pixel localization
arrelon = @(x) x(:);
arrtrim = @(x) x(1:end-1);
[Jsub,Isub] = meshgrid([-1:+1],[-1:+1]); 
Isub = arrelon(Isub); Jsub = arrelon(Jsub);

%% core algorithm
st = ssize-tsize;
mT = tsize.*2+1; nT = mT; mT = tsize.*2+1; nT = mT;
prog = 0;
if nargout>1, pcfStack = []; tempSt = []; searchSt = []; end
for ii = 1:size(ijOut)
    prog = 100.*(ii./length(ijOut));
    
    if ijOut(ii) 
        ijResult(ii,:) = [NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN]; % write NaN's
    else
        %% grab templates
        subI1 = I1(ijTBox(ii,1):ijTBox(ii,2),ijTBox(ii,3):ijTBox(ii,4));
        subI2 = I2(ijSBox(ii,1):ijSBox(ii,2),ijSBox(ii,3):ijSBox(ii,4));
            
        if not(std(subI1(:))==0)
            ccs = normxcorr2(subI1,subI2);

            [mC,nC] = size(ccs);
            ccs = ccs((mC-1)/2-(ssize)+(tsize)+1:(mC-1)/2+(ssize)-(tsize)+1, ...
                (nC-1)/2-(ssize)+(tsize)+1:(nC-1)/2+(ssize)-(tsize)+1);
            mC = size(ccs,1);
            intPix = -floor(mC/2):floor(mC/2);      

            [val,~] = max(ccs(:));
            snr = val./mean(abs(ccs(:)));
            
            if nargout>1
                pcfStack = cat(3,pcfStack, ccs);
                tempSt = cat(3, tempSt, subI1);
                searchSt = cat(3, searchSt, subI2);
            end
            
            if not(isnan(snr))
                [iSh,jSh] = find(ccs==val);
                % it is possible there is more than one highest candidate,
                % here taking the first....
                if min(iSh,jSh)>1 & max(iSh,jSh)<mC
                    cs = ccs(iSh-1:iSh+1,jSh-1:jSh+1);
                    cs = cs-(mean(arrtrim(fftshift(cs)))); % normalize
                    cs = arrelon(max(cs,0));
                    dI = sum(Isub.*cs);
                    dJ = sum(Jsub.*cs);                  
                     %% Gaussian estimation
                     iTop = iSh(1); jTop = jSh(1);
                     [ sigI,sigJ,rho,~,fac] = estGauss( ccs, iTop, jTop, dI, dJ);
                     iSh = intPix(iSh(1))+1; jSh = intPix(jSh(1))+1;
                else
                    dI = 0; dJ = 0; val = 0; snr = 0;
                    sigI = 0; sigJ = 0; rho= 0; fac = 1; 
                end
            else
                % no data in imagery
                iSh = 0; jSh = 0;
                dI = 0; dJ = 0; val = 0; snr = 0;
                sigI = 0; sigJ = 0; rho= 0; fac = 1; 
            end
        else
            % no data in imagery
            iSh = 0; jSh = 0;
            dI = 0; dJ = 0; val = 0; snr = 0;
            sigI = 0; sigJ = 0; rho= 0; fac = 1;
        end

        %% put together
        ijResult(ii,:) = [iSh jSh dI dJ val snr sigI sigJ rho fac];
    end
end
fprintf('\n');

end



