'''
Processing code accompanying the manuscript:

The ERA5-Land Soil-Temperature Bias in Permafrost Regions

submitted to The Cryosphere for publication in 2020 by Bin Cao (c)

Contactor: Bin Cao (bin.cao@itpcas.ac.cn)

Run from Python console after setting up to set the working/download

'''

import cdsapi
import numpy as np
from os import path
from datetime import datetime, timedelta

# ==============================================================================
# CLASS & FUNCTIONS
# ==============================================================================


class ERA5LDayDownload():
    '''download hourly ERA5-Land reanalysis'''
    
    def __init__(self, dir_out, dateRange, area, variables):
        self.dir_out = dir_out
        self.rea = 'reanalysis-era5-land-monthly-means'
        self.dateRange = dateRange
        self.dates = self.makeDays()
        self.area = area
        self.var  = variables
        self.analysis = 'reanalysis-era5-land'
        
    def makeDays(self):
        '''get download dates based on given date range'''
        
        beg = self.dateRange['beg']
        end = self.dateRange['end']
        nday = (end - beg).days
        dates = [beg + timedelta(days=x) for x in range(nday+1)]
        
        return dates
    
    def makeFile(self, datei):
        '''make outfile name based on the download date'''
        
        yi   = datei.year
        moni = datei.month
        dayi = datei.day
        dnum = str(yi)+str(moni).zfill(2)+str(dayi).zfill(2)
        fname = 'era5_land_' + dnum  + '.nc'
        fname = path.join(self.dir_out, fname)
        
        return fname
    
    def makeArea(self):
        '''make download area'''
        
        areaStr = [str(self.area['north']) + '/'+ str(self.area['west'])+'/'+ 
                   str(self.area['south']) + '/'+ str(self.area['east'])]
        
        return areaStr
    
    def makeTimes(self):
        '''create hours'''
        times = ['00:00','01:00','02:00', '03:00','04:00','05:00',
                 '06:00','07:00','08:00', '09:00','10:00','11:00',
                 '12:00','13:00','14:00', '15:00','16:00','17:00',
                 '18:00','19:00','20:00', '21:00','22:00','23:00']
        
        return times
    
    def makeDate(self, datei):
        '''make datei for the outfile name'''
        
        yi   = datei.year
        moni = datei.month
        dayi = datei.day
        dname = [str(yi) + '-' + str(moni).zfill(2) + '-' + str(dayi).zfill(2)]
        
        return dname
    
    def dayDownload(self, datei):
        
        #for h in np.arange(24):
        fname = self.makeFile(datei)
        if path.isfile(fname):
            print(fname + ' ALREADY EXISTS!')
        else:
            cds = cdsapi.Client()
            # download
            cds.retrieve(
                self.analysis,
                {'variable': self.var,
                 'year':  [datei.year],
                 'month': [datei.month],
                 'day':   [datei.day],
                 'time':  self.makeTimes(), 
                 'area':  self.makeArea(), # N, W, S, E. Default: global
                 'format': 'netcdf'
                },
                fname)
        
    def retrive(self):
        
        for datei in self.dates:
            self.dayDownload(datei)
        
    

class ERA5LMonDownload():
    '''download montly ERA5-Land reanalysis'''
    
    def __init__(self, dir_out, dateRange, area, variables):
        self.dir_out = dir_out
        self.rea = 'reanalysis-era5-land-monthly-means'
        self.dateRange = dateRange
        self.yr  = self.makeYears()
        self.var = variables
        self.area = area
        
    def makeYears(self):
        '''get download dates based on given date range'''
        
        beg = self.dateRange['beg'].year
        end = self.dateRange['end'].year
        years = np.arange(beg, end+1)
        
        return years
    
    def makeOutFile(self, yr, area = 'global'):
        '''make the outfile name'''
        
        dnum  = str(yr)# + str(moni).zfill(2)
        fname = 'era5_land_' + area + '_month_' + dnum  + '.nc' 
        fname = path.join(self.dir_out, fname)
        
        return fname
    
    def makeMon(self):
        '''all months'''
        
        mons = ['01','02','03','04','05','06',
                '07','08','09','10','11','12']
        
        return mons
    
    def makeArea(self):
        '''make download area'''
        
        areaStr = [str(self.area['north']) + '/'+ str(self.area['west'])+'/'+ 
                   str(self.area['south']) + '/'+ str(self.area['east'])]
        
        return areaStr
    
    def yrDownload(self, yr):
        
        
        fname = self.makeOutFile(yr)
        
        cds = cdsapi.Client()
        cds.retrieve(
                self.rea,
                {'format':'netcdf',
                 'product_type':'monthly_averaged_reanalysis',
                 'variable': self.var,
                 'year': str(yr),
                 'month': self.makeMon(),
                 'time':'00:00',
                 'area': self.makeArea()
                 },
                fname)
    
    def retrive(self):
        
        for yr in self.yr:
            self.yrDownload(yr)
        
# ==============================================================================
# SETTING-UP 
# ==============================================================================

dir_out = '/Users/bincao/Desktop'
dateRange = {'beg': datetime(2017, 1, 1),
             'end': datetime(2017, 12, 31)}
area = {'north': 66, 'south': 62, 'west': -112, 'east': -108}
variables = ['2m_temperature',
             'snow_depth', 
             'snow_density',
             'snow_depth_water_equivalent',
             'soil_temperature_level_1', 
             'soil_temperature_level_2',
             'soil_temperature_level_3', 
             'soil_temperature_level_4'
             ]

# ==============================================================================
# DOWNLOAD
# ==============================================================================

# hourly
era5d = ERA5LDayDownload(dir_out, dateRange, area, variables)
era5d.retrive()

# monthly
era5m = ERA5LMonDownload(dir_out, dateRange, area, variables)
era5m.retrive()

