%==========================================================================
%  gs_evol_1d_vert_constT -- calculate grain size evolution in a 1-D column
%  model assuming a constant temperature at all depths.  Grain growth
%  parameters are loaded from file: bestfit_gg_params.mat.
%
%  Mark D. Behn, November 2019
%==========================================================================

clear

% Model Parameters
g = 9.8;
R = 8.314;

slope = 3;  % arbitrary slope

ub = 0;     % Assumed basal velocity (set to zero)
H = 1.0e3;  % Ice thickness 
rho = 920;  % Average Ice denisty from Gow et al. (JGR, 1997)

% Setup grid for GSE calculations
dz = 2.5;   % Verticle grid spacing (m)
z = 0:dz:H; % Vector of vertical grid values (m)

T = 253*ones(1,length(z));


%% Set Grain Size Evolution Parameters
% Initial Grain size for calculations
d1 = 0.001.*ones(1,length(z)); % meters
d3 = 0.003.*ones(1,length(z)); % meters
d5 = 0.005.*ones(1,length(z)); % meters

% Set Various Fudge Factors Associated with Austin & Evans [2007]
c = 3;                % geometric constant (=3 for spherical grains)
gamma = 0.065;        % average specific grain boundary energy 
lambda_disl = 0.01;   % fraction of work by disl creep converted 
                      %   to int. energy (=0.1 for AE07)
lambda_GBS = 0.01;    % fraction of work by GBS converted to int. energy 
                      %   (note thatthis term has been added compared to 
                      %    AE07, as they assumed that lambda_diff = 0).
Qg_wb = 42e3;  % kJ/mol (w/ bubbles based on Fit in Figure 5; Table 3 of Azuma et al.)

% Load best fitting grain growth parameters (generated in script
%   plot_gg_iterativefit_paper.m)
load bestfit_gg_params.mat


% Set Flow Law Parameters based on Goldsby & Kohlstedt (JGR, 2001)
if T > 259
    % Goldsby-Kohlstedt Constants for T > 259K
    n1 = 1.8;    % GBS
    n2 = 4;      % Dislocation creep
    A1o = 3e26;
    A2o = 6e28;
    Q1 = 192e3;
    Q2 = 180e3;  %Value wrong in K&G01
    m1 = 1.4;
else
    % Goldsby-Kohlstedt Constants for T <= 259K
    n1 = 1.8;
    n2 = 4;
    A1o = 3.9e-3;
    A2o = 4e4;  %Value wrong in K&G01
    Q1 = 49e3;
    Q2 = 60e3;
    m1 = 1.4;
end

% Set Parameters for Basal slip-accommodated GBS
n3 = 2.4;
A3o = 5.5e7;
Q3 = 60e3;

A1 = A1o.*exp(-Q1./(R.*T));
A2 = A2o.*exp(-Q2./(R.*T));

tau = rho.*g.*(H-z).*sind(slope);tau(end) = 1e-9;

d_pz = ((tau/1e6).^-1.5)./4./1e3;

d_fb = ((tau./1e6).^(n2./n1).*(A2o./A1o).*...
    (exp(-Q2./R./273)./exp(-Q1./R./273))).^(-1/1.4);

sr1 = A1o.*(d1.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));
sr0_1 = sr1 + sr2;

sr1 = A1o.*(d3.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));
sr0_3 = sr1 + sr2;

sr1 = A1o.*(d5.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));
sr0_5 = sr1 + sr2;


sr1_pz = A1o.*(d_pz.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
sr2_pz = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));

sr1o_pz = sr1_pz;
sr0_pz = sr1_pz + sr2_pz;

sr1_fb = A1o.*(d_fb.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
sr2_fb = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));

sr1o_fb = sr1_fb;
sr0_fb = sr1_fb + sr2_fb;


uzI0_1 = zeros(1,length(sr0_1));
uzI0_3 = zeros(1,length(sr0_3));
uzI0_5 = zeros(1,length(sr0_5));
uz_pz = zeros(1,length(sr0_1));
uz_fb = zeros(1,length(sr0_fb));

for i = 1:length(sr0_1)-1
    uzI0_1(i+1) = sum(sr0_1(1:i)*dz*2);
    uzI0_3(i+1) = sum(sr0_3(1:i)*dz*2);
    uzI0_5(i+1) = sum(sr0_5(1:i)*dz*2);

    uz_pz(i+1) = sum(sr0_pz(1:i)*dz*2);
    uz_fb(i+1) = sum(sr0_fb(1:i)*dz*2);
end


%% wattmeter w/ bubbles (lambda = 0.01; p-values fit w/ lab + ice cores)

ko = [ko_bestmax_wb ko_best_wb ko_bestmin_wb];
p = [pbestmin_wb pbest_wb pbestmax_wb];


Qg = ones(1,length(z)).*Qg_wb;
d_watt_wb = ones(length(p),length(z));
uzI_watt_wb = ones(length(p),length(z));
sr_gbs_watt_wb = ones(length(p),length(z));
sr_disl_watt_wb = ones(length(p),length(z));

%Loop through each set of grain growth parameters
for l = 1:length(p)

    P = ones(1,length(z)).*p(l);
    Kg = ones(1,length(z)).*ko(l);

    sr = sr0_1; beta = 0.5; d = d1;

    dpin = 1e1;
    dmin = 1e-8;
    dd = ones(1,length(d));
    ddn = ones(1,length(d));

    while mean(abs(ddn(1:end-2))) > .001

        di = ((Kg.*exp(-Qg./(R.*T)).*(1./P).*c.*gamma) ./ ...
            ((lambda_GBS - beta*lambda_GBS + ...
            beta*lambda_disl).*tau.*sr)).^(1./(1+P));    

        dd = di - d;
        ddn = dd./d;       % normalized grain size difference
        d = d + dd.*.1;

        d(d>dpin) = dpin;
        d(d<dmin) = dmin;
        d(isnan(d)==1) = dpin;

        sr1 = A1o.*(d.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
        sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));

        sr = sr1 + sr2;
        beta = sr2./sr; beta(end) = 0;

        uzI(1) = 0;
        for i = 1:length(sr)-1
            uzI(i+1) = sum(sr(1:i)*dz*2);
        end
    end

    d_watt_wb(l,:) = d;
    uzI_watt_wb(l,:) = uzI;
    sr_gbs_watt_wb(l,:) = sr1;
    sr_disl_watt_wb(l,:) = sr2;

end


%% wattmeter w/ bubbles (lambda = 0.005; p-values fit w/ lab + ice cores)

d_watt_wb_smL = ones(length(p),length(z));
uzI_watt_wb_smL = ones(length(p),length(z));
sr_gbs_watt_wb_smL = ones(length(p),length(z));
sr_disl_watt_wb_smL = ones(length(p),length(z));
lambda_disl = 0.005;  % fraction of work by disl creep converted 
                      %   to int. energy (=0.1 for AE07)
lambda_GBS = 0.005;   % fraction of work by GBS converted to int. energy 
                      %   (note thatthis term has been added compared to 
                      %    AE07, as they assumed that lambda_diff = 0).


%Loop through each set of grain growth parameters
for l = 1:length(p)

    P = ones(1,length(z)).*p(l);
    Kg = ones(1,length(z)).*ko(l);

    sr = sr0_1; beta = 0.5; d = d1;

    dpin = 1e1;
    dmin = 1e-8;
    dd = ones(1,length(d));
    ddn = ones(1,length(d));

    while mean(abs(ddn(1:end-2))) > .001

        di = ((Kg.*exp(-Qg./(R.*T)).*(1./P).*c.*gamma) ./ ...
            ((lambda_GBS - beta*lambda_GBS + ...
            beta*lambda_disl).*tau.*sr)).^(1./(1+P));    

        dd = di - d;
        ddn = dd./d;       % normalized grain size difference
        d = d + dd.*.1;

        d(d>dpin) = dpin;
        d(d<dmin) = dmin;
        d(isnan(d)==1) = dpin;

        sr1 = A1o.*(d.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
        sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));

        sr = sr1 + sr2;
        beta = sr2./sr; beta(end) = 0;

        uzI(1) = 0;
        for i = 1:length(sr)-1
            uzI(i+1) = sum(sr(1:i)*dz*2);
        end
    end

    d_watt_wb_smL(l,:) = d;
    uzI_watt_wb_smL(l,:) = uzI;
    sr_gbs_watt_wb_smL(l,:) = sr1;
    sr_disl_watt_wb_smL(l,:) = sr2;

end



%% wattmeter w/ bubbles (lambda = 0.01; p-values fit w/ lab data exp. AL5 only)

ko = [ko_bestmax_AL5 ko_best_AL5 ko_bestmin_AL5];
p = [pbestmin_AL5 pbest_AL5 pbestmax_AL5];

d_watt_wb_lab = ones(length(p),length(z));
uzI_watt_wb_lab = ones(length(p),length(z));
sr_gbs_watt_wb_lab = ones(length(p),length(z));
sr_disl_watt_wb_lab = ones(length(p),length(z));
lambda_disl = 0.01;   % fraction of work by disl creep converted 
                      %   to int. energy (=0.1 for AE07)
lambda_GBS = 0.01;    % fraction of work by GBS converted to int. energy 
                      %   (note thatthis term has been added compared to 
                      %   AE07, as they assumed that lambda_diff = 0).

%Loop through each set of grain growth parameters
for l = 1:length(p)

    P = ones(1,length(z)).*p(l);
    Kg = ones(1,length(z)).*ko(l);

    sr = sr0_1; beta = 0.5; d = d1;

    dpin = 1e1;
    dmin = 1e-8;
    dd = ones(1,length(d));
    ddn = ones(1,length(d));

    while mean(abs(ddn(1:end-2))) > .001

        di = ((Kg.*exp(-Qg./(R.*T)).*(1./P).*c.*gamma) ./ ...
            ((lambda_GBS - beta*lambda_GBS + ...
            beta*lambda_disl).*tau.*sr)).^(1./(1+P));    

        dd = di - d;
        ddn = dd./d;       % normalized grain size difference
        d = d + dd.*.1;

        d(d>dpin) = dpin;
        d(d<dmin) = dmin;
        d(isnan(d)==1) = dpin;

        sr1 = A1o.*(d.^-1.4).*((tau./1e6).^n1).*exp(-Q1./(R.*T));
        sr2 = A2o.*((tau./1e6).^n2).*exp(-Q2./(R.*T));

        sr = sr1 + sr2;
        beta = sr2./sr; beta(end) = 0;

        uzI(1) = 0;
        for i = 1:length(sr)-1
            uzI(i+1) = sum(sr(1:i)*dz*2);
        end
    end

    d_watt_wb_lab(l,:) = d;
    uzI_watt_wb_lab(l,:) = uzI;
    sr_gbs_watt_wb_lab(l,:) = sr1;
    sr_disl_watt_wb_lab(l,:) = sr2;

end


%%
figure(21)
clf

subplot(241)
semilogx(d_pz*1e3,H-z,'g','linewidth',2)
hold on

patch([d_watt_wb_lab(1,:)*1e3,fliplr(d_watt_wb_lab(3,:)*1e3)],...
    [H-z,fliplr(H-z)],'r','edgecolor','none')
alpha(0.25)
semilogx(d_watt_wb_lab(2,:)*1e3,H-z,'r','linewidth',2)

patch([d_watt_wb(1,:)*1e3,fliplr(d_watt_wb(3,:)*1e3)],...
    [H-z,fliplr(H-z)],'b','edgecolor','none')
alpha(0.25)
semilogx(d_watt_wb(2,:)*1e3,H-z,'b','linewidth',2)

semilogx(d_pz*1e3,H-z,'g','linewidth',1)

axis([3e-1 1e2 0 H])
set(gca,'XTick',[1e-1 1e0 1e1 1e2])
text(3e-1,-45,'a)','horizontalalign','left')
text(80,850,'3^o slope','horizontalalign','right')
text(80,950,'253 K','horizontalalign','right')
axis ij
xlabel('grain size (mm)')
ylabel('depth (m)')
[pos0,pos]=fixposition(1,-0.02,1,0,1.1,0,1.1,0);
set(gca,'layer','top','linewidth',0.75)

subplot(242)
plot(uzI0_1*3600*24*365,H-z,'k','linewidth',2)
hold on
plot(uzI0_3*3600*24*365,H-z,'k--','linewidth',2)
plot(uzI0_5*3600*24*365,H-z,'k:','linewidth',2)
plot(uz_pz*3600*24*365,H-z,'g','linewidth',2)
%plot(uz_fb*3600*24*365,H-z,'r')

patch([uzI_watt_wb_lab(1,:)*3600*24*365,fliplr(uzI_watt_wb_lab(3,:)*3600*24*365)],...
    [H-z,fliplr(H-z)],'r','edgecolor','none')
alpha(0.25)
plot(uzI_watt_wb_lab(2,:)*3600*24*365,H-z,'r','linewidth',2)

patch([uzI_watt_wb(1,:)*3600*24*365,fliplr(uzI_watt_wb(3,:)*3600*24*365)],...
    [H-z,fliplr(H-z)],'b','edgecolor','none')
alpha(0.25)
plot(uzI_watt_wb(2,:)*3600*24*365,H-z,'b','linewidth',2)

plot(uz_pz*3600*24*365,H-z,'g','linewidth',2)

axis ij
xlabel('Velocity (m/yr)')
text(1e-4,-45,'b)','horizontalalign','left')
set(gca,'XLim',[0 75],'XTick',0:25:75)
set(gca,'YLim',[0 H],'YTickLabel','')

[pos0,pos]=fixposition(1,-0.01,1,0,1.1,0,1.1,0);
set(gca,'layer','top','linewidth',0.75)

subplot(243)
semilogx(sr0_1,H-z,'k','linewidth',2)
hold on
semilogx(sr0_3,H-z,'k--','linewidth',2)
semilogx(sr0_5,H-z,'k:','linewidth',2)
semilogx(sr0_pz,H-z,'g','linewidth',2)
%semilogx(sr0_fb,H-z,'r')
patch([sr_gbs_watt_wb_lab(1,:)+sr_disl_watt_wb_lab(1,:),...
    fliplr(sr_gbs_watt_wb_lab(3,:)+sr_disl_watt_wb_lab(3,:))],...
    [H-z,fliplr(H-z)],'r','edgecolor','none')
alpha(0.25)

patch([sr_gbs_watt_wb(1,:)+sr_disl_watt_wb(1,:),...
    fliplr(sr_gbs_watt_wb(3,:)+sr_disl_watt_wb(3,:))],...
    [H-z,fliplr(H-z)],'b','edgecolor','none')
alpha(0.25)

semilogx(sr_gbs_watt_wb_lab(2,:)+sr_disl_watt_wb_lab(2,:),H-z,'r','linewidth',2)
semilogx(sr_gbs_watt_wb(2,:)+sr_disl_watt_wb(2,:),H-z,'b','linewidth',2)

semilogx(sr0_pz,H-z,'g','linewidth',2)

axis ij

text(1e-16,-45,'c)','horizontalalign','left')
xlabel('strain-rate (1/s)')
set(gca,'YTickLabel','')
set(gca,'YLim',[0 H])
set(gca,'XLim',[1e-16 1e-8],'XTick',[1e-16,1e-14,1e-12,1e-10,1e-8],...
    'XTickLabel',{'10^{-16}','','10^{-12}','','10^{-8}'})
[pos0,pos]=fixposition(1,0,1,0,1.1,0,1.1,0);

lg = legend('d = 1 mm','d = 3 mm','d = 10 mm','piezometer',...
    ['wattmeter' newline '(lab AL5 only)'],...
    ['wattmeter' newline '(lab + ice core)'],'location','southeast');
set(lg,'fontsize',9)
lgp = get(lg,'Position');
%set(lg,'Position',[lgp(1)+.58 lgp(2) lgp(3) lgp(4)])
set(lg,'Position',[lgp(1)-.055 lgp(2)-.005 lgp(3) lgp(4)])
set(gca,'layer','top','linewidth',0.75)


subplot(244)
plot(diff(log10(sr0_1))./diff(log10(tau)),H-z(2:end),'k','linewidth',2)
hold on
plot(diff(log10(sr0_3))./diff(log10(tau)),H-z(2:end),'k--','linewidth',2)
plot(diff(log10(sr0_5))./diff(log10(tau)),H-z(2:end),'k:','linewidth',2)
plot(diff(log10(sr0_pz))./diff(log10(tau)),H-z(2:end),'g','linewidth',2)

patch([diff(log10(sr_gbs_watt_wb_lab(1,:)+sr_disl_watt_wb_lab(1,:)))./diff(log10(tau)),...
    fliplr(diff(log10(sr_gbs_watt_wb_lab(3,:)+sr_disl_watt_wb_lab(3,:)))./diff(log10(tau)))],...
    [H-z(2:end), fliplr(H-z(2:end))],'r','edgecolor','none')
alpha(0.25)

%plot(diff(log10(sr_gbs_watt_wb_lab(1,:)+sr_disl_watt_wb_lab(1,:)))./diff(log10(tau)),H-z(2:end),'r')
plot(diff(log10(sr_gbs_watt_wb_lab(2,1:end-1)+sr_disl_watt_wb_lab(2,1:end-1)))./diff(log10(tau(1:end-1))),H-z(2:end-1),'r','linewidth',2)
%plot(diff(log10(sr_gbs_watt_wb_lab(3,:)+sr_disl_watt_wb_lab(3,:)))./diff(log10(tau)),H-z(2:end),'r')


patch([diff(log10(sr_gbs_watt_wb(1,:)+sr_disl_watt_wb(1,:)))./diff(log10(tau)),...
    fliplr(diff(log10(sr_gbs_watt_wb(3,:)+sr_disl_watt_wb(3,:)))./diff(log10(tau)))],...
    [H-z(2:end), fliplr(H-z(2:end))],'b','edgecolor','none')
alpha(0.25)

%plot(diff(log10(sr_gbs_watt_wb(1,:)+sr_disl_watt_wb(1,:)))./diff(log10(tau)),H-z(2:end),'b')
plot(diff(log10(sr_gbs_watt_wb(2,1:end-1)+sr_disl_watt_wb(2,1:end-1)))./diff(log10(tau(1:end-1))),H-z(2:end-1),'b','linewidth',2)
%plot(diff(log10(sr_gbs_watt_wb(3,:)+sr_disl_watt_wb(3,:)))./diff(log10(tau)),H-z(2:end),'b')

plot(diff(log10(sr0_pz))./diff(log10(tau)),H-z(2:end),'g','linewidth',2)

text(1.5,-45,'d)','horizontalalign','left')
set(gca,'YTickLabel','')

xlabel('n_{eff}')
axis ij
axis([1.5 4.5 0 H])
[pos0,pos]=fixposition(1,0.01,1,0,1.1,0,1.1,0);
set(gca,'layer','top','linewidth',0.75)




