function [v_out, Dv_out] = shoot_v1(v_0,parameters)
%[v_out Dv_out] = shoot(v_0,parameters)
%Generic single shooting method o.d.e integration routine that calls matlab o.d.e.
%solvers. shoot_v1 takes a vector v_0 specifying initial conditions and a
%parameter structure that contains information about the differential
%equation solved, integration interval, any event functions that determine the endpoint of
%integration as well as any model parameters and solver specifications and
%outputs a vector specifying the final state of the solution as well the
%Jacobian of that final state with respect to the initial conditions
%
%Input variables are
%   v_0:     column vector specifying intial conditions. In the
%            documentation below, ndegf is the length of v_0
%   parameters: parameter structure with fields
%       t_span: 2-by-1 vector specifying initial and final time for
%               integration; if an event function is specified (see below)
%               this only gives the maximum length of integration.
%       model: optional substructure containing all required model
%               parameters needed to evaluate the functions specified
%               through the 'functions' substructure in 'parameters'
%       functions: required substructure with fields
%       evolve: required field giving the function handle of a function in
%               the current path of the form
%                   parameters.functions.evolve = @evolvefunction(t,v,n,params)
%               if 'parameters' has a substructure 'model' as specified above, or
%                   @evolvefunction(t,v,n)
%               if the 'model' substructure is absent. In either case,
%               evolvefunction must be defined for
%                   n = 0: the right-hand side of the differential equation
%                   system, written as an ndegf-by-one column vector
%                   n = 1: the derivative (or Jacobian) of the right-hand
%                   side of the differential equation system with respect
%                   to the state vector v, written as a ndegf-by-ndegf
%                   matrix
%                   n = 2: the second derivative array of of the right-hand
%                   side of the differential equation system with respect
%                   to the state vector v, written as a
%                   ndegf-by-ndegf-by-ndegf array
%               if the 'params' input argument is required, shoot will pass
%               the parameters.model substructure in its place to
%               the function specified through evolve
%       mass:   optional field giving the function handle of a function in
%               the current path of the form
%                   parameters.functions.mass = @massfunction(t,params)
%               or
%                   parameters.functions.mass = @massfunction(t)
%               depending on whether a parameters.model substructure is
%               supplied. massfunction evaluates a mass matrix. Only used
%               if parameters.solver (below) specifies a method that
%               supports mass matrices (default is yes)
%       events: optional field giving the function handle of a function in
%               the current path of the form
%                   parameters.functions.events = @eventfunction(t,v,n,params)
%               or
%                   parameters.functions.events = @eventfunction(t,v,n)
%               depending on whether a parameters.model substructure is
%               supplied. In either case, eventfunction must output the
%               following:
%                   n = 0: outputs arguments
%                   [event_out,isterminal,direction] as used by event
%                   functions defined through the 'odeset' command in
%                   matlab o.d.e integrators. The integration terminates at
%                   a zero of event_out if isterminal is true and the
%                   change of sign in event_out as it passes through its
%                   zero is in the direction specified by direction (if the
%                   latter is +1 or -1), or always if direction evaluates
%                   to zero
%                   n = 1: outputs devent_out, the derivative with respect
%                   to v of event_out defined for n=0
%               If an event function is specified, then an event *must*
%               occur in the interval given by t_span or shoot will exit
%               with an error.
%       solver: optional substructure with fields
%           method: string that specifies the matlab ode solver to be used,
%               allows '15s', '23t' and '45'. Defaults to 15s.
%           RelTol: specifies 'RelTol' relative error tolerance passed to
%               matlab ode solver
%           AbsTol: specifies 'AbsTol' absolute error tolerance passed to
%               matlab ode solver
%       flags: optional substructure that allows a Jacobian test to be
%               computed. If a field 'test' under flags is set to true, the
%               output v_f is a structure that contains the following
%               fields as generated by Jacobian_test_v2
%           diff:   analytically-computed derivative matrix
%           diff_num: numerically-computed derivative matrix
%           error:  element-wise error in numerically-computed
%                   derivative matrix
%           error_norm: matrix norm of difference between diff and
%                   diff_num, scaled to norm of diff
%               if flags.test is set to true, there are three other options
%               available to refine the derivative test
%           derivative_test: instead of testing the Jacobian of the
%               right-hand side of the evolution equation for v and of the
%               evolution equation for the Jacobian of v with respect to
%               its initial conditions, the test routine now tests directly
%               the Jacobian of the evolution equation for v, and its
%               second derivative array. The computed derivatives are only
%               in the form of a block matrix consisting of [Jacobian of
%               right-hand side, zeros ; derivatives of the Jacobian
%               reshaped as a column vector with respect to v reshaped as a
%               row vector, zeros]. This option helps to identify errors in
%               the second derivative computation in 'evolve'.
%           identity: tests numerical against analytical derivatives in the
%               case where all entries in v and in the evolving Jacobian
%               are unity
%           local: uses the values of specified as input arguments to shoot
%               instead
%           If neither 'identity' nor 'local' is set, derivatives are
%               computed around random values of v and of the evolving
%               Jacobian, with these values lying between 0 and 1.
%
%Output:
%   v_out:        Final state vector after integration
%   Dv_out:       Jacobian of v_out with respect to v_0 (optional)
%
%Dependencies: Jacobian_test_v2 if any of the flag options are set.
%
%Christian Schoof, January 2015, June 2016
%same as v0, but with documentation corrected; event function mode
%successfully tested; also possible error in mass matrix generator fixed

%set interval length
t_span = parameters.t_span;

%Establish number of degrees of freedom
parameters.ndegf = length(v_0);

%solver settings
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'method') || (~strcmp(parameters.solver.method,'15s') && ~strcmp(parameters.solver.method,'45') && ~strcmp(parameters.solver.method,'23t'))
    parameters.solver.method = '15s';
end
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'RelTol')
    parameters.solver.RelTol = 1e-12;
end
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'AbsTol')
    parameters.solver.AbsTol = 1e-12;
end

%set mass matrix and Jacobian
if nargout > 1
    if ~isfield(parameters.functions,'events')
        options = odeset('Mass',@(t)mass_augmented(t,parameters),'MStateDep','none','Jacobian',@(t,v)Jacobian_augmented(t,v,parameters),'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol);
    else
        options = odeset('Mass',@(t)mass_augmented(t,parameters),'MStateDep','none','Jacobian',@(t,v)Jacobian_augmented(t,v,parameters),'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol,'Events',@(t,v)events_augmented(t,v,parameters));
    end
else
    if ~isfield(parameters.functions,'events')
        options = odeset('Mass',@(t)mass(t,parameters),'MStateDep','none','Jacobian',@(t,v)Jacobian(t,v,parameters),'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol);        
    else
        options = odeset('Mass',@(t)mass(t,parameters),'MStateDep','none','Jacobian',@(t,v)Jacobian(t,v,parameters),'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol,'Events',@(t,v)events(t,v,parameters));
    end
end


%construct initial conditions
if nargout > 1
    %Construct augmented initial conditions
    V_0 = zeros(parameters.ndegf*(parameters.ndegf+1),1);
    V_0(1:parameters.ndegf) = v_0(1:parameters.ndegf); %terms corresponding to input state vector
    V_0(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1)) = reshape(eye(parameters.ndegf),parameters.ndegf^2,1); %now add terms corresponding to Jacobian with respect to state vector
else
    V_0 = v_0(1:parameters.ndegf);
end

%Test Jacobian (optional)
if nargin > 1 && isfield(parameters,'flags') && isfield(parameters.flags,'test') && parameters.flags.test
    if isfield(parameters.flags,'derivative_test') && parameters.flags.derivative_test
        if isfield(parameters.flags,'identity') && parameters.flags.identity
            v_out = Jacobian_test_v2(@(v,parameters)evolve_derivative_test(1,v,parameters),@(v,parameters)Jacobian_derivative_test(1,v,parameters),[ones(parameters.ndegf,1); reshape(eye(parameters.ndegf),parameters.ndegf^2,1); ones(parameters.ndegf,1)],parameters,sqrt(eps));
        elseif isfield(parameters.flags,'local') && parameters.flags.local
            t_in = t_span(1);
            v_out = Jacobian_test_v2(@(v,parameters)evolve_derivative_test(t_in,v,parameters),@(v,parameters)Jacobian_derivative_test(t_in,v,parameters),V_0,parameters,sqrt(eps));
        else
            trand = rand(1);
            v_out = Jacobian_test_v2(@(v,parameters)evolve_derivative_test(trand,v,parameters),@(v,parameters)Jacobian_derivative_test(trand,v,parameters),rand(size(V_0)),parameters,sqrt(eps));
        end
        if nargout > 1, Dv_out = []; end
        return        
    elseif nargout > 1
        if isfield(parameters.flags,'identity') && parameters.flags.identity
            v_out = Jacobian_test_v2(@(v,parameters)evolve_augmented(1,v,parameters),@(v,parameters)Jacobian_augmented(1,v,parameters),[ones(parameters.ndegf,1); reshape(eye(parameters.ndegf),parameters.ndegf^2,1); ones(parameters.ndegf,1)],parameters,sqrt(eps));
        elseif isfield(parameters.flags,'local') && parameters.flags.local
            t_in = t_span(1);
            v_out = Jacobian_test_v2(@(v,parameters)evolve_augmented(t_in,v,parameters),@(v,parameters)Jacobian_augmented(t_in,v,parameters),V_0,parameters,sqrt(eps));
        else
            trand = rand(1);
            v_out = Jacobian_test_v2(@(v,parameters)evolve_augmented(trand,v,parameters),@(v,parameters)Jacobian_augmented(trand,v,parameters),rand(size(V_0)),parameters,sqrt(eps));
        end
        Dv_out = [];
        return
    else
        if isfield(parameters.flags,'identity') && parameters.flags.identity
            v_out  = Jacobian_test_v2(@(v)evolve(1,v,parameters),@(v)Jacobian(1,v,parameters),ones(size(V_0)),parameters,sqrt(eps));           
        elseif isfield(parameters.flags,'local') && parameters.flags.local
            t_in = t_span(1);
            v_out  = Jacobian_test_v2(@(v,parameters)evolve(t_in,v,parameters),@(v,parameters)Jacobian(t_in,v,parameters),V_0,parameters,sqrt(eps));           
        else
            trand = rand(1);
            v_out  = Jacobian_test_v2(@(v,parameters)evolve(trand,v,parameters),@(v,parameters)Jacobian(trand,v,parameters),rand(size(V_0)),parameters,sqrt(eps));
        end
        return
    end
end

if isfield(parameters.functions,'events')
    if nargout > 1
        %use supplied solver specification to integrate forward in time
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
            case '23t'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
            case '45'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
        end
    else
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@(t,v)evolve(t,v,parameters),t_span,V_0,options);
            case '23t'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@(t,v)evolve(t,v,parameters),t_span,V_0,options);
            case '45' 
                [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@(t,v)evolve(t,v,parameters),t_span,V_0,options);   
        end
    end
else
    if nargout > 1
        %use supplied solver specification to integrate forward in time
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol] = ode15s(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
            case '23t'
                [t_sol, V_sol] = ode23t(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
            case '45'
                [t_sol, V_sol] = ode45(@(t,v)evolve_augmented(t,v,parameters),t_span,V_0,options);
        end
    else
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol] = ode15s(@(t,v)evolve(t,v,parameters),t_span,V_0,options);
            case '23t'
                [t_sol, V_sol] = ode23t(@(t,v)evolve(t,v,parameters),t_span,V_0,options);
            case '45' 
                [t_sol, V_sol] = ode45(@(t,v)evolve(t,v,parameters),t_span,V_0,options);   
        end
    end
end

%test whether event function was triggered
if isfield(parameters.functions,'events') && isempty(tfin), error('terminating event did not occur'), end

%extract final state in integration
V_sol = V_sol(end,:).';

%state vector component
v_out = V_sol(1:parameters.ndegf);

%Jacobian, if necessary, and adjust for variable integration length if an
%event function is used
if nargout > 1
    Dv_out = V_sol(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1));
    Dv_out = reshape(Dv_out,parameters.ndegf,parameters.ndegf);
    %correct advected Jacobian for displacement of endpoint of integration 
    if isfield(parameters.functions,'events')
        if isfield(parameters,'model')
            Devents_f = parameters.functions.events(tfin,v_out,1,parameters.model);
            evolve_f = parameters.functions.evolve(tfin,v_out,0,parameters.model);
        else
            Devents_f = parameters.functions.events(tfin,v_out,1);   
            evolve_f = parameters.functions.evolve(tfin,v_out,0);
        end
        Dv_out = (eye(parameters.ndegf) - 1/(Devents_f.'*evolve_f)*evolve_f*Devents_f.')*Dv_out;
    end
end

end

%Re-routing functions
function [event_out,isterminal,direction] = events(t,v,parameters)
%Re-routes to event function that terminates integration
%This function is specified by a function handle given as the field
%functions.events in the parameters structure. The corresponding function
%needs to be of the form
%parameters.functions.events = @eventfunction(t,v,n,params)
%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @eventfunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of events function (here zero)
%   params: model parameters needed for function evaluation
%events evaluates eventfunction with t, v_in, n=0 and params=parameters.model
    if isfield(parameters,'model')
         [event_out,isterminal,direction] = parameters.functions.events(t,v,0,parameters.model);
    else
         [event_out,isterminal,direction] = parameters.functions.events(t,v,0);       
    end
end

function [event_out,isterminal,direction] = events_augmented(t,v_augmented,parameters)
%Re-routes to event function that terminates integration
%This function is specified by a function handle given as the field
%functions.events in the parameters structure. The corresponding function
%needs to be of the form
%parameters.functions.events = @eventfunction(t,v_in,n,params)
%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @eventfunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of events function (here zero)
%   params: model parameters needed for function evaluation
%events evaluates eventfunction with t, v, n=0 and params=parameters.model.
%Works as 'events' above but extracts the appropriate part of the input
%state vector, which in the augmented case also includes the components of
%the advected Jacobian
    v = v_augmented(1:parameters.ndegf);
    if isfield(parameters,'model')
         [event_out,isterminal,direction] = parameters.functions.events(t,v,0,parameters.model);
    else
         [event_out,isterminal,direction] = parameters.functions.events(t,v,0);       
    end
end


function m_out = mass(t,parameters)
%Re-routes to a mass matrix function given optionally as the field
%functions.mass in the parameters structure, in the form
%parameters.functions.mass = @massfunction(t,n,params)
%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.mass = @massfunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   params: model parameters needed for function evaluation
%events evaluates eventfunction with t, v_in, n=0 and params=parameters.model    
    if isfield(parameters.functions,'mass')
        if isfield(parameters,'model')
            m_out = parameters.functions.mass(t,parameters.model);
        else
            m_out = parameters.functions.mass(t);
        end
    else
        m_out = eye(parameters.ndegf);
    end
end

function m_out = mass_augmented(t,parameters)
%Re-routes to a mass matrix function given optionally as the field
%functions.mass in the parameters structure, in the form
%parameters.functions.mass = @massfunction(t,params)
%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.mass = @massfunction(t)
%The arguments are, in order:
%   t:    time
%   params: model parameters needed for function evaluation
%events evaluates eventfunction with t, v_in, n=0 and
%params=parameters.model. Works the same way as 'mass' above but expands
%mass matrix to evolve Jacobian
    if isfield(parameters.functions,'mass')
        if isfield(parameters,'model')
            m_out = kron(eye(parameters.ndegf+1),parameters.functions.mass(t,parameters.model));
        else
            m_out = kron(eye(parameters.ndegf),parameters.functions.mass(t));
        end
    else
        m_out = eye(parameters.ndegf*(parameters.ndegf+1));
    end
end

function f_out = evolve(t,v,parameters)
%Re-routes to right-hand side of differential equation, given as a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%evolve evaluates eventfunction with t, v, n=0 and params=parameters.model
    if isfield(parameters,'model')
         f_out = parameters.functions.evolve(t,v,0,parameters.model);
    else
         f_out = parameters.functions.evolve(t,v,0);       
    end
end

function f_out = evolve_augmented(t,v_augmented,parameters)
%Re-routes to right-hand side of differential equation, given as a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%Works as 'evolve' above but appends the right-hand side of the evolution
%equation for the Jacobian of the final state as a function of the initial
%state
    v = v_augmented(1:parameters.ndegf);
    f_out = zeros(parameters.ndegf*(parameters.ndegf+1),1);
    if isfield(parameters,'model')
        F = parameters.functions.evolve(t,v,0,parameters.model);
        DF = parameters.functions.evolve(t,v,1,parameters.model);
    else
        F = parameters.functions.evolve(t,v,0);
        DF = parameters.functions.evolve(t,v,1);
    end
    f_out(1:parameters.ndegf) = F;
    f_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1)) = kron(eye(parameters.ndegf),DF)*v_augmented(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1));
end


function J_out = Jacobian(t,v,parameters)
%Re-routes to gradient of right-hand side of differential equation, given through a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%evolve evaluates eventfunction with t, v, n=1 and params=parameters.model
    if isfield(parameters,'model')
         J_out = parameters.functions.evolve(t,v,1,parameters.model);
    else
         J_out = parameters.functions.evolve(t,v,1);       
    end
end

    
function J_out = Jacobian_augmented(t,v_augmented,parameters)
%Re-routes to gradient of evolve_augmented above, computed through a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%evolve evaluates eventfunction with t, v, n=1 and params=parameters.model
    v = v_augmented(1:parameters.ndegf);
    if isfield(parameters,'model')
        DF = parameters.functions.evolve(t,v,1,parameters.model);
        D2F = parameters.functions.evolve(t,v,2,parameters.model);
    else
        DF = parameters.functions.evolve(t,v,1);
        D2F = parameters.functions.evolve(t,v,2);        
    end
    J_out = kron(eye(parameters.ndegf+1),DF);
    for ii=1:parameters.ndegf
        J_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1),ii) =  kron(eye(parameters.ndegf),D2F(:,:,ii))*v_augmented(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1));
    end
end

function f_out = evolve_derivative_test(t,v_augmented,parameters)
%Re-routes to right-hand side of differential equation, given as a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%Works as 'evolve' above but appends the Jacobian of right-hand of the
%differential equation, reshaped as a column vector
    v = v_augmented(1:parameters.ndegf);
    f_out = zeros(parameters.ndegf*(parameters.ndegf+1),1);
    if isfield(parameters,'model')
        F = parameters.functions.evolve(t,v,0,parameters.model);
        DF = parameters.functions.evolve(t,v,1,parameters.model);
    else
        F = parameters.functions.evolve(t,v,0);
        DF = parameters.functions.evolve(t,v,1);
    end
    f_out(1:parameters.ndegf) = F;
    f_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1)) = reshape(DF,parameters.ndegf^2,1);
end

function J_out = Jacobian_derivative_test(t,v_augmented,parameters)
%Re-routes to gradient of evolve_derivative_test above, computed through a function
%handle in the field parameters.functions.evolve in the parameters
%structure in the form
%parameters.functions.evolve = @evolvefunction(t,v,n,params)
%%if the parameters structure has a field 'model' in which model parameters
%are specified, otherwise it needs to be of the form
%parameters.functions.events = @evolvefunction(t,v,n)
%The arguments are, in order:
%   t:    time
%   v:    current state vector
%   n:    order of differentiation of evolution function (here zero)
%   params: model parameters needed for function evaluation
%evolve evaluates eventfunction with t, v, n=1 and params=parameters.model
    v = v_augmented(1:parameters.ndegf);
    if isfield(parameters,'model')
        DF = parameters.functions.evolve(t,v,1,parameters.model);
        D2F = parameters.functions.evolve(t,v,2,parameters.model);
    else
        DF = parameters.functions.evolve(t,v,1);
        D2F = parameters.functions.evolve(t,v,2);        
    end
    J_out = zeros(parameters.ndegf*(parameters.ndegf+1));
    J_out(1:parameters.ndegf,1:parameters.ndegf) = DF;
    for ii=1:parameters.ndegf
        J_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1),ii) = reshape(D2F(:,:,ii),parameters.ndegf^2,1);
    end
end