function fout = parallel_zero_boundary(v_in,type1,type2,lmin,ds,scalefactor,scale,direction,npts,parameters,srchparams,flags)
%fout = parallel_stability_boundary_v3(v_in,type1,type2,lmin,ds,scalefactor,scale,direction,npts,parameters,flags)
%Computes the parameter combinations for which closed orbits of the
%following dynamical system attain a minimum value of N = 0:
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R)*(N/N_u)^(-m_slide) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K)*(N/N_u)^(-m_slide) - c_2*S_K*|N|^(n_Glen-1)*N
% -V_p_0*(uh/uh_bar)*dN/dt = Q_tot - Q_R - (n_c-1)*Q_K
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
%
%Input variables:
%   v_in:       Initial guess for steady state solution [S_R;S_K;N] at
%               first bifurcation point concatenated with initial guesses
%               for the type2 parameter
%   type1:      First parameter type to be solved for. Choices are 'uh',
%               'Q_tot', 'n_c', 'L' and 'V_p'. Defaults to 'L'.
%   type2:      Second parameter type to be solved for. Choices are 'uh',
%               'Q_tot', 'n_c', 'L' and 'V_p'. Defaults to 'Q_tot', must
%               not be the same as type1. While parameters of type1 and
%               type2 are varied, all other model parameters are held
%               fixed. The only difference between the type1 and type2
%               parameters in practical terms is that the first point
%               computed has a prescribed value of the type1 aprameter.
%   lmin:       first value to be take by parameter of type1
%   ds:         arc length continuation step size
%   scalefactor: (diagonal) metric for computation of arc length, takes
%               the form of a 5-by-1 vector [a_1 a_2 a_3 a_4 a_5] so that arc
%               length is computed as \sum_i a_i dvl_i^2 if scale is
%               'linear', and as  \sum_i a_i d(log(l_i))^2 if scale is
%               'log', with l_i being the ith parameter (i = 1, 2); l_1 is
%               the type1 parameter and l2 is the type2 parameter
%   scale:  step size computed from change in parameters
%                ('linear', default) or from change in log(parameter)
%                ('log')
%   direction:  initial direction for changes in l, choices +1 and -1
%   npts:       number of points for which bifurcation points are to be
%               found
%   parameters: parameter structure containing all parameters for the
%               dynamical system. Those corresponding to the type1 and
%               type2 parameters are overwritten during the solution
%               process.
%   srchparams: optional parameter structure passed to arc length solver
%               ArcLength_single_symmetric, see documentation for that
%               function
%Output:
%   fout:       Output structure containing the following fields
%       parameters: The parameter structure originally supplied, with any
%               missing fields set to default values. Also contains fields
%               type1 and type2 specifying which parameters were varied and
%               did not take the values specified in this structure
%       uh, Q_tot, n_c, L or V_p:   There will be two fields out of these
%               possible five, corresponding to the type1 and type2
%               parameter values computed at the bifurcation. Each is a
%               npts-by-1 vector of parameter values
%       S_R, S_K, N: npts-by-1 vector of values of S_R, S_K and N at the
%               steady states corredponding to the type1 and type2 parmeter
%               values
%       error_flag: npts-by-1 vector of booleans indicating whether arc
%               length continuation solver did not (true) converge
%       thirdeig: npts-by-1 vector giving the third (real) eigenvalue of
%               the dynamical system at the bifurcation point


%check input and set defaults as needed; for inital guess v_in, see further below
%parameter changes
if nargin < 2 || isempty(type1) || (~strcmp(type1,'uh') && ~strcmp(type1,'Q_tot') && ~strcmp(type1,'n_c') && ~strcmp(type1,'L') && ~strcmp(type1,'V_p'))
    type1 = 'L';
end
if nargin < 3 || isempty(type2) || (~strcmp(type2,'uh') && ~strcmp(type2,'Q_tot') && ~strcmp(type2,'n_c') && ~strcmp(type2,'L') && ~strcmp(type2,'V_p'))
    type2 = 'Q_tot';
end
%parameter range
if nargin < 4 || isempty(lmin)
    lmin = 1e-3;
end
if nargin < 5 || isempty(ds)
    ds = 1e-2;
end
if nargin < 6 || isempty(scalefactor)
    scalefactor = ones(length(v_in)+1,1);
end
if nargin < 7 || isempty(scale) || (~strcmp(scale,'log') && ~strcmp(scale,'linear'))
    scale = 'linear';
end
if nargin < 8 || isempty(direction)
    direction = +1;
end
%number of parameter steps
if nargin < 9 || isempty(npts)
    npts = 1000;
end
%physical parameters
if nargin < 10 || ~isfield(parameters,'L')
    parameters.L = 10;%1/eps;
end
if nargin < 10 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if ~isfield(parameters,'Q_tot')
    parameters.Q_tot = parameters.n_c;
end
if ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
end
if ~isfield(parameters,'m_store')
    parameters.m_store = 0;
end
if ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if ~isfield(parameters,'T')
    parameters.T = 1;
end
if ~isfield(parameters,'nu')
    parameters.nu = 1;
end
%set number of degrees of freedom
parameters.ndegf = length(v_in)-1;

if nargin < 11
    srchparams.itmax = 10;
    srchparams.toldelta = 1000*sqrt(eps);
    srchparams.verbose = 0;
end

%Ensure parameters to be solved for are specified as part of parameters
%structure
parameters.type1 = type1;
parameters.type2 = type2;

%initialize output
if strcmp(type1,type2), error('two different parameter types must be specified'), end
fout.type1 = type1;
switch type1
    case 'uh'
        fout.uh = zeros(npts,1);
        fout.channelize.uh = zeros(npts,1);
    case 'Q_tot'
        fout.Q_tot = zeros(npts,1);
        fout.channelize.Q_tot = zeros(npts,1);
    case 'n_c'
        fout.n_c = zeros(npts,1);
        fout.channelize.n_c = zeros(npts,1);
    case 'L'
        fout.L = zeros(npts,1);
        fout.channelize.L = zeros(npts,1);
    case 'V_p'
        fout.V_p_0 = zeros(npts,1);
        fout.channelize.V_p_0 = zeros(npts,1);
end
fout.type2 = type2;
switch type2
    case 'uh'
        fout.uh = zeros(npts,1);
        fout.channelize.uh = zeros(npts,1);
    case 'Q_tot'
        fout.Q_tot = zeros(npts,1);
        fout.channelize.Q_tot = zeros(npts,1);
    case 'n_c'
        fout.n_c = zeros(npts,1);
        fout.channelize.n_c = zeros(npts,1);
    case 'L'
        fout.L = zeros(npts,1);
        fout.channelize.L = zeros(npts,1);
    case 'V_p'
        fout.V_p_0 = zeros(npts,1);
        fout.channelize.V_p_0 = zeros(npts,1);
end
fout.S_R = zeros(npts,1);
fout.S_K = zeros(npts,1);
fout.N = zeros(npts,1);
fout.error = false(npts,1);
fout.Dv = zeros(parameters.ndegf,parameters.ndegf,npts);
fout.eigmax = zeros(1,npts);
if parameters.n_c == 1 && ~strcmp(type2,'n_c')
    fout.eig_n_c_1 = zeros(1,npts);
end
fout.t_orbit = zeros(1,npts);
fout.parameters = parameters;

%set initial guess if needed
if nargin < 2 || isempty(lmin)
    lmin = 1;
end
if nargin == 0 || isempty(v_in)
    v_in = [1 1 2 1]';
end
v_in = [v_in; lmin];

[vtemp, fout.error, faux] = ArcLength_single_symmetric(@parallel_shoot_par_zero_v2,v_in,ds,scalefactor,scale,direction,npts,parameters,srchparams);

fout.S_R(:) = vtemp(1,:);
fout.S_K(:) = vtemp(2,:);
fout.N(:) = vtemp(3,:);
switch type2
        case 'uh'
            fout.uh(:) = vtemp(4,:);
        case 'Q_tot'
            fout.Q_tot(:) = vtemp(4,:);
        case 'n_c'
            fout.n_c(:) = vtemp(4,:);
        case 'L'
            fout.L(:) = vtemp(4,:);
        case 'V_p'
            fout.V_p_0(:) = vtemp(4,:);
end   
switch type1
        case 'uh'
            fout.uh(:) = vtemp(5,:);
        case 'Q_tot'
            fout.Q_tot(:) = vtemp(5,:);
        case 'n_c'
            fout.n_c(:) = vtemp(5,:);
        case 'L'
            fout.L(:) = vtemp(5,:);
        case 'V_p'
            fout.V_p_0(:) = vtemp(5,:);
end   

for ii=2:npts
    fauxtemp = faux{ii}
    fout.Dv(:,:,ii) = fauxtemp.DvP;
    fout.eigmax(ii) = max(abs(eig(fauxtemp.DvP)));
    if parameters.n_c == 1 && ~strcmp(type2,'n_c')
        fout.eig_n_c_1(ii) = fauxtemp.DvP(1,1);
    end
    fout.t_orbit(ii) = fauxtemp.t_orbit;
end

end
