function [v_out,parameters_out,error] = parallel_steady(v_in,parameters,srchparams,flags)
%[v_out,parameters] = parallel_steady(v_in,parameters,flags)
%Solves conduits-in parallel problem, consisting of
% c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N = 0
% c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N = 0
% Q_R + (n_c-1)*Q_K = Q_tot
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% S_R, S_K and N are the variables solved for, all other quantities are
% specified as parameters.
%
%Input variables are
%   v_in:       3-by-1 vector containing initial guesses for [S_R; S_K; N]
%   parameters: optional parameter structure with fields
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%   flags       Optional flag structure with field
%       test    outputs Jacobian test if set to true as v_out
%
%Output variables are
%   v_out:      3-by-1 vector containing solution [S_R; S_K; N]; returns
%               numerical and analytical Jacobians if flags.test is set
%   parameters: parameter structure used in calculation, with defaults set
%               for any fields not set in the input variable parameters 
%   error:      flag indicating whether an error occured in computing v_out

%check input and set defaults where necessary
%initial guess
if nargin == 0 || isempty(v_in)
    v_in = [1 1 1];%[.17, .17, 1.8]';
end
%parameters
if nargin < 2 || ~isfield(parameters,'L')
    parameters.L = 1/eps;%10;
end
if nargin < 2 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if nargin < 2 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 2 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 2 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 2 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 2 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 2 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'T')
    parameters.T = 1;
end
if nargin < 2 || ~isfield(parameters,'nu')
    parameters.nu = 1;
end
if nargin < 2 || ~isfield(parameters,'Q_tot')
    parameters.Q_tot = 1;
end

%Search parameters
if nargin < 3 || ~isfield(srchparams,'itmax')
    srchparams.itmax = 10;
end
if nargin < 3 || ~isfield(srchparams,'toldelta')
    srchparams.toldelta = 100*sqrt(eps);
end
if nargin < 3 || ~isfield(srchparams,'tolgrad')
    srchparams.tolgrad = 100*sqrt(eps);
end
if nargin < 3 || ~isfield(srchparams,'verbose')
    srchparams.verbose = 0;
end

%Test Jacobian (optional)
if nargin > 3 && isfield(flags,'test') && flags.test
        v_out = Jacobian_test_v2(@parallel_constraint,@parallel_Jacobian,rand(size(v_in)),parameters,sqrt(eps));
        return
end

%Solve
[v_out, error_flag] = Newton(@parallel_constraint,@parallel_Jacobian,v_in,parameters,srchparams);
if error_flag, warning('Convergence to prescribed tolerance not acheived'), end

if nargout > 1
    parameters_out = parameters;
end
if nargout > 2
    error = error_flag;
end

end

function fout = parallel_constraint(v_in,parameters)

%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi); 

%output
fout= zeros(3,1);
fout(1) = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
fout(2) = c_1*Q_K*Psi/T + uh*(1-S_K/S_0_K) - c_2*S_K*abs(N)^(n_Glen-1)*N;
fout(3) =  Q_R + (n_c-1)*Q_K - Q_tot;

end

function fout = parallel_Jacobian(v_in,parameters)

%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = -(beta-1)*Q_K/Psi;

%output
fout= zeros(3,3);
fout(1,1) = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
fout(1,3) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
fout(2,2) = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
fout(2,3) = (c_1*dQ_KdPsi/T*Psi + c_1*Q_K/T)*dPsidN-  n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
fout(3,1) =  dQ_RdS_R;
fout(3,2) = (n_c-1)*dQ_KdS_K;
fout(3,3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

end