function [v_out, Dv_out, faux] = parallel_shoot_par_zero_v2(v_0,parameters)
%[v_out Dv_out] = parallel_shoot_v2(v_0,parameters)
%Solves time-dependent conduits-in parallel problem with a view to finding
%closed orbits. The conduits-in-parallel problem consists of
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R)*(N/N_u)^(-m_slide) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K)*(N/N_u)^(-m_slide) - c_2*S_K*|N|^(n_Glen-1)*N
% -V_p_0*(uh/uh_bar)*dN/dt = Q_tot - Q_R - (n_c-1)*Q_K
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
%The code works through a shooting method, integrating the dynamical system
%from an given initial state v_0 = [S_R; S_K; N] until the trajectory intersects
%a given surface defined as the ith nullcline (i = 1 is the S_R nullcline,
%i =2 the S_K nullcline etc) in the same direction as the flow at the
%initial point (meaning the degrees of freedom not corresponding to i move
%in the same direction as originally), having previously also crossed the
%nullcline in the opposite direction. The shooting method aims to find an
%intial state on the nullcline so that the trajectory returns to the same
%point.
%
%The code outputs computes a vector consisting of the mismatch
%between final and intial states on the orbit computed by forward
%integration (effectively, when the initial state is on the nullcline, this
%is the difference between Poincare map A(v_0) and its argument v_0, with
%ith nullcline as the assumed transversal surface that defines the Poincare
%map).
%
%In addition, it outputs the value of the ith degree of freedom on the
%nullcline minus a value vzero defined through the parameters structure.
%The purpose of this is to solve for parameter values for which  the closed
%orbit just reaches vzero for that variable. In practice, this is intended
%to work for i = 3 (so the degree of freedom in question is N) and vzero =
%0, allowing parameter regimes in which zero overburden is reached to be
%identified.
%
%If two output arguments are specified, also outputs Jacobian of the first
%output argument as a function of the initial state vector and of the two
%parameters specified as part of input vector v_0. The parameter types are
%specified as part of the input structure. 
%
% All quantities other than the initial state vector are specified as
% parameters.
%
%Input variables are
%   v_0:     5-by-1 vector containing initial state vector v_0 = ([S_R;
%            S_K; N] + offset) concatenated with the parameter values l1
%            and l2 in the form [S_R; S_K; N; l2; l1]. The parameter types
%            for l1 and l2 are specified through the parameters structure
%            in field 'type1', and 'type2' respectively, defaulting to
%            Q_tot and V_p. offset is a vector containing all zeros except,
%            when vzero=0, we have offset(parameters.i_crossing) = 1, to
%            ensure that if the orbit corssing the nullcline at an axis is
%            to be solved for, the relevant entry in v_0 does not equate to
%            zero.
%   parameters: optional parameter structure with fields
%       t_span: 2-by-1 vector specifying initial and final time for
%               integration;  only needs to be long enough for the
%               trajectory to cross the relevant nullcline with the
%               corresponding flow  component (in the dynamical systems
%               sense) to be in the same direction as at the start of the
%               integration (meaning the dot product of the
%               final flow directions with the normal to the nullcline is
%               of the same sign as the dot product between that normal and
%               the initial flow direction) AFTER having crossed the
%               nullcline once in the opposite direction. Forward
%               integration terminates at that point. 
%       solver: optional substructure with fields
%           methods: string that specifies the matlab ode solver to be used,
%               allows '15s', '23t' and '45'. Defaults to 15s.
%           RelTol: specifies 'RelTol' relative error tolerance passed to
%               matlab ode solver
%           AbsTol: specifies 'AbsTol' absolute error tolerance passed to
%               matlab ode solver
%           delta_v_0: the code displaces the intial conditions by this
%               amount in the flow direction before starting integration,
%               to avoid spurious "return to initial value" right at the
%               start of the integration. Defaults to 100*eps;
%       flags: optional substructure that allows a Jacobian test to be
%               computed. If a field 'test' under flags is set to true, the
%               output v_f is a structure that contains the following
%               fields as generated by Jacobian_test_v2
%           diff:   analytically-computed derivative matrix
%           diff_num: numerically-computed derivative matrix
%           error:  element-wise error in numerically-computed
%                   derivative matrix
%           error_norm: matrix norm of difference between diff and
%                   diff_num, scaled to norm of diff
%       i_crossing: identifies state variable whose nullcline needs to be
%               crossed to cause termination  of integration of ode system
%       vzero:  Last component of output argument v_out computes
%               v(i_crossing) on the nullcline minus vzero. Defaults to
%               vzero = 0;
%       type1:   defines the parameter whose value will be set to l1,
%               regardless of whether that parameter is already specified
%               in the parameters structure. Choices are 'uh', 'Q_tot',
%               'L', 'n_c', 'V_p' and 'none', defaults to 'Q_tot'
%       type2:   defines the parameter whose value will be set to l1,
%               regardless of whether that parameter is already specified
%               in the parameters structure. Choices are 'uh', 'Q_tot',
%               'L', 'n_c', 'V_p' and 'none', defaults to 'V_p'
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate factor due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       N_u     Normalization factor for N in sliding law
%       m_slide sliding exponent
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%       V_p_0:  Scale factor for storage capacity
%       uh_bar: Scale factor for uh in calculation of storage capacity
%       m_store:Exponent for dependence of storage capacity on sliding
%               velocity
%
%Output:
%   v_out:        difference difference of final state vector ([S_R; S_K; N]
%                 + offset) and initial state vector, concatenated with
%                 difference between v(i_crossing) on the nullcline and vzero
%   Dv_out:       Jacobian of v_out with respect to v_0 (optional)
%   faux:         Optional structure with additional information on closed
%                 orbit, with fields
%       DvP:      When [S_R; S_K; N] is a fixed point of the Poincare map,
%                 this returns the Jacobian of Poincare map as a 3-by-3
%                 matrix; as the Poincare map is defined on a
%                 two-dimensional manifold, DvP has a nullspace (given by
%                 the flow direction of the dynamical system at the fixed
%                 point) and  hence  a zero eigenvalue.
%       t_orbit:  period of orbit
%
%Works similarly to parallel_shoot_par_nullcline but requires two
%parameters l1 and l2 as part of the input v_in, and also computes
%criterion v(i_crossing) - vzero on the nullcline.
%
%works as v0 except this version adds an offset vector to make sure values
%in v_0 are not equal to zero at the solution, to facilitate lograithmic
%steps in arc length continuation method.
%
%Code currently contains problem-specific sections. May be able to pass
%names functional handles of suitably defined anonymous functions through
%parameters tructure to make code truly general; investigate for future
%use.

%%%%set problem-specific parameters if needed
if nargin < 2 || ~isfield(parameters,'type1')
    parameters.type1 = 'Q_tot';
end
if nargin < 2 || ~isfield(parameters,'type2')
    parameters.type2 = 'V_p';
end
if nargin < 2 || ~isfield(parameters,'vzero')
    parameters.vzero = 0;
end
if nargin < 2 || ~isfield(parameters,'i_crossing')
    parameters.i_crossing = parameters.ndegf;   %default to last state variable
end
if nargin < 2 || ~isfield(parameters,'Q_tot')
    parameters.Q_tot = 1;
end
if nargin < 2 || ~isfield(parameters,'L')
    parameters.L = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if nargin < 2 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 2 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 2 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 2 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 2 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 2 || ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if nargin < 2 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'T')
    parameters.T = 1;
end
if nargin < 2 || ~isfield(parameters,'nu')
    parameters.nu = 1;
end
if nargin < 2 || ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
elseif parameters.V_p_0 == 0
    error('Storage capacity must not be zero for closed orbit solver')
end
if nargin < 2 || ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if nargin < 2 || ~isfield(parameters,'m_store')
    parameters.m_store = 0;
end

            
%%%%non-problem specific code
%solver settings
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'method') || (~strcmp(parameters.solver.method,'15s') && ~strcmp(parameters.solver.method,'45') && ~strcmp(parameters.solver.method,'23t'))
    parameters.solver.method = '15s';
end
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'RelTol')
    parameters.solver.RelTol = 1e-12;
end
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'AbsTol')
    parameters.solver.AbsTol = 1e-12;
end
if nargin < 2 ||  ~isfield(parameters,'solver') || ~isfield(parameters.solver,'delta_v_0')
    parameters.solver.delta_v_0 =  100*eps*norm(v_0);
end

%set mass matrix, Jacobian and event function
if nargout > 1
    options = odeset('Mass',@mass_augmented,'MStateDep','none','Jacobian',@Jacobian_augmented,'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol,'Events',@crossing);
else
    options = odeset('Mass',@mass,'MStateDep','none','Jacobian',@Jacobian,'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol,'Events',@crossing);
end    
    
    
%set time span if needed
if nargin < 2 || ~isfield(parameters,'t_span')
    t_span = [0 20];
else
    t_span = parameters.t_span;
end

%Establish number of degrees of freedom --- this is to facilitate a
%non-problem-specific version of the code in future; obviously
%parameters.ndegf should be 3 here.
parameters.ndegf = length(v_0)-2;

%set adjustable parameter as specfied by field 'type'
switch parameters.type1
    case 'uh'
        parameters.uh = v_0(end);
    case 'Q_tot'
        parameters.Q_tot = v_0(end);
    case 'n_c'
        parameters.n_c = v_0(end);
    case 'L'
        parameters.L = v_0(end);
    case 'V_p'
        parameters.V_p_0 = v_0(end);
end
%set adjustable parameter as specfied by field 'type'
switch parameters.type2
    case 'uh'
        parameters.uh = v_0(end-1);
    case 'Q_tot'
        parameters.Q_tot = v_0(end-1);
    case 'n_c'
        parameters.n_c = v_0(end-1);
    case 'L'
        parameters.L = v_0(end-1);
    case 'V_p'
        parameters.V_p_0 = v_0(end-1);
end

%Initial conditions Add offset
if parameters.vzero == 0
    v_0(parameters.i_crossing) = v_0(parameters.i_crossing)-1;
end

if nargout > 1
    %Construct augmented initial conditions
    V_0 = zeros(parameters.ndegf*(parameters.ndegf+3),1);
    V_0(1:parameters.ndegf) = v_0(1:parameters.ndegf); %terms corresponding to input state vector
    V_0(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1)) = reshape(eye(parameters.ndegf),parameters.ndegf^2,1); %now add terms corresponding to Jacobian with respect to state vector
else
    V_0 = v_0(1:parameters.ndegf);
end
    
%Event function parameters
dv_0dt_init = step(0,v_0(1:parameters.ndegf));
parameters.dir_crossing = sign(dv_0dt_init);
parameters.dir_crossing(parameters.i_crossing) = [];    %define direction of flow for the other degrees of freedom at initial point

%Test Jacobian (optional)
if nargin > 1 && isfield(parameters,'flags') && isfield(parameters.flags,'test') && parameters.flags.test
    if nargout > 1
        if isfield(parameters.flags,'identity') && parameters.flags.identity
            v_out = Jacobian_test_v2(@step_augmented_test,@Jacobian_augmented_test,[ones(parameters.ndegf,1); reshape(eye(parameters.ndegf),parameters.ndegf^2,1); ones(2*parameters.ndegf,1)],parameters,sqrt(eps));
            v_out.par = Jacobian_test_v2(@step_test_par,@step_par_test_par,v_0(end),ones(parameters.ndegf,1),sqrt(eps));
        elseif isfield(parameters.flags,'local') && parameters.flags.local
            v_out = Jacobian_test_v2(@step_augmented_test,@Jacobian_augmented_test,V_0,parameters,sqrt(eps));
            v_out.par = Jacobian_test_v2(@step_test_par,@step_par_test_par,rand(1),v_0(1:parameters.ndegf),sqrt(eps));
        else
            v_out = Jacobian_test_v2(@step_augmented_test,@Jacobian_augmented_test,rand(size(V_0)),parameters,sqrt(eps));
            v_out.par = Jacobian_test_v2(@step_test_par,@step_par_test_par,rand(1),rand(size(v_0(1:parameters.ndegf))),sqrt(eps));
        end
        Dv_out = [];
        return
    else
        if isfield(parameters.flags,'identity') && parameters.flags.identity
              v_out  = Jacobian_test_v2(@step_test,@Jacobian_test,ones(size(V_0)),parameters,sqrt(eps));           
        elseif isfield(parameters.flags,'local') && parameters.flags.local
             v_out  = Jacobian_test_v2(@step_test,@Jacobian_test,V_0,parameters,sqrt(eps));           
        else
            v_out  = Jacobian_test_v2(@step_test,@Jacobian_test,rand(size(V_0)),parameters,sqrt(eps));
        end
        return
    end
end

if nargout > 1
    %use supplied solver specification to integrate forward in time
    switch parameters.solver.method
        case '15s'
            [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@step_augmented,t_span,V_0,options);
        case '23t'
            [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@step_augmented,t_span,V_0,options);
        case '45'
            [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@step_augmented,t_span,V_0,options);
    end
else
     switch parameters.solver.method
         case '15s'
            [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@step,t_span,V_0,options);
         case '23t'
            [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@step,t_span,V_0,options);
         case '45' 
            [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@step,t_span,V_0,options);   
     end
end
%test whether orbit crosses nullcline at all
if isempty(tfin), error(strcat('orbit did not return to initial value of degree of freedom #',num2str(parameters.i_crossing))), end
%continue if orbit did not cross nullcline in opposite direction
if length(tfin) < 2
    t_span = t_span + max(t_sol);
    V_0=V_sol(end,:)';
    dv_0dt_init = step(0,v_0(1:parameters.ndegf));
    parameters.dir_crossing = sign(dv_0dt_init);
    parameters.dir_crossing(parameters.i_crossing) = [];    %define direction of flow for the other degrees of freedom at initial point
    %increment initial conditions slightly in flow direction in order not to
    %trigger event function spuriously;
    V_0(1:parameters.ndegf) = V_0(1:parameters.ndegf) + parameters.solver.delta_v_0*dv_0dt_init/norm(dv_0dt_init);
    %repeat integration
    if nargout > 1
        %use supplied solver specification to integrate forward in time
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@step_augmented,t_span,V_0,options);
            case '23t'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@step_augmented,t_span,V_0,options);
            case '45'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@step_augmented,t_span,V_0,options);
        end
    else
        switch parameters.solver.method
            case '15s'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode15s(@step,t_span,V_0,options);
            case '23t'
                [t_sol, V_sol, tfin, Vfin, ifin] = ode23t(@step,t_span,V_0,options);
            case '45' 
                [t_sol, V_sol, tfin, Vfin, ifin] = ode45(@step,t_span,V_0,options);   
        end
    end
end
%test whether orbit crosses nullcline at all
if ~sum(ifin), error(strcat('orbit did not return to initial value of degree of freedom #',num2str(parameters.i_crossing))), end

%extract final state vector and advected Jacobian
V_sol = V_sol(end,:).';
v_f = V_sol(1:parameters.ndegf);
if nargout > 1
    Dv_f = V_sol(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+3));
    Dv_f = reshape(Dv_f,parameters.ndegf,parameters.ndegf+2);
end

%set up output
v_out = zeros(parameters.ndegf+1,1);
if nargout > 1
    Dv_out = zeros(parameters.ndegf+1,parameters.ndegf+2);
end

%function for rootfinding: difference  between initial and final state
v_out(1:parameters.ndegf) = v_f-v_0(1:parameters.ndegf);

if nargout > 1
    %Jacobian uncorrected for integration interval length / termination at
    %surface prescribed by crossing
    %correct for changing interval length controlled by forcing third variable
    %to return to original value:
    dvdt_f = step(t_sol(end),v_f);
    Mass_f = mass(t_sol(end));
    dvdt_f = Mass_f\dvdt_f;   %rate of change of final state with respect to final time
    Dg = Dcrossing(t_sol(end),v_f); %produces a row vector normal to nullcline (gradient vector of function whose zero level set defines nullcline)
    Dv_f = Dv_f - dvdt_f*(Dg*Dv_f)/(Dg*dvdt_f);
    %correct for explicit parameter dependence of nullcline
    parameters.type = parameters.type2;
    Dmug = Dmucrossing(t_sol(end),v_f);
    Dv_f(:,parameters.ndegf+1) = Dv_f(:,parameters.ndegf+1) - dvdt_f*Dmug/(Dg*dvdt_f);
    parameters.type = parameters.type1;
    Dmug = Dmucrossing(t_sol(end),v_f);
    Dv_f(:,parameters.ndegf+2) = Dv_f(:,parameters.ndegf+2) - dvdt_f*Dmug/(Dg*dvdt_f);
    Dv_f(1:parameters.ndegf,1:parameters.ndegf) = Dv_f(1:parameters.ndegf,1:parameters.ndegf) - eye(parameters.ndegf);
    %arrange output
    Dv_out(1:parameters.ndegf,:) = Dv_f(1:parameters.ndegf,:);
end
if nargout == 3
    faux.DvP = Dv_f(1:parameters.ndegf,1:parameters.ndegf)+eye(parameters.ndegf);
    faux.t_orbit = tfin(end);
end

%steady state compoments of output
v_out(parameters.ndegf+1) = v_f(parameters.i_crossing)-parameters.vzero;
Dv_out(parameters.ndegf+1,:) = Dv_f(parameters.i_crossing,:);
Dv_out(parameters.ndegf+1,parameters.i_crossing) = Dv_out(parameters.ndegf+1,parameters.i_crossing)+1;

%%%%%%non-problem specific functions, re-route to problem-specific functions
%below; may be able to pass names functional handles of suitably defined
%anonymous functions through parameters tructure to make code truly
%general; investigate for future use
function [event_out,isterminal,direction] = crossing(t,v_in)
%Event function that determines when state vector passes through the ith
%nullcline; terminates if the sign of the other components of the flow are
%the same as for the initial condition for the integration. this may not
%correspond to a full loop, so the integartion routine in the main program
%body will continue the integration if the state vector has not also passed
%the nullcline in the other direction (i.e. if there has not been an event
%for which the integration did not terminate
    event_aux = step(t,v_in);
    event_out = event_aux(parameters.i_crossing);
    event_aux(parameters.i_crossing) = [];
    if parameters.dir_crossing == sign(event_aux)
        isterminal = true;
    else
        isterminal = false;
    end
    direction = 0;
end

function Devent = Dcrossing(t,v_in)
%Gradient of first argument of event function with respect to argument v_in
    Devent_aux = Jacobian(t,v_in);
    Devent = Devent_aux(parameters.i_crossing,:);
end

function Dmuevent = Dmucrossing(t,v_in)
%Gradient of first argument of event function with respect to argument v_in
    Devent_aux = step_par(t,v_in);
    Dmuevent = Devent_aux(parameters.i_crossing);
end

function Mass_out = mass_augmented(t)
    Mass_out = kron(eye(parameters.ndegf+3),mass(t));
    type_in = parameters.type;
    parameters.type = parameters.type2;
    Mass_out(parameters.ndegf*(parameters.ndegf+1)+1:parameters.ndegf*(parameters.ndegf+2),1:parameters.ndegf) = Dlmass(t);
    parameters.type = parameters.type1;
    Mass_out(parameters.ndegf*(parameters.ndegf+2)+1:parameters.ndegf*(parameters.ndegf+3),1:parameters.ndegf) = Dlmass(t);
    parameters.type = type_in;
end
 
function v_test_out = step_augmented_test(v_in,parameters)
    v_test_out = step_augmented(0,v_in);
end

function v_test_out = step_test(v_in,parameters)
    v_test_out = step(0,v_in);
end

function v_test_out = step_test_par(l_in,v_in)
    switch parameters.type
        case 'uh'
            parameters.uh = l_in;
        case 'Q_tot'
            parameters.Q_tot = l_in;
        case 'n_c'
            parameters.n_c = l_in;
        case 'L'
            parameters.L = l_in;
        case 'V_p'
            parameters.V_p_0 = l_in;
    end
    v_test_out = step(0,v_in);
end

function v_test_out = step_par_test_par(l_in,v_in)
    switch parameters.type
        case 'uh'
            parameters.uh = l_in;
        case 'Q_tot'
            parameters.Q_tot = l_in;
        case 'n_c'
            parameters.n_c = l_in;
        case 'L'
            parameters.L = l_in;
        case 'V_p'
            parameters.V_p_0 = l_in;
    end
    v_test_out = step_par(0,v_in);
end

function v_out = step_augmented(t,v_in)
v_out = zeros(parameters.ndegf*(parameters.ndegf+2),1);
F = step(t,v_in(1:parameters.ndegf));
DF = Jacobian(t,v_in(1:parameters.ndegf));
parameters.type = parameters.type1;
Dl1F = step_par(t,v_in(1:parameters.ndegf));
parameters.type = parameters.type2;
Dl2F = step_par(t,v_in(1:parameters.ndegf));
v_out(1:parameters.ndegf) = F;
v_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1)) = kron(eye(parameters.ndegf),DF)*v_in(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+1));
v_out(parameters.ndegf*(parameters.ndegf+1)+1:parameters.ndegf*(parameters.ndegf+2)) = ...
    Dl2F+DF*v_in(parameters.ndegf*(parameters.ndegf+1)+1:parameters.ndegf*(parameters.ndegf+2));
v_out(parameters.ndegf*(parameters.ndegf+2)+1:parameters.ndegf*(parameters.ndegf+3)) = ...
    Dl1F+DF*v_in(parameters.ndegf*(parameters.ndegf+2)+1:parameters.ndegf*(parameters.ndegf+3));
end

function J_test_out = Jacobian_augmented_test(v_in,parameters)
    J_test_out = Jacobian_augmented(0,v_in);
end

function J_test_out = Jacobian_test(v_in,parameters)
    J_test_out = Jacobian(0,v_in);
end
    
function J_out = Jacobian_augmented(t,v_in)
%Jacobian of parallel_Step with regard to input argument v_in
DF = Jacobian(t,v_in(1:parameters.ndegf));
D2F = DJacobian(t,v_in(1:parameters.ndegf));
parameters.type = parameters.type1;
D2l1F = Jacobian_par(t,v_in(1:parameters.ndegf));
parameters.type = parameters.type2;
D2l2F = Jacobian_par(t,v_in(1:parameters.ndegf));
J_out = kron(eye(parameters.ndegf+3),DF);
for ii=1:parameters.ndegf
    J_out(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+3),ii) =  kron(eye(parameters.ndegf+2),D2F(:,:,ii))*v_in(parameters.ndegf+1:parameters.ndegf*(parameters.ndegf+3));
end
J_out(parameters.ndegf*(parameters.ndegf+1)+1:parameters.ndegf*(parameters.ndegf+2),1:parameters.ndegf) = ...
    J_out(parameters.ndegf*(parameters.ndegf+1)+1:parameters.ndegf*(parameters.ndegf+2),1:parameters.ndegf) + D2l2F; 
J_out(parameters.ndegf*(parameters.ndegf+2)+1:parameters.ndegf*(parameters.ndegf+3),1:parameters.ndegf) = ...
    J_out(parameters.ndegf*(parameters.ndegf+2)+1:parameters.ndegf*(parameters.ndegf+3),1:parameters.ndegf) + D2l1F; 
end

%%%%%%%%%%%%%%%%%%%%Problem-specific functions
function m_out = mass(t)
    V_p = parameters.V_p_0*storefactor(parameters.uh,parameters);
    m_out = eye(3);
    m_out(3,3) = V_p*m_out(3,3);
end

function Dm_out = Dlmass(t)
Dm_out = zeros(3);
switch parameters.type
    case 'uh'
        Dm_out(3,3) = parameters.V_p_0*Dstorefactor(parameters.uh,parameters);
    case 'Q_tot'
    case 'n_c'
    case 'L'
    case 'V_p'
        Dm_out(3,3) = storefactor(parameters.uh,parameters);
end 
end
    
function s_out = storefactor(uh,parameters)
s_out = (uh/parameters.uh_bar).^parameters.m_store;
end

function Ds_out = Dstorefactor(uh,parameters)
Ds_out = parameters.m_store*(uh/parameters.uh_bar).^(parameters.m_store-1)/parameters.uh_bar;
end
    
function F_out = step(t,v_F_in)
%right-hand side of dv/dt = f(v,l,t)
%initialize
F_out = zeros(parameters.ndegf,1);
%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_F_in(1);
S_K = v_F_in(2);
N = v_F_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
%compute output
F_out(1) = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
F_out(2) = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*abs(N)^(n_Glen-1)*N;
F_out(3) =  Q_R + (n_c-1)*Q_K - Q_tot;
end

function DlF_out = step_par(t,v_DlF_in)
%derivative of right-hand side of dv/dt = f(v,l,t) with respect to l
DlF_out = zeros(parameters.ndegf,1);
%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_DlF_in(1);
S_K = v_DlF_in(2);
N = v_DlF_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
%compute output
switch parameters.type
    case 'uh'
        DlF_out(1) = nu*(1-S_R/S_0_R);
        DlF_out(2) = (1-S_K/S_0_K);
    case 'Q_tot'
        DlF_out(3) = -1;
    case 'n_c'
        DlF_out(3) = Q_K;
    case 'L'
        dPsidL = N/L^2;
        dQ_RdPsi = (beta-1)*Q_R/Psi;
        dQ_KdPsi = (beta-1)*Q_K/Psi;
        DlF_out(1) = c_1*(Q_R + dQ_RdPsi*Psi)*dPsidL;
        DlF_out(2) = c_1*(Q_K + dQ_KdPsi*Psi)*dPsidL/T;
        DlF_out(3) = (dQ_RdPsi+(n_c-1)*dQ_KdPsi)*dPsidL;
    case 'V_p'
end
end

function DF_out = Jacobian(t,v_DF_in)
%Derivative matrix of right-hand side of dv/dt = f(v,l,t) 
%initialize
DF_out = zeros(parameters.ndegf,parameters.ndegf);
%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_DF_in(1);
S_K = v_DF_in(2);
N = v_DF_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
%Jacobian
DF_out(1,1) = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
DF_out(1,3) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
DF_out(2,2) = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
DF_out(2,3) = (c_1*dQ_KdPsi*Psi + c_1*Q_K)*dPsidN/T -  n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
DF_out(3,1) =  dQ_RdS_R;
DF_out(3,2) = (n_c-1)*dQ_KdS_K;
DF_out(3,3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;
end


function D2lF_out = Jacobian_par(t,v_DF_in)
%Derivative of Jacobian matrix grad_v (v,l,t) with respect to l
%initialize
D2lF_out = zeros(parameters.ndegf,parameters.ndegf);
%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_DF_in(1);
S_K = v_DF_in(2);
N = v_DF_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi;
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
%Jacobian
switch parameters.type
    case 'uh'
        D2lF_out(1,1) = -nu/S_0_R;
        D2lF_out(2,2) = -1/S_0_K;
    case 'Q_tot'
    case 'n_c'
        D2lF_out(3,2) = dQ_KdS_K;
    case 'L'
        dPsidL = N/L^2;
        d2PsidNdL = 1/L^2;
        d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
        d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
        d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
        d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
        D2lF_out(1,1) = c_1*(d2Q_RdS_RdPsi*Psi+dQ_RdS_R)*dPsidL;
        D2lF_out(1,3) = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN*dPsidL + (c_1*dQ_RdPsi*Psi+c_1*Q_R)*d2PsidNdL;
        D2lF_out(2,2) = c_1*(d2Q_KdS_KdPsi*Psi + dQ_KdS_K)/T*dPsidL;
        D2lF_out(2,3) = c_1*(d2Q_KdPsi2*Psi + 2*dQ_KdPsi)/T*dPsidN*dPsidL + (c_1*dQ_KdPsi*Psi + c_1*Q_K)/T*d2PsidNdL;
        D2lF_out(3,1) = d2Q_RdS_RdPsi*dPsidL;
        D2lF_out(3,2) = (n_c-1)*d2Q_KdS_KdPsi*dPsidL;
        D2lF_out(3,3) = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN*dPsidL + (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*d2PsidNdL;
    case 'V_p'
end
end

function D2F_out = DJacobian(t,v_D2F_in)
%Second derivative array of right-hand side of dv/dt = f(v,l,t)
%initialize
D2F_out = zeros(parameters.ndegf,parameters.ndegf,parameters.ndegf);
%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_D2F_in(1);
S_K = v_D2F_in(2);
N = v_D2F_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
%Higher order derivatives
d2Q_RdS_R2 = alpha*(alpha-1)*Q_R/S_R^2;
d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
d2Q_KdS_K2 = alpha*(alpha-1)*Q_K/S_K^2;
d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
d2v_cRdS_RdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cRdN2 = c_2*n_Glen*(n_Glen-1)*S_R*abs(N)^(n_Glen-3)*N;
d2v_cKdS_KdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cKdN2 = c_2*n_Glen*(n_Glen-1)*S_K*abs(N)^(n_Glen-3)*N;
%Construct second derivitve array
D2F_out(1,1,1) = c_1*d2Q_RdS_R2*Psi;
D2F_out(1,1,3) = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidN - d2v_cRdS_RdN;
D2F_out(1,3,1) = D2F_out(1,1,3);
D2F_out(1,3,3) = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN^2 - d2v_cRdN2;
D2F_out(2,2,2) = c_1*d2Q_KdS_K2*Psi/T;
D2F_out(2,2,3) = c_1*(d2Q_KdS_KdPsi*Psi + dQ_KdS_K)*dPsidN/T - d2v_cKdS_KdN;
D2F_out(2,3,2) = D2F_out(2,2,3);
D2F_out(2,3,3) = c_1*(d2Q_KdPsi2*Psi + 2*dQ_KdPsi)*dPsidN^2/T - d2v_cKdN2;
D2F_out(3,1,1) = d2Q_RdS_R2;
D2F_out(3,1,3) = d2Q_RdS_RdPsi*dPsidN;
D2F_out(3,3,1) = D2F_out(3,1,3);
D2F_out(3,2,2) = (n_c-1)*d2Q_KdS_K2;
D2F_out(3,2,3) = (n_c-1)*d2Q_KdS_KdPsi*dPsidN;
D2F_out(3,3,2) = D2F_out(3,2,3);
D2F_out(3,3,3) = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN^2;
end

end