function [tout, vout] = parallel_shoot_closed_orbit_nullcline(v_0,parameters)
%[tout, v_out] = parallel_shoot_closed_orbit_nullcline(v_0,parameters)
%Solves time-dependent conduits-in parallel problem with a view to finding
%closed orbits. The conduits-in-parallel problem consists of
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R)*(N/N_u)^(-m_slide) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K)*(N/N_u)^(-m_slide) - c_2*S_K*|N|^(n_Glen-1)*N
% -V_p_0*(uh/uh_bar)*dN/dt = Q_tot - Q_R - (n_c-1)*Q_K
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
%The code integrates forward in time until one of the degrees of freedom
%(defined by the field 'i_crossing' in the parameters structure) returns to
%its initial value. The code is designed to do do this starting on the
%i_crossing-th nullcline. It integrates forward in time until it returns to
%the initial value of the specified degree of freedom, moving in the
%original direction, having crossed the nullcline exactly once in the
%opposite direction.
%
%Input variables are
%   v_0:     3-by-1 vector containing initial state vector [S_R; S_K; N]
%   parameters: optional parameter structure with fields
%       t_span: 2-by-1 vector specifying initial and final time for
%               integration;  only needs to be long enough for the
%               trajectory to cross the relevant nullcline with the
%               corresponding flow  component (in the dynamical systems
%               sense) to be in the same direction as at the start of the
%               integration (meaning the dot product of the
%               final flow directions with the normal to the nullcline is
%               of the same sign as the dot product between that normal and
%               the initial flow direction) AFTER having crossed the
%               nullcline once in the opposite direction. Forward
%               integration terminates at that point. 
%       solver: optional substructure with fields
%           methods: string that specifies the matlab ode solver to be used,
%               allows '15s', '23t' and '45'. Defaults to 15s.
%           RelTol: specifies 'RelTol' relative error tolerance passed to
%               matlab ode solver
%           AbsTol: specifies 'AbsTol' absolute error tolerance passed to
%               matlab ode solver
%           delta_v_0: the code displaces the intial conditions by this
%               amount in the flow direction before starting integration,
%               to avoid spurious "return to initial value" right at the
%               start of the integration. Defaults to 100*eps;
%       i_crossing: identifies state variable whose nullcline needs to be
%               crossed to cause termination  of integration of ode system
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate factor due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       N_u     Normalization factor for N in sliding law
%       m_slide sliding exponent
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%       V_p_0:  Scale factor for storage capacity
%       uh_bar: Scale factor for uh in calculation of storage capacity
%       m_store:Exponent for dependence of storage capacity on sliding
%               velocity
%
%Output:
%   tout:       time stamps at which solutions [S_R; S_K; N] are computed
%   vout:       solutions [S_R; S_K; N]
%
%Code currently contains problem-specific sections. May be able to pass
%names functional handles of suitably defined anonymous functions through
%parameters tructure to make code truly general; investigate for future
%use.

%%%%set problem-specific parameters if needed
if nargin < 2 || ~isfield(parameters,'type')
    parameters.type = 'Q_tot';
end
if nargin < 2 || ~isfield(parameters,'Q_tot')
    parameters.Q_tot = 1;
end
if nargin < 2 || ~isfield(parameters,'L')
    parameters.L = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if nargin < 2 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 2 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 2 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 2 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 2 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 2 || ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if nargin < 2 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'T')
    parameters.T = 1;
end
if nargin < 2 || ~isfield(parameters,'nu')
    parameters.nu = 1;
end
if nargin < 2 || ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
elseif parameters.V_p_0 == 0
    error('Storage capacity must not be zero for closed orbit solver')
end
if nargin < 2 || ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if nargin < 2 || ~isfield(parameters,'m_store')
    parameters.m_store = 0;
end

            
%%%%non-problem specific code
%solver settings
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'method') || (~strcmp(parameters.solver.method,'15s') && ~strcmp(parameters.solver.method,'45') && ~strcmp(parameters.solver.method,'23t'))
    parameters.solver.method = '15s';
end
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'RelTol')
    parameters.solver.RelTol = 1e-12;
end
if nargin < 2 || ~isfield(parameters,'solver') || ~isfield(parameters.solver,'AbsTol')
    parameters.solver.AbsTol = 1e-12;
end
if nargin < 2 ||  ~isfield(parameters,'solver') || ~isfield(parameters.solver,'delta_v_0')
    parameters.solver.delta_v_0 =  100*eps*norm(v_0);
end

%set mass matrix, Jacobian and event function
options = odeset('Mass',@mass,'MStateDep','none','Jacobian',@Jacobian,'RelTol',parameters.solver.RelTol,'AbsTol',parameters.solver.AbsTol,'Events',@crossing);
    
%set time span if needed
if nargin < 2 || ~isfield(parameters,'t_span')
    t_span = [0 20];
else
    t_span = parameters.t_span;
end

%Establish number of degrees of freedom --- this is to facilitate a
%non-problem-specific version of the code in future; obviously
%parameters.ndegf should be 3 here.
parameters.ndegf = length(v_0);
    
%Event function parameters
%Define parameter for intial evolution of N:
%set which variable crosses its original value to terminate integration
if nargin < 2 || ~isfield(parameters,'i_crossing')
    parameters.i_crossing = parameters.ndegf;   %default to last state variable
end
dv_0dt_init = step(0,v_0);
parameters.dir_crossing = sign(dv_0dt_init);
parameters.dir_crossing(parameters.i_crossing) = [];    %define direction of flow for the other degrees of freedom at initial point
v_0 = v_0 + parameters.solver.delta_v_0*dv_0dt_init/norm(dv_0dt_init);

switch parameters.solver.method
    case '15s'
        [tout1, vout1, tfin, Vfin, ifin] = ode15s(@step,t_span,v_0,options);
    case '23t'
        [tout1, Vout1, tfin, Vfin, ifin] = ode23t(@step,t_span,v_0,options);
    case '45' 
        [tout1, vout1, tfin, Vfin, ifin] = ode45(@step,t_span,v_0,options);   
end

%test whether orbit crosses nullcline at all
if isempty(tfin), warning(strcat('orbit did not return to initial value of degree of freedom #',num2str(parameters.i_crossing))), end
%continue if orbit did not cross nullcline in opposite direction
if length(tfin) < 2
    t_span = t_span + max(tout1);
    v_0=vout1(end,:)';
    dv_0dt_init = step(0,v_0);
    parameters.dir_crossing = sign(dv_0dt_init);
    parameters.dir_crossing(parameters.i_crossing) = [];    %define direction of flow for the other degrees of freedom at initial point
    %increment initial conditions slightly in flow direction in order not to
    %trigger event function spuriously;
    V_0(1:parameters.ndegf) = V_0(1:parameters.ndegf) + parameters.solver.delta_v_0*dv_0dt_init/norm(dv_0dt_init);
    %repeat integration
    switch parameters.solver.method
        case '15s'
            [tout2, vout2, tfin, Vfin, ifin] = ode15s(@step,t_span,V_0,options);
        case '23t'
            [tout2, vout2, tfin, Vfin, ifin] = ode23t(@step,t_span,V_0,options);
        case '45' 
            [tout2, vout2, tfin, Vfin, ifin] = ode45(@step,t_span,V_0,options);   
    end
    vout = [vout1; vout2];
    tout = [tout1; tout2];
else
    vout = vout1;
    tout = tout1;
end

%test whether orbit crosses nullcline at all
if ~sum(ifin), warning(strcat('orbit did not return to initial value of degree of freedom #',num2str(parameters.i_crossing))), end



%%%%%%non-problem specific functions, re-route to problem-specific functions
%below; may be able to pass names functional handles of suitably defined
%anonymous functions through parameters tructure to make code truly
%general; investigate for future use
function [event_out,isterminal,direction] = crossing(t,v_in)
%Event function that determines when state vector passes through the ith
%nullcline; terminates if the sign of the other components of the flow are
%the same as for the initial condition for the integration. this may not
%correspond to a full loop, so the integartion routine in the main program
%body will continue the integration if the state vector has not also passed
%the nullcline in the other direction (i.e. if there has not been an event
%for which the integration did not terminate
    event_aux = step(t,v_in);
    event_out = event_aux(parameters.i_crossing);
    event_aux(parameters.i_crossing) = [];
    if parameters.dir_crossing == sign(event_aux)
        isterminal = true;
    else
        isterminal = false;
    end
    direction = 0;
end

%%%%%%%%%%%%%%%%%%%%Problem-specific functions
function m_out = mass(t)
    V_p = parameters.V_p_0*storefactor(parameters.uh,parameters);
    m_out = eye(3);
    m_out(3,3) = V_p*m_out(3,3);
end
    
function s_out = storefactor(uh,parameters)
s_out = (uh/parameters.uh_bar).^parameters.m_store;
end

function F_out = step(t,v_F_in)
%right-hand side of dv/dt = f(v,l,t)
%initialize
F_out = zeros(parameters.ndegf,1);
%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_F_in(1);
S_K = v_F_in(2);
N = v_F_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
%compute output
F_out(1) = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
F_out(2) = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*abs(N)^(n_Glen-1)*N;
F_out(3) =  Q_R + (n_c-1)*Q_K - Q_tot;
end

function DF_out = Jacobian(t,v_DF_in)
%Derivative matrix of right-hand side of dv/dt = f(v,l,t) 
%initialize
DF_out = zeros(parameters.ndegf,parameters.ndegf);
%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_DF_in(1);
S_K = v_DF_in(2);
N = v_DF_in(3);
%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
%Jacobian
DF_out(1,1) = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
DF_out(1,3) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
DF_out(2,2) = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
DF_out(2,3) = (c_1*dQ_KdPsi*Psi + c_1*Q_K)*dPsidN/T -  n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
DF_out(3,1) =  dQ_RdS_R;
DF_out(3,2) = (n_c-1)*dQ_KdS_K;
DF_out(3,3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;
end

end