function [vout,parameters_out,error] = parallel_nullcline_v2(vin,parameters,ds,scale,direction,npts,srchparams,flags)
%[v_out,parameters,error] = parallel_nullcline(v_in,parameters,ds,scale,direction,npts,srchparams,flags)
%Solves for solutions (S_R,N) of the nullcline problem for the
%dynamical system whose right-hand side is
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N
% V_p*dN/dt = Q_R- Q_tot
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% S_R, S_K and N are the variables solved for. The code sets solves for
% zeros of two of the three components of the right-hand side of the
% dynamical system above, using a continuation method to sample the
% resulting curve in (S_R,N) space. The component omitted by the
% calculation is identified by a parameter 'i_crossing', specifieed as a
% field in the structure 'parameters'
%
%Input variables are
%   v_in:       2-by-1 vector containing initial guesses for [S_R; N];
%               at the start of the continuation, the code fixes the
%               (i_crossing)th component of the vector [S_R; N] (see
%               the definition of the field i_crossing in parameters below)
%               and solves for the remaining two, and the continues the
%               solution curve in the direction given by the input
%               parameter 'direction'
%   parameters: optional parameter structure with fields
%       Q_tot   total discharge through system
%       L       length of domain
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate factor due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       i_crossing: define which of the components of the right-hand side
%               will not be set to zero by the code; for instance, if
%               i_crossing == 1, the code will be solving for zeros of
%               [c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N; ... 
%                = Q_R - Q_tot]
%       ds:     arc length "distance" between points on the solution curve,
%               see documentation of ArcLength_single_symmetric_v1.m..
%               Defaults to 0.05
%       scale:  string specifying the "scale" input argument for
%               ArcLength_single_symmetric_v1.m. Choices are 'linear' and
%               'log'. Defaults to 'log'
%       direction:  specify the direction in which the (i_crossing)th
%               component of vin should move on the first arc length
%               iteration step. Choices are +1 and -1. Defaults to -1
%       npts:   number of points along solution curve to be computed.
%               Defaults to 100;
%       srchparams: srchparams structure for use in
%               ArcLength_single_symmetric_v1.m
%
%Output variables are
%   v_out:      3-by-npts vector containing solution [S_R; S_K; N]; returns
%               numerical and analytical Jacobians if flags.test is set
%   parameters: parameter structure used in calculation, with defaults set
%               for any fields not set in the input variable parameters 
%   error:      npts-by-1 vector of flags indicating whether an error
%               occured in computing v_out
%
%Works as v1 but allows effective-pressure-dependent sliding velocity.

%check input and set defaults where necessary
%initial guess
if nargin == 0 || isempty(vin)
    vin = [1 1].';
end
%parameters
if nargin < 2 || ~isfield(parameters,'L')
    parameters.L = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 2 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 2 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 2 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 2 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 2 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 2 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 2 || ~isfield(parameters,'Q_tot')
    parameters.Q_tot = 1;
end
if nargin <2 || ~isfield(parameters,'i_crossing')
    parameters.i_crossing = 3;
end

%arc length continuation parameters
if nargin < 3 || isempty(ds)
    ds = 0.1;
end
if nargin < 4 || isempty(scale)
    scale = 'log';
end
if nargin < 5 || isempty(direction)
    direction = -1;
end
if nargin < 6 || isempty(npts)
    npts = 100;
end

%Search parameters
if nargin < 7 || ~isfield(srchparams,'itmax')
    srchparams.itmax = 10;
end
if nargin < 7 || ~isfield(srchparams,'toldelta')
    srchparams.toldelta = 100*sqrt(eps);
end
if nargin < 7 || ~isfield(srchparams,'tolgrad')
    srchparams.tolgrad = 100*sqrt(eps);
end
if nargin < 7 || ~isfield(srchparams,'verbose')
    srchparams.verbose = 0;
end

%Test Jacobian (optional)
if nargin > 7 && isfield(flags,'test') && flags.test
        vout = Jacobian_test_single(@nullclinefun,rand(size(vin)),parameters,sqrt(eps));
        return
end

%reordering to use the assumption in ArcLength_single_symmetric_v1 that the
%first iteration holds the last component in vin constant, and that the
%direction parameter dictates the initial direction for changes in that
%component
i_crossing = parameters.i_crossing;

switch i_crossing
    case 1
        vin = [vin(2);vin(1)];
    case 2
        vin = [vin(1);vin(2)];
end
        
        
%Solve
[vout, error_flag] = ArcLength_single_symmetric_v1(@nullclinefun,vin,ds,[],scale, direction,npts,parameters,srchparams);
if any(error_flag), warning('Convergence to prescribed tolerance not acheived'), end

switch i_crossing
    case 1
        vout = vout([2;1],:);
    case 2
        vout = vout([1;2],:);
end


if nargout > 1
    parameters_out = parameters;
end
if nargout > 2
    error = error_flag;
end

end

function [fout Dfout] = nullclinefun(vin,parameters)
%fout Dfout = nullclinefun(vin,parameters)
%Computes two commponents of the right-hand-side of the dynamical
%system
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N
% -V_p_0*dN/dt = Q_tot - Q_R
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% S_R and N are the variables solved for, all other quantities are
% specified as parameters.
%
%The purpose of this code is to faciliate the computation of nullclines.
%
%Input arguments
%   v_init:     2-by-1 vector containing initial condition for [S_R; N]
%   parameters: optional parameter structure with fields
%       Q_tot   total discharge through system
%       L       length of domain
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate factor due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       V_p_0:  Scale factor for storage capacity
%       uh_bar: Scale factor for uh in calculation of storage capacity
%       i_crossing: Defines the line *omitted* from the dynamical system
%               (unlike  parallel_shoot_nullcline,
%               parallel_shoot_par_nullcline, where i_crossing defines a
%               nullcline as a surface)
%
%Output arguments
%   fout:       the right-hand side of the dynamical system, omitting the
%               line defined by i_crossing
%   Dfout:      optionally, the Jacobian of fout with respect to vin


%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element

%index 
i_crossing = parameters.i_crossing;

%unpack input
switch i_crossing
    case 1
        S_R = vin(2);
        N = vin(1);
    case 2
        S_R = vin(1);
        N = vin(2);
end

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;

%output
fout = zeros(1,1);
switch i_crossing
    case 1
        fout = Q_R - Q_tot;
    case 2     
        fout = c_1*Q_R*Psi + uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
end

if nargout == 2
    dPsidN = -1/L;
    dQ_RdS_R = alpha*Q_R/S_R;
    dQ_RdPsi = (beta-1)*Q_R/Psi;

    %output
    Dfout= zeros(1,2);
    switch i_crossing
        case 1
            Dfout(1,2) =  dQ_RdS_R;
            Dfout(1,1) = dQ_RdPsi*dPsidN;
        case 2
            Dfout(1,1) = c_1*dQ_RdS_R*Psi - uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
            Dfout(1,2) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
    end
end

end
