function fout = line_steady_v2(parameters,flags)
%function [fout, foutaux] = line_steady_v2(N,parameters)
%Computes functions whose zeros define the steady state effective pressure
%at a node in a one-dimensional network.
%Input variables are:
%parameters:    Parameter structure with the many of the same fields as
%               would be required to run the network.m
%   
%   N_plus      Previously solved-for N at the next node downstream
%   grid:       structure with subgields n_nodes and n_edges
%               The code assumes a network with n_nodes nodes, and
%               n_edges = (n_nodes-1) edges, as generated by the line_grid
%               subroutine in setup.m The index n_nodes corresponds to a
%               Dirichlet node at with N is set to N_bdy_nodes
%   L:          List ((n_nodes-1)-by-one vector) of edge lengths
%   n_c:        List ((n_nodes-1)-by-one vector) of conduits per edge
%   Phi_0:      Reduced hydraulic potential at each node (n_nodes-by-one
%               vector), given by rho_i g s + (rho_w-rho_i) g b, s being
%               local ice surface elevation and b local bed elevation at
%               the node. rho_i and rho_w are ice and water density and g
%               is acceleration due to gravity
%   N_bdy_nodes:Effective pressure at boundary node (index n_nodes)
%   c_1:        Melting term is c_1*Q*Psi, where Q is discharge and Psi
%               hydraulic gradient along each network edge
%   c_2:        Creep closure is c_2*S*N^n_Glen, where S is conduit
%               cross-section or storage cavity size, N is effective
%               pressure
%   c_3:        Coefficient in Manning or Darcy-Weisbach law, discharge is
%               Q = c_3*S^alpha*Psi^(beta-1)
%   alpha:      Exponent relating conduit cross-section to discharge
%   beta:       Exponent relating Hydraulic gradient to discharge
%   n_Glen:     Glen's law exponent in creep closure relation
%   epsilon:    Regualrizer for the square-root dependence of discharge on
%               hydraulic gradient
%   uh:         List (n_nodes-by-one vector) of 'cavity' opening rates for
%               each edge
%   S_0_R:      cut-off size for cavity opening term for R conduits
%   S_0_K:      cut-off size for cavity opening term for pure cavity K
%               conduits
%   V_0_K:      cut-off size for cavity opening term for storage cavity K
%               terms, only needed if Kamb_storage is set (see below)
%   T:          List ((n_nodes-1)-by-one vector) of 'cavity' tortuosities
%               along each edge 
%   nu:         List ((n_nodes-1)-by-one vector) of reduction factors in
%               opening rates from 'cavity' to  'channel' conduits for each
%               edge
%  q_in_0:      water supply rate to each node (n_nodes-by-one array,
%               where n_nodes is the number of nodes in the network).
%               This is scaled in a time-dependent if qscaled is set, see
%               immediately below
%   qscaled, q_0:   qscaled is a flag, q_0 is a scaling parameter for
%               q_in_0. If qscaled is set, then water supply to each node
%               is given by q_in = q_0*q_in_0. q_0 is only used if qscaled
%               is set
%   Kamb_storage:Flag that sets whether a Kamb-type storage cavity volume
%               needs to be solved for at each node
%   uh_K:       List (n_nodes-by-one vector) of storage cavity opening
%               rates at each node
%   filename:    Filename to which output is saved, same format as output
%               from time-dependent runs created by network, with a single
%               time stamp and a single solution for N, S_R, S_K and (if
%               needed) V_K
%
%works as line_steady but includes percolation threshold


%unpack inputs
n_nodes = parameters.grid.n_nodes;        %number of nodes
n_edges = parameters.grid.n_edges;
N_bdy_nodes = parameters.N_bdy_nodes;   %Effective pressure at last node

%Check input
if n_edges ~= n_nodes-1, error('The non-periodic one-dimensional network must have one more node than it has edges'), end
if length(N_bdy_nodes) > 1, error('There must only be one Dirichlet node'), end

if ~isfield(parameters,'qscaled')|| ~parameters.qscaled || ~isfield(parameters,'q_0')
    parameters.q_in=parameters.q_in_0;
else
    parameters.q_in=parameters.q_0*parameters.q_in_0;
end
fout.parameters=parameters;
solpars = parameters;
fout.t = 0;
fout.N = zeros(n_nodes,1);
fout.S_R = zeros(n_edges,1);
fout.S_K = zeros(n_edges,1);
if parameters.Kamb_storage
    fout.V_K = zeros(n_nodes,1);
end
fout.N(n_nodes)=N_bdy_nodes;
for ii=n_nodes-1:-1:1
    solpars.N_plus=fout.N(ii+1);
    N=fout.N(ii+1);%+solpars.L(n_nodes-1);
    solpars.node=ii;
    if nargin ==2 && isfield(flags,'test') && flags.test && ii==round(n_nodes/2)
        fout.difftest=Jacobian_test_v2(@line_steady_function_v2,@line_steady_gradient_v2,rand(1),solpars,sqrt(eps));
    end
    srchparams.verbose=0;
    srchparams.itmax = 500;
    [N, error_flag, faux] = Newton(@line_steady_function_v2,@line_steady_gradient_v2,N,solpars,srchparams);
    fout.N(ii)=N;
    fout.S_R(ii)=faux.S_R;
    fout.S_K(ii)=faux.S_K;
    if parameters.Kamb_storage
        fout.V_K(ii)=faux.V_K;
    end
end
save(parameters.filename,'fout')