##==== HEADER ========================================================================
## 
## Purpose:
## This is a script to simulate snow cover diode effect
##
## Modification History:
## code generation -- April 2013, Hasi
##
## Autors: 
## Hasi (Andreas Hasler), andreas_hasler@gmx.ch, BC Forest Dept.
## 
## Description:		super simple numerical model for heat fluxes based on 
##					electrotechnical scheme.
##						
##======================================================================================

## clear current workspace to avoid env. dependency
#rm(list=ls())

##======================================================================================
##   control variables
##======================================================================================

## number of years to simulate
years <- 8

## temperature amplitude (peak value) and MAAT
A <- 10
MAAT <- -2

## define Rsnow and Rgr (only ratio relevant); set Rgr = 1
Rsnow <- 10
Rgr <- 1
Rvar <- F   ## T dependent snow insulation with Rsnow as peak value

## shift in snow cover (only with exact off)
shift <- 0

## define capacity behaving (Cgr=inf: Tgr = const, Cgr=0: Tgr = Tsurf); set 1000
Cgr <- 1000
## this means that Tgr will approach Tsurf by a dampening of 1/RC = 0.001 per time step
## note that the capacity value is dependent from time steps (here daily)!!!


## set Tsurf to zero if snow
lat <- F

## cal surface temp from voltage devider (only with shift = 0)
exact <- F

#path <- "/Users/andi/Documents/projekte/Canada/FieldData/"

## import of timeseries of daily mean temp with field "wx_Tair" in first row
import <- F
file <- ("/Tair_file.csv")

## set plot mode: "png","pdf" or "scr" (for screen)
pl <- "scr"

## legend T or F
legend <- T

##======================================================================================
##   preparation
##======================================================================================

## load toolbox
source("R_source/Ttools.R")

if (pl != "scr") print("input and output path is: "&getwd()&"/plots/thermaldiode")

##======================================================================================
## generate input and prepare output
##======================================================================================

## sinusodial temperature signal with amplitude A and wavelength 365 days
y <- 365
t <- 1:(years*y)
if (!import) Tair <- A * sin(t * (2*pi/y)) + MAAT else {
	d <- read.csv(file)
	d$t <- posix(d$t)
	d <- ts.normal(d,2000,2015,2001:(2000+years))
	Tair <- d$wx_Tair
	MAAT <- round(mean(Tair),2)
}

## create simple snow cover based on Tair with shift = x days
snow <- -1 * (MAAT + A * sin((t-shift) * (2*pi/y))) / (A-MAAT)
if (import) snow <- -1 * Tair / A
snow[snow<0] <- 0

## prepare vector for output with initial values
Tsurf <- Tair * 0 + MAAT
Tgr <- Tsurf

##======================================================================================
## timeloop
##======================================================================================

## loop with one time step per day
for (i in 2:(years*y)){
	if (snow[i] > 0) Rs <- Rsnow else Rs <- 0 
	if (Rvar) Rs <- Rsnow*snow[i]
	if (exact) if (Tair[i] > Tgr[i-1]) Rs <- 0 else Rs <- Rsnow
	if (!exact) Tsurf[i] <- ((Tair[i])/(Rs+Rgr)*Rgr)
	if (exact) Tsurf[i] <- ((Tair[i]-Tgr[i-1])/(Rs+Rgr)*Rgr)+Tgr[i-1]
	if ((snow[i] > 0) & (Tsurf[i]>0)&(lat)) Tsurf[i] <- 0
	#Tgr[i] <- Tgr[i-1] + ((Tsurf[i]-Tgr[i-1])*(1/(Cgr*Rgr))) no difference! < 0.01%
	Tgr[i] <- Tgr[i-1] + ((((Tsurf[i-1]+Tsurf[i])/2)-Tgr[i-1])*(1/(Cgr*Rgr))) }

##======================================================================================
## calc MAGT
##======================================================================================

st <- (years-1) * y
en <- years * y
MAGT <- round(mean(Tgr[st:en]),2)
SO <- MAGT - MAAT

##======================================================================================
##   ploting
##======================================================================================
	
title <- "A"&A&"_Rs"&Rsnow	
if (MAAT!=0) title <- "" & title &"_MAAT"&MAAT
if (lat) title <- "" & title &"_lat"
if (exact) title <- "" & title &"_ex"
if (shift>0) title <- "" & title &"_sh"&shift
if (Rvar) title <- "" & title &"_Rvar"

if(exact&(shift>0)) stop("dont use 'exact' and 'shift' at same run!!!")

	
## define grafic device size
plfn <- paste("plots/thermaldiode/",title,sep='')
pldev(x=4,y=4,pld=pl,plfn=plfn)

ylim <- c(-20,20)

plot(Tair,type = "l",xlab="days",ylab="T (degC)",ylim=ylim,main=title,col="grey")
lines(Tsurf,lty=2,col="black")
lines(Tgr,lty=3,col="black")
if (Rvar&!exact) lines(snow*Rsnow,col="blue")

abline(h=0)

text(en,20,"MAGT: ",cex=0.6)
text(en,18,""&MAGT&"°C",cex=0.6)
text(en,16,"SO: ",cex=0.6)
text(en,14,""&SO&"°C",cex=0.6)

legend("topleft",c("Tair","Tsurf","Tgr"),lty=1:3,cex=0.8,col=c("grey","black","black"),bty="n"
)

## close ploting dev for png or pdf
if (pl!="scr") dev.off()

##======================================================================================
##   end
##======================================================================================
