'''
Created on Aug 20, 2012
@author: Alexander H. Jarosch

This script re-creates Figure 3 in the paper:
Numerical mass conservation issues in shallow ice models of mountain glaciers: the use of flux limiters and a benchmark

by

Alexander H. Jarosch, Christian G. Schoof, and Faron S. Anslow
for review in The Cryosphere Discussions 2012 (tc-2012-143)

Copyright 2012 Alexander H. Jarosch

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>
'''

import numpy
import matplotlib.pyplot as plt

def main():
    
    # These parameters are defined in section 7.1
    A = 1e-16
    n = 3.
    g = 9.81
    rho = 910.
    Gamma = 2.*A*(rho*g)**n / (n+2) # we introduce Gamma to shorten the equations.
    mdot_0 = 2.
    x_m = 20000.
    x_s = 7000.
    b_0 = 500.
    c_stab = 0.165
    
    dx = 200.
    dt = 1.0
    t_total = 50000.
    Nt = t_total/dt
    
    # create a space vector
    x = numpy.arange(0.,x_m + x_m/2. + dx,dx)
    # define the mass balance vector
    m_dot = accumulation(x, mdot_0, n, x_m)
    # define bed vector. Here we use capital B instead of b.
    B = numpy.zeros(len(x))
    B[x<x_s]=b_0
    
    # set the intial ice surface S to the bed elevation. Again we use capital S instead of s.
    S = B
    Nx = len(S)
    
    # finite difference indixes. k is used in this 1D example. 
    # Note that python starts counting vector entries with 0 instead of 1, as e.g. MATLAB does
    k = numpy.arange(0,Nx)
    kp = numpy.hstack([numpy.arange(1,Nx),Nx-1])
    kpp = numpy.hstack([numpy.arange(2,Nx),Nx-1,Nx-1])
    km = numpy.hstack([0,numpy.arange(0,Nx-1)])
    kmm = numpy.hstack([0,0,numpy.arange(0,Nx-2)])
    
    # MUSCL scheme loop
    for t in range(int(Nt)+1):
        stab_t = 0.
        while stab_t < dt:
            H = S - B
            
            # MUSCL scheme up. "up" denotes here the k+1/2 flux boundary
            r_up_m = (H[k]-H[km])/(H[kp]-H[k])                      # Eq. 23
            H_up_m = H[k] + 0.5 * phi(r_up_m)*(H[kp]-H[k])          # Eq. 19
            r_up_p = (H[kp]-H[k])/(H[kpp]-H[kp])                    # Eq. 23, now k+1 is used instead of k
            H_up_p = H[kp] - 0.5 * phi(r_up_p)*(H[kpp]-H[kp])       # Eq. 20
            
            # surface slope gradient
            s_grad_up = ((S[kp]-S[k])**2. / dx**2.)**((n-1.)/2.)
            D_up_m = Gamma * H_up_m**(n+2.) * s_grad_up             # like Eq. 26, now using Eq. 19 instead of Eq. 20
            D_up_p = Gamma * H_up_p**(n+2.) * s_grad_up             # Eq. 26
            
            D_up_min = numpy.minimum(D_up_m,D_up_p);                # Eq. 27
            D_up_max = numpy.maximum(D_up_m,D_up_p);                # Eq. 28
            D_up = numpy.zeros(Nx)
            
            # Eq. 29 
            D_up[numpy.logical_and(S[kp]<=S[k],H_up_m<=H_up_p)] = D_up_min[numpy.logical_and(S[kp]<=S[k],H_up_m<=H_up_p)]
            D_up[numpy.logical_and(S[kp]<=S[k],H_up_m>H_up_p)] = D_up_max[numpy.logical_and(S[kp]<=S[k],H_up_m>H_up_p)]
            D_up[numpy.logical_and(S[kp]>S[k],H_up_m<=H_up_p)] = D_up_max[numpy.logical_and(S[kp]>S[k],H_up_m<=H_up_p)]
            D_up[numpy.logical_and(S[kp]>S[k],H_up_m>H_up_p)] = D_up_min[numpy.logical_and(S[kp]>S[k],H_up_m>H_up_p)]

            # MUSCL scheme down. "down" denotes here the k-1/2 flux boundary
            r_dn_m = (H[km]-H[kmm])/(H[k]-H[km])
            H_dn_m = H[km] + 0.5 * phi(r_dn_m)*(H[k]-H[km])
            r_dn_p = (H[k]-H[km])/(H[kp]-H[k])
            H_dn_p = H[k] - 0.5 * phi(r_dn_p)*(H[kp]-H[k])
            
            # calculate the slope gradient
            s_grad_dn = ((S[k]-S[km])**2. / dx**2.)**((n-1.)/2.)
            D_dn_m = Gamma * H_dn_m**(n+2.) * s_grad_dn
            D_dn_p = Gamma * H_dn_p**(n+2.) * s_grad_dn
            
            D_dn_min = numpy.minimum(D_dn_m,D_dn_p);
            D_dn_max = numpy.maximum(D_dn_m,D_dn_p);
            D_dn = numpy.zeros(Nx)
            
            D_dn[numpy.logical_and(S[k]<=S[km],H_dn_m<=H_dn_p)] = D_dn_min[numpy.logical_and(S[k]<=S[km],H_dn_m<=H_dn_p)]
            D_dn[numpy.logical_and(S[k]<=S[km],H_dn_m>H_dn_p)] = D_dn_max[numpy.logical_and(S[k]<=S[km],H_dn_m>H_dn_p)]
            D_dn[numpy.logical_and(S[k]>S[km],H_dn_m<=H_dn_p)] = D_dn_max[numpy.logical_and(S[k]>S[km],H_dn_m<=H_dn_p)]
            D_dn[numpy.logical_and(S[k]>S[km],H_dn_m>H_dn_p)] = D_dn_min[numpy.logical_and(S[k]>S[km],H_dn_m>H_dn_p)]
            
            dt_stab = c_stab * dx**2. / max(max(abs(D_up)),max(abs(D_dn)))      # Eq. 33
            dt_use = min(dt_stab,dt-stab_t)
            stab_t = stab_t + dt_use
            
            #explicit time stepping scheme
            div_q = (D_up * (S[kp] - S[k])/dx - D_dn * (S[k] - S[km])/dx)/dx    # Eq. 32
            S = S[k] + (m_dot + div_q)*dt_use                                   # Eq. 31
            
            S = numpy.maximum(S,B)                                              # Eq. 7
        
            print 'MUSCL Year %d, dt_use %f' % (t,dt_use)
    
    
        if numpy.mod(t,1000.)==0.0:
            plt.plot(x/1000,S,'-b')
            
    # calculate the volume difference
    p1, = plt.plot(x/1000,S,'-b')
    H = S-B
    vol_muscl = numpy.trapz(H,x)
    
    # Type I loop, which performs the same benchmark, just with a type I scheme
    S = B     
    for t in range(int(Nt)+1):
        stab_t = 0.
        while stab_t < dt:
            H = S - B
            
            H_up = 0.5 * ( H[kp] + H[k] )
            H_dn = 0.5 * ( H[k] + H[km] )
            s_grad_up = ((S[kp]-S[k])**2. / dx**2.)**((n-1.)/2.)
            s_grad_dn = ((S[k]-S[km])**2. / dx**2.)**((n-1.)/2.)
            D_up = Gamma * H_up**(n+2) * s_grad_up
            D_dn = Gamma * H_dn**(n+2) * s_grad_dn
            
            
            dt_stab = c_stab * dx**2. / max(max(abs(D_up)),max(abs(D_dn)))
            dt_use = min(dt_stab,dt-stab_t)
            stab_t = stab_t + dt_use
            
            #explicit time stepping scheme
            div_q = (D_up * (S[kp] - S[k])/dx - D_dn * (S[k] - S[km])/dx)/dx
            S = S[k] + (m_dot + div_q)*dt_use
            
            S = numpy.maximum(S,B)    
        
            print 'TypeI Year %d, dt_use %f' % (t,dt_use)
    
    
        if numpy.mod(t,1000.)==0.0:
            plt.plot(x/1000,S,'-r')
            
    # calculate the volume difference
    p2, = plt.plot(x/1000,S,'-r')
    H = S-B
    vol_typeI = numpy.trapz(H,x)
                
    # create the explicit steady state solution form section 6, which we call s here
    s_x_s_x_m = s_eval_x_s_x_m(x,x_s,x_m,n,A,mdot_0,rho,g)
    s_x_s = s_eval_x_s(x,x_s,x_m,n,A,mdot_0,rho,g,b_0)
    # combine the solutions
    s = s_x_s+b_0
    s[x >= x_s] = s_x_s_x_m[x >= x_s]
    # correct s
    s[x>x_m] = 0.
    
    h = s-B
    vol_exact = numpy.trapz(h,x)
    vol_err_muscl = (vol_muscl-vol_exact)/vol_exact*100
    vol_err_typeI = (vol_typeI-vol_exact)/vol_exact*100
    
    print "vol exact: %e" % vol_exact
    print "vol muscl: %e" % vol_muscl
    print "vol typeI: %e" % vol_typeI
    print "err muscl %0.3f" % vol_err_muscl
    print "err typeI %0.3f" % vol_err_typeI
    
    p3, = plt.plot(x/1000,B,'-k',linewidth=2)
    p4, = plt.plot(x/1000,s,'-m',linewidth=2)
    plt.xlabel('x [km]')
    plt.ylabel('z [m]')
    plt.title('50000 years evolution')
    plt.legend([p1, p2, p3, p4], ["MUSCL superbee", "Type I", "bed", "Eqs. (52) and (53)"])
    plt.show()

    
    
def accumulation(x,mdot_0,n,x_m):
    # Eq. 50
    mdot = ((n*mdot_0)/(x_m**(2.*n-1.)))*x**(n-1.)*(abs(x_m-x)**(n-2.))*(x_m-x)*(abs(x_m-x)-x)
    
    mdot[x>x_m] = 0.
    
    return mdot 
    
def s_eval_x_s_x_m(x,x_s,x_m,n,A,mdot_0,rho,g):
    # Eq. 52
    s_x_s_x_m = (((2.*n+2.)*(n+2.)**(1./n)*mdot_0**(1./n))/(24.*A**(1./n)*rho*g*x_m**((2.*n-1)/n))*(x_m+2.*x)*(x_m-x)**2.)**(n/(2.*n+2.))
    
    return s_x_s_x_m
    
def s_eval_x_s(x,x_s,x_m,n,A,mdot_0,rho,g,b_0):
    # Eq. 54 
    h_splus = (((2.*n+2.)*(n+2.)**(1./n)*mdot_0**(1./n))/(24.*A**(1./n)*rho*g*x_m**((2.*n-1.)/n))*(x_m+2.*x_s)*(x_m-x_s)**2.)**(n/(2.*n+2.))
    # Eq. 55
    h_sminus = numpy.maximum(h_splus - b_0,0.)
    # Eq. 53
    h_back = (h_sminus**((2.*n+2.)/n)-h_splus**((2.*n+2.)/n)+((2.*n+2.)*(n+2.)**(1./n)*mdot_0**(1./n))/(24.*A**(1./n)*rho*g*x_m**((2.*n-1.)/n))*(x_m+2.*x)*(x_m-x)**2.)**(n/(2.*n+2.))
    
    return h_back

def phi(r):
    
    # minmod limiter Eq. 24
#    val_phi = numpy.maximum(0,numpy.minimum(1,r))
    
    # superbee limiter Eq. 25
    val_phi = numpy.maximum(0,numpy.minimum(2*r,1),numpy.minimum(r,2))
    
    return val_phi

''' DEFINE which routine to run as the main '''

if __name__=="__main__":
    main()
