##==== HEADER ========================================================================
## 
## Purpose:
## This is a script to plot temperatur data from British Columbia
## Ministry of Forest, Land and Natural Resources Operation
##
## Modification History:
## code generation -- August 2013, Hasi
##
## Autors: 
## Hasi (Andreas Hasler), andreas_hasler@gmx.ch, BC Forest Dept.
## 
## Description:		Plots of SO and TO grouped by surface type.
##
## Dependencies:		Ttools.R (own toolbox for temperature timeseries treatment,
## 				(contained in STATws.R workspace)
##
## I/O, variables and units:
## input:			STATws.RData workspace (generated by procedure "stat_TTSMAT.R")
##            with dfs SOTO_XXX
##				    toolbox Ttools.R
##		
## output:			barplots
##
## units:			temperature in °C
##						
##======================================================================================

## clear current workspace to avoid env. dependency
#rm(list=ls())

##======================================================================================
##   control variables
##======================================================================================

## relative path of profile list
proflist <- "meta/SO_TO_proflist.csv"

## set plot mode: "png","pdf", "scr" (for screen), or "stu" (for R studio plot pane)
pl <- "scr"

usedsites <- c("GUN","HUD","MID","NON","PIN","POP","TET")


##======================================================================================
##   preparation
##======================================================================================

## load workspace with all data and tools 
if (!exists("STAWS")) load("RData/STATws.RData")

## load toolbox "Ttools" (requires R-library "caTools")
source("R_source/Ttools.R")

## get list of sites:
sites <- usedsites
lsl <- length(sites) ## length of short list

## get profile lists 
prlist <- read.csv(proflist, as.is=TRUE,sep=";")
if(dim(prlist)[2]==1) prlist <- read.csv(proflist, as.is=TRUE)

ord <- prlist$order
typ <- prlist$surftype

##======================================================================================
##======================================================================================
##======================================================================================
##   main loop through sites

for (k in 1:lsl){	
	## get site name	
	site <- sites[k]
	sitn <- (slist$sitename[site==slist$site])[1]
	print("ploting "&site)
	
	## getting dataframe SOTO of respective sensor assigned to "d"
	dfn <- "SOTO_" & site 
  d <- get(dfn)
  cols <- dim(d)[2]
  
  locx <- names(d)[2:cols]
  locx <- paste(site,locx,sep="_")
  SOx <- as.numeric(d[1,2:cols])
  dTx <- as.numeric(d[2,2:cols])
	TOx <- as.numeric(d[3,2:cols])
	vSOx <- as.numeric(d[4,2:cols])
	vdTx <- as.numeric(d[5,2:cols])
	vTOx <- as.numeric(d[6,2:cols])
  
  if (k==1) {
  loc <- locx
  SO <- SOx
  dT <- dTx
  TO <- TOx
  vSO <- vSOx
  vdT <- vdTx
  vTO <- vTOx
  } else {
  loc <- c(loc,locx)  
  SO <- c(SO,SOx)
  dT <- c(dT,dTx)
  TO <- c(TO,TOx)
  vSO <- c(vSO,vSOx)
  vdT <- c(vdT,vdTx)
  vTO <- c(vTO,vTOx)
  }
  
  SO[is.na(SO)] <- dT[is.na(SO)]
	vSO[is.na(vSO)] <- vdT[is.na(vSO)]
  
}#   END of main loop through the sites


##======================================================================================
##   ordering by surf type
##======================================================================================
d <- data.frame(loc=loc,SO = SO, TO=TO, vSO=vSO,vTO=vTO,typ=typ,ord=ord)
d <- df.sort(d,"ord")
loc <- d$loc
typ <-d$typ
SO <- d$SO
TO <- d$TO
vSO <- d$vSO/2
vTO <- d$vTO/2

##======================================================================================
##   ploting
##======================================================================================

## define grafic device name and size
plfn <- "plots/SOTO"
pldev(x=8,y=4,pld=pl,plfn=plfn)
## make rows and cols on sheet c(rows,cols):
par(mfrow=c(2,1),xpd=FALSE,cex=0.9,mar=c(3,4,2,1))
## mar is c(bot,left,top,r)

## plot parameters 
col <- as.character(typ)
col[typ=="r"] <- "grey"
col[typ=="s"] <- "brown"
col[typ=="d"] <- "lightgrey"
col[typ=="f"] <- "darkgreen"

## end of not used **************

## plot SO
ylim = c(-1,7)
ylab = "SO (°C)"
barpl <- barplot(SO,ylim=ylim,ylab=ylab,width=0.9,space=0.11,col=col)
axis(1,at=(1:length(loc)),loc,las=2,padj=-0.5,tck=0, lwd=0,cex.axis=0.7)
arrows(barpl,SO-(vSO),barpl,SO+(vSO),code=3,angle=90,length=0.04,lwd=1)
abline(h=0)
abline(h=c(2,4,6),lty=2)

## plot TO
ylim = c(-4,4)
ylab = "TO (°C)"
barpl <- barplot(TO,ylim=ylim,ylab=ylab,col=col)
arrows(barpl,TO-(vTO),barpl,TO+(vTO),code=3,angle=90,length=0.04,lwd=1)
abline(h=0)
abline(h=c(-4,-2,2,4),lty=2)

## close ploting dev for png or pdf
if ((pl!="scr")&(pl!="stu"))  dev.off()

##======================================================================================
##======================================================================================
##   end
##======================================================================================
