% This script makes figure 7 of the Greene et al paper on methods
% to recover seasonal ice dynamics from ITS_LIVE data. 
% Chad A. Greene, April 2020. 

% This script analyzes GPS data from PROMICE gps station KAN_L. 

%% Map of all stations: 
% There's need to run this section unless you want to.

if false 
load promice_GPS.mat 
stn = fieldnames(promice); 

figure
hold on

for stni = 1:length(stn)
   t = promice.(stn{stni}).matdate; 
   x = promice.(stn{stni}).x_psn_m; 
   y = promice.(stn{stni}).y_psn_m; 

   scatter(x,y,10,t,'filled')
   text(median(x,'omitnan'),median(y,'omitnan'),stn{stni},'horiz','center','vert','top','color','b','clipping','on')
end

greenland % Arctic mapping tools function plots and outline of greenland.

cb = colorbar; 
datetick(cb,'y','yyyy')
end

%% Offsets 
%           last / first reliable time 
% 22-Aug-12	NA	   16:00
% 06-Jul-15 
% 28-Apr-15	11:00	13:00
% 16-Jul-16	15:00	17:00
% 1-Sep-17	17:00	20:00
% 28-Aug-18	19:00	21:00

%% Load KAN_L data: 

load promice_GPS.mat

stn = fieldnames(promice); %  station names 
stni=3; % KAN_L is station 3

t = promice.(stn{stni}).matdate; 
xps = promice.(stn{stni}).x_psn_m; 
yps = promice.(stn{stni}).y_psn_m; 
hdop = promice.(stn{stni}).HorDilOfPrecGPS;

% Convert to UTM b/c ITS_LIVE for Greenland solves velocities in UTM directions 
[lat,lon] = psn2ll(xps,yps); 
[xu,yu,utmzone] = ll2utm(lat,lon);

clear xps yps % I'll miss you, polar stereographic coordinates.

%% Fix offsets:
% Here I'm calculating offsets as daily avearge positions before and after
% the step change. Hopefully by averaging ~24 hours of position data we're
% not as subject to noise, as we might be if we just considered the hourly
% measurements before and after the step change. 
% 
% After removing the difference between the two daily average positions, 
% the typical daily displacement is added back. 

% Get typical daily displacement: 
px_tmp = polyfit(t(isfinite(xu)),xu(isfinite(xu)),1);
py_tmp = polyfit(t(isfinite(yu)),yu(isfinite(yu)),1);
xrate = px_tmp(1); % m per day 
yrate = py_tmp(1); 

ind = 59997; % 06-Jul-2015 20:00:00
xb4 = mean(xu(ind-25:ind-1),'omitnan'); % Average position the day before the offset  
xaft = mean(xu(ind:ind+23),'omitnan');  % Average position the day after the offset 
xu(ind:end) = xu(ind:end) - xaft + xb4 + xrate; % subtract the difference between the two days' average positions, then add back the aount of displacement we expect for a typical day day anyway 
yb4 = mean(yu(ind-25:ind-1),'omitnan'); 
yaft = mean(yu(ind:ind+23),'omitnan'); 
yu(ind:end) = yu(ind:end) - yaft + yb4 + yrate; 

ind = 69018; % 16-Jul-2016 17:00:00
xb4 = mean(xu(ind-25:ind-1),'omitnan'); 
xaft = mean(xu(ind:ind+23),'omitnan'); 
xu(ind:end) = xu(ind:end) - xaft + xb4 + xrate; 
yb4 = mean(yu(ind-25:ind-1),'omitnan'); 
yaft = mean(yu(ind:ind+23),'omitnan'); 
yu(ind:end) = yu(ind:end) - yaft + yb4 + yrate;

ind = 78908; % 1-Sept-2017 21:00:00
xb4 = mean(xu(ind-25:ind-1),'omitnan'); 
xaft = mean(xu(ind:ind+23),'omitnan'); 
xu(ind:end) = xu(ind:end) - xaft + xb4 + xrate;
yb4 = mean(yu(ind-25:ind-1),'omitnan'); 
yaft = mean(yu(ind:ind+23),'omitnan'); 
yu(ind:end) = yu(ind:end) - yaft + yb4 + yrate; 

% ind = 87574; % 28-Aug-2018 21:00:00
% xb4 = mean(xu(ind-25:ind-1),'omitnan'); 
% xaft = mean(xu(ind:ind+23),'omitnan'); 
% xu(ind:end) = xu(ind:end) - xaft + xb4 + xrate;
% yb4 = mean(yu(ind-25:ind-1),'omitnan'); 
% yaft = mean(yu(ind:ind+23),'omitnan'); 
% yu(ind:end) = yu(ind:end) - yaft + yb4 + yrate; 

%% Remove bad data: 

% Remove constant values related to data logger issues: 
bad = find(diff(xu)==0 & diff(yu)==0)+1; 
xu(bad) = []; 
yu(bad) = []; 
t(bad) = []; 
hdop(bad) = []; 

% Remove other bad or missing data: 
bad = hdop<0 | hdop>5 | isnan(xu) | isnan(yu) | isnan(t); 
xu(bad) = []; 
yu(bad) = []; 
t(bad) = []; 

% Calculate trends so we can detrend and remove outliers: 
[px,Sx,mux] = polyfit(t,xu,1); 
[py,Sy,muy] = polyfit(t,yu,1); 
x_trend = polyval(px,t,Sx,mux); % Just the trend part of x (will come in handy later, in addition to detrending now) 
y_trend = polyval(py,t,Sy,muy); 
x_dt = xu - x_trend; % detrended x
y_dt = yu - y_trend;

% Define outliers as >2.5 robust standard deviations away from the trend (robust std is 1.4826*MAD)
outliers = abs(x_dt)>2.5*median(abs(x_dt))*1.4826 | abs(y_dt)>2.5*median(abs(y_dt))*1.4826; 
xu(outliers) = []; 
yu(outliers) = []; 
t(outliers) = []; 
x_trend(outliers) = []; 
y_trend(outliers) = []; 
x_dt(outliers) = []; 
y_dt(outliers) = []; 

% Check on our progress:
figure
subplot(3,2,1) 
plot(t,x_dt,'.')
axis tight
ylabel 'detrended x positon (m)'
datetick('x','yyyy','keeplimits') 

subplot(3,2,2) 
plot(t,y_dt,'.')
axis tight
ylabel 'detrended y positon (m)'
datetick('x','yyyy','keeplimits') 

%% Calculate interannual variabilty in detrended position 
% Some years the ice moves faster than other years, and the detrended
% positions keep a record of that by offsetting the seasonal cycles of 
% some years relative to other years. To assess the typical seasonal cycle, 
% we need to remove those offsets by assessing interannual variability and
% subtracting it. 
% 
% To assess interannual variability, loop through each year in the time series. 
% For each year, calculate the mean position and assign it to the mean time
% of all measurements taken that year. Then do a pchip interpolation (because 
% it's smooth and it's willing to extrapolate) to calculate the amount of 
% interannual variability contributing to each position measurement. 

[yr,~,~] = datevec(t); % gives the year corresponding to each measurement.
uyr = unique(yr);      % a list of the years that have any data.

% Preallocate: 
tyr = nan(size(uyr)); 
xyr = tyr; 
yyr = tyr; 

% Loop through each year: 
for k = 1:length(uyr)
   
   % Get the indices of all data this year: 
   ind = yr==uyr(k); 
   
   % Calculate the mean positions and corresponding mean time for each year:
   xyr(k) = mean(x_dt(ind)); 
   yyr(k) = mean(y_dt(ind)); 
   tyr(k) = mean(t(ind)); 
end

% Interpolate to get the "interannual" component of each detrended measurement: 
x_int = interp1(tyr,xyr,t,'pchip'); 
y_int = interp1(tyr,yyr,t,'pchip'); 

% Add to the figure we started earlier: 
subplot(3,2,1) 
hold on
plot(t,x_int,'.') % plots interannual variability

subplot(3,2,2) 
hold on
plot(t,y_int,'.')

% Now plot detrended positions with interannual variability removed: 
subplot(3,2,3) 
plot(t,x_dt-x_int,'.')
axis tight
datetick('x','yyyy','keeplimits') 
ylabel 'detrended x minus interannual (m)'

subplot(3,2,4) 
plot(t,y_dt-y_int,'.')
axis tight
datetick('x','yyyy','keeplimits') 
ylabel 'detrended y minus interannual (m)'

%% Get sinusoid fits to positions: 
% Now we can remove interannual variability from the detrended positions 
% and fit sinusoids to get the seasonal component of x and y: 

% Fit sinusoids to the position data, using the sinefit function from Climate Data Toolbox:  
ft_gps_x = sinefit(t,x_dt-x_int); % ft format is [Amplitude DOY_of_max]
ft_gps_y = sinefit(t,y_dt-y_int);

% Convert position sinusoids from position to velocity using calc/trig identity:  
ft_gps_vx = [ft_gps_x(1)*2*pi ft_gps_x(2)-365.25/4];
ft_gps_vy = [ft_gps_y(1)*2*pi ft_gps_y(2)-365.25/4];

% Wrap "phase" (in days) to days of year: 
if ft_gps_vx(2)<=0
   ft_gps_vx(2) = ft_gps_vx(2)+365.25; 
end
if ft_gps_vy(2)<=0
   ft_gps_vy(2) = ft_gps_vy(2)+365.25; 
end
if ft_gps_vx(2)>365.25
   ft_gps_vx(2) = ft_gps_vx(2)-365.25; 
end
if ft_gps_vy(2)>365.25
   ft_gps_vy(2) = ft_gps_vy(2)-365.25; 
end

% Plot sinusoids on the plots we've already started: 
subplot(3,2,3) 
hold on
plot(t,sineval(ft_gps_x,t),'.')

subplot(3,2,4) 
hold on
plot(t,sineval(ft_gps_y,t),'.')

%% Second iteration of interannual/sinefit: 
% It's possible that some of what we interpreted earlier as interannual variability
% was actually just aliased seasonal variability. So remove our first guess
% at the seasonal component and try again: 

if false 
   % Loop through each year: 
   for k = 1:length(uyr)

      % Get the indices of all data this year: 
      ind = yr==uyr(k); 

      % Calculate the mean positions and corresponding mean time for each year:
      xyr(k) = mean(x_dt(ind)-sineval(ft_gps_x,t(ind))); 
      yyr(k) = mean(y_dt(ind)-sineval(ft_gps_y,t(ind))); 
      tyr(k) = mean(t(ind)); 
   end

   % Interpolate to get the "interannual" component of each detrended measurement: 
   x_int = interp1(tyr,xyr,t,'pchip'); 
   y_int = interp1(tyr,yyr,t,'pchip'); 

   % Fit sinusoids to the position data, using the sinefit function from Climate Data Toolbox:  
   ft_gps_x = sinefit(t,x_dt-x_int); % ft format is [Amplitude DOY_of_max]
   ft_gps_y = sinefit(t,y_dt-y_int);

   % Convert position sinusoids from position to velocity using calc/trig identity:  
   ft_gps_vx = [ft_gps_x(1)*2*pi ft_gps_x(2)-365.25/4];
   ft_gps_vy = [ft_gps_y(1)*2*pi ft_gps_y(2)-365.25/4];
end

%% Convert positions to velocities: 

% Let's now say that the x and y positions are equal to their linear 
% trends plus interannual variability plus seasonal sinusoid fits: 
x_fit = x_trend + x_int + sineval(ft_gps_x,t); 
y_fit = y_trend + y_int + sineval(ft_gps_y,t);

% Then of course velocity is just gradient in position over gradient in time:
vx_fit = gradient(x_fit)./gradient(t/365.25);
vy_fit = gradient(y_fit)./gradient(t/365.25); % equivalent to: gradient(y_trend+y_int)./gradient(t/365.25) + sineval(ft_gps_vy,t) 

% Plot velocities: 
subplot(3,2,5) 
plot(t,vx_fit,'.') 
axis tight
datetick('x','yyyy','keeplimits') 
ylabel 'x velocity fit (m/yr)'

subplot(3,2,6) 
plot(t,vy_fit,'.') 
axis tight
datetick('x','yyyy','keeplimits') 
ylabel 'y velocity fit (m/yr)'

% export_fig KAN_L_processing_steps.png -r300 

%% ITS_LIVE data 
% Lots of secret Chad functions are used below this; sorry: 

% Load the ITS_LIVE time series from the grid cell closest to the median KAN_L location: 
L = load('point_ts_KAN_L.mat');

% Indices of ITS_LIVE image pairs corresponding to GPS timespan, with a little extra for proper assessment of interannual variability: 
ind = L.t(:,2)>min(t-800) & L.t(:,1)<max(t); 

% Calculate amplitude and phase of seasonality: 
[Ax,phx] = itslive_seasonal(L.t(ind,:),L.vx(ind),L.vx_err(ind));
[Ay,phy] = itslive_seasonal(L.t(ind,:),L.vy(ind),L.vy_err(ind));

% Print both GPS vs ITS_LIVE, just for comparison: 
[Ax phx;
ft_gps_vx]

[Ay phy;
ft_gps_vy]

% Create a continuous time array: 
ti = min(L.t(ind,1)):max(L.t(ind,2));

% Get continuous time series in x and y: 
vx_its = itslive_tsinterp(L.t(ind,:),L.vx(ind),L.vx_err(ind),ti);
vy_its = itslive_tsinterp(L.t(ind,:),L.vy(ind),L.vy_err(ind),ti);

% Secular velocities from gps should just be from the raw position data 
px = polyfit(t/365.25,xu,1); % the first output is m/yr
py = polyfit(t/365.25,yu,1);
mean(vx_its(ti>=min(t) & ti<=max(t))); % mean itslive velocity for the same time period
mean(vy_its(ti>=min(t) & ti<=max(t))); 

% Table 2: 
round([px(1) mean(vx_its(ti>=min(t) & ti<=max(t))) mean(vx_its(ti>=min(t) & ti<=max(t)))-px(1); 
   ft_gps_vx(1) Ax Ax-ft_gps_vx(1); 
   ft_gps_vx(2) phx phx-ft_gps_vx(2); 
   py(1) mean(vy_its(ti>=min(t) & ti<=max(t))) mean(vy_its(ti>=min(t) & ti<=max(t)))-py(1);
   ft_gps_vy(1) Ay Ay-ft_gps_vy(1); 
   ft_gps_vy(2) phy phy-ft_gps_vy(2)],1)

% Could calculate secular velocities as wmean(L.vx(ind),1./L.vx_err(ind).^2) but that
% would be influenced by seasonal bias in measurements. And we can't really say the 
% secular velocity uncertainty is related to the formal sum of weights of all the 
% constituent image pair measurements, because that would assume all 5000+ measurement
% covered the entire timespan, when in fact only 20 image pairs cover any given
% date on average. 

% interannual x std is 3.85 m/yr and y is 3.38 m/yr: 
[Ax_sigma,phx_sigma] = itslive_seasonal_unc(sum(ind),3.85,Ax)
[Ay_sigma,phy_sigma] = itslive_seasonal_unc(sum(ind),3.38,Ay)

%% Plot GPS data with ITS_LIVE data: 

% blue
gps_raw_color = [0.65          0.81          0.89]; 
gps_fit_color = [0.12          0.47          0.71];

itslive_bar_color = 0.7*[1 1 1]; 
itslive_dot_color = 0.2*[1 1 1];
itslive_fit_color = [0.89          0.10          0.11]; 

gps_raw_color = hex2rgb('a4d0ff');
gps_fit_color = hex2rgb('3c78e4'); 

%itslive_bar_color = hex2rgb('d8ccbd');
%itslive_dot_color = hex2rgb('42382e'); 
itslive_fit_color = hex2rgb('d83b2d'); 

MarkerSize = 0.1; 
fs = 7; % fontsize
vp = 0.03; % vertical padding between plots
hp = 0.02; 

figure('pos',[61.00   300  607  241])

subsubplot(2,2,1,'hpad',hp,'vpad',vp)
plot(t,x_dt,'.','color',gps_raw_color,'MarkerSize',MarkerSize)
hold on
plot(t,sineval(ft_gps_x,t)+x_int,'.','color',gps_fit_color,'MarkerSize',MarkerSize*30)
box off
axis tight
ylabel 'GPS detrended position (m)'
ax(1) = gca; 
xlim([min(t)-365 max(t)]) % t from gps 
ylim([-15 15])
set(gca,'fontsize',fs,'xtick',datenum(2000:2020,1,1))
datetick('x','yyyy','keeplimits','keepticks') 
ntitle(' a ','fontsize',fs+1,'fontweight','bold','location','northwest')
title('UTM easting','fontsize',fs+1)
set(gca,'xcolor',get(ax(1),'ycolor'))

subsubplot(2,2,2,'hpad',hp,'vpad',vp)
plot(t,y_dt,'.','color',gps_raw_color,'MarkerSize',MarkerSize)
hold on
plot(t,sineval(ft_gps_y,t)+y_int,'.','color',gps_fit_color,'MarkerSize',MarkerSize*30)
box off
axis tight
ax(2) = gca; 
xlim([min(t)-365 max(t)]) % t from gps 
ylim([-15 15])
set(gca,'fontsize',fs,'xtick',datenum(2000:2020,1,1))
datetick('x','yyyy','keeplimits','keepticks') 
ntitle(' b ','fontsize',fs+1,'fontweight','bold','location','northwest')
title('UTM northing','fontsize',fs+1)
set(gca,'ycolor',get(ax(1),'ycolor'),'xcolor',get(ax(1),'ycolor'))

subsubplot(2,2,3,'hpad',hp,'vpad',vp)
[h1x,h2x,h3x]=itslive_tsplot(L.t(ind,:),L.vx(ind),L.vx_err(ind),'datenum');
box off
axis tight
hold on
plot(ti,vx_its,'linewidth',1,'color',itslive_fit_color)
plot(t,vx_fit,'.','color',gps_fit_color,'MarkerSize',MarkerSize*30)
ylabel 'Velocity (m yr^{-1})'
ax(3) = gca; 
xlim([min(t)-365 max(t)]) % t from gps 
ylim(-111 + 100*[-1 1])
set(gca,'fontsize',fs,'xtick',datenum(2000:2020,1,1))
datetick('x','yyyy','keeplimits','keepticks') 
ntitle(' c ','fontsize',fs+1,'fontweight','bold','location','northwest')
set(gca,'ycolor',get(ax(1),'ycolor'))

subsubplot(2,2,4,'hpad',hp,'vpad',vp)
[h1y,h2y,h3y]=itslive_tsplot(L.t(ind,:),L.vy(ind),L.vy_err(ind),'datenum');
box off
axis tight
xlim([min(t) max(t)])
hold on
plot(ti,vy_its,'linewidth',1,'color',itslive_fit_color)
plot(t,vy_fit,'.','color',gps_fit_color,'MarkerSize',MarkerSize*30)
ax(4) = gca; 
xlim([min(t)-365 max(t)]) % t from gps 
ylim(-30 + 100*[-1 1])
set(gca,'fontsize',fs,'xtick',datenum(2000:2020,1,1))
datetick('x','yyyy','keeplimits','keepticks') 
ntitle(' d ','fontsize',fs+1,'fontweight','bold','location','northwest')
set(gca,'ycolor',get(ax(1),'ycolor'))

% Set colors: 
h1x.MarkerSize = 0.1; 
h1x.Color = itslive_dot_color;
set(h2x(:),'LineWidth',0.1,'color',itslive_bar_color); 
set(h3x(:),'LineWidth',0.1,'color',itslive_bar_color); 

h1y.MarkerSize = 0.1; 
h1y.Color = itslive_dot_color;
set(h2y(:),'LineWidth',0.1,'color',itslive_bar_color); 
set(h3y(:),'LineWidth',0.1,'color',itslive_bar_color); 

ax(3).Position(4) = ax(3).Position(4)*1.2; 
ax(1).Position(4) = ax(1).Position(4)*.7;
ax(1).Position(2) = ax(1).Position(2)+0.3*ax(1).Position(4); 

ax(4).Position(4) = ax(4).Position(4)*1.2; 
ax(2).Position(4) = ax(2).Position(4)*.7;
ax(2).Position(2) = ax(2).Position(2)+0.3*ax(2).Position(4); 

ax(3).Position(2) = ax(3).Position(2)-0.01; 
ax(4).Position(2) = ax(4).Position(2)-0.01; 
% export_fig KAN_L_itslive_GPS.png -r600 -painters
