% This script makes figure 4 of the Greene et al paper on methods
% to recover seasonal ice dynamics from ITS_LIVE data. 
% This script walks through the steps of analyzing ITS_LIVE data for 
% seasonality.
% Let me know if you have any questions. chad@chadagreene.com
% Chad A. Greene, April 2020

% Note to self: This is from itslive_seasonal_methods.m, October 17, 2019. 

%% 

savefigures = false; 
fs = 7; % fontsize 

%% Load data

load byrd_test_data % velocity time series from an arbitrary 240x240 m pixel near the Byrd Glacier grounding line

ind = t(:,1)>datenum('june 1, 2006'); 
t = t(ind,:); 
v = v(ind); 
v_err = v_err(ind); 

tm = mean(t,2); % mean times
ti = min(t(:,1)):max(t(:,2)); 
tid = datetime(ti,'convertfrom','datenum'); 

%% Plot the raw data 

%col = brewermap(8,'Set1'); % colormap (Editor discouraged this)
col = hex2rgb({'cf4e45';'974fb8';'8daa40'});
col = hex2rgb({'ca4761';'8c50c0';'a09f3a'});

figure('pos',[128 320 408*18/12 300]) 
set(gca,'fontsize',fs) 

[h1,h2,h3] = itslive_tsplot(t,v,v_err,'inliers','thresh',75); % ignore outliers and high-noise measurements, for clarity

box off
axis tight
ylabel('Ice speed (m yr^{-1})')

ylim(844 + [-1 1]*150)
h1.MarkerSize = 0.1; 
h1.Color = 0.3*[1 1 1];
set(h2(:),'LineWidth',0.1,'color',0.8*[1 1 1]); 
set(h3(:),'LineWidth',0.1,'color',0.8*[1 1 1]); 
set(gca,'fontsize',fs) 

%export_fig itslive_seasonal_methods_0.png -r600 -painters

%%

[A,ph,var_resid,sigma,v_int,p,S,mu,t_yearly,v_yearly] = itslive_seasonal_deets(t,v,v_err); 

vp = polyval(p,ti,S,mu); 

hold on 
pl(1) = plot(tid,vp,'linewidth',0.75,'color',col(3,:)); 

%%

vi_int = interp1(t_yearly,v_yearly,ti,'pchip'); 

pl(2) = plot(tid,vi_int+vp,'linewidth',0.75,'color',col(2,:));

%export_fig itslive_seasonal_methods_2.png -r600 -painters
%% 

vi_s = sineval([A ph],ti); 

pl(3) = plot(tid,vi_int+vp+vi_s,'linewidth',0.75,'color',col(1,:));

set(gca,'xtick',datetime(2006:2020,1,1))
% export_fig itslive_byrd_timeseries.png -r600 -painters

%export_fig itslive_seasonal_methods_3.png -r600 -painters


%% Add a manually generated legend:

pos = get(gca,'position'); 

ax = axes('position',[pos(1) pos(2)+pos(4) pos(3) 1-pos(2)-pos(4)]);

hold on
h10 = plot(0.02,0.45,'+','markersize',5,'linewidth',0.5,'color',h2(1).Color); 
h11 = plot(0.02,0.45,'.','markersize',0.1,'linewidth',0.1,'color',h1(1).Color); 


if false 

   txt(1) = text(0.02,0.5,'  measurement','color',h1.Color,'fontsize',fs,'vert','middle');

   h12 = plot([0.2 0.215],0.5*[1 1],'-','markersize',5,'linewidth',0.75,'color',pl(1).Color); 
   txt(2) = text(0.215,0.5,' polynomial','color',pl(1).Color,'fontsize',fs,'vert','middle');

   h13 = plot([0.35 0.365],0.5*[1 1],'-','markersize',5,'linewidth',0.75,'color',pl(2).Color); 
   %txt(3) = text(0.365,0.5,{' polynomial +';' moving avg.'},'color',pl(2).Color,'fontsize',fs,'vert','middle');
   txt(3) = text(0.365,0.5,' polynomial + moving avg.','color',pl(2).Color,'fontsize',fs,'vert','middle');

   h14 = plot([0.65 0.665],0.5*[1 1],'-','markersize',5,'linewidth',0.75,'color',pl(3).Color); 
   %txt(4) = text(0.515,0.5,{' polynomial +';' moving avg. +';' seasonal'},'color',pl(3).Color,'fontsize',fs,'vert','middle');
   txt(4) = text(0.665,0.5,' polynomial + moving avg. + seasonal','color',pl(3).Color,'fontsize',fs,'vert','middle');

else
   txt(1) = text(0.02,0.55,' \color[rgb]{0.3,0.3,0.3} measurements \color[rgb]{0.63,0.63,0.23} 4^{th} order polynomial fit \color[rgb]{0.55,0.31,0.75} with interannual variability \color[rgb]{0.79,0.28,0.38} and a seasonal sinusoid fit.',...
   'fontsize',fs,'vert','middle');
end
axis([0 1 0 1])
axis off

% move it down to the bottom: 
ax.Position(2) = pos(2);

% export_fig itslive_byrd_timeseries.png -r600 -painters
return


%% * * * * * * * SUBFUNCTIONS * * * * * * * * * * * * * * * 
% These functions are altered versions of itslive_interannual and itslive_seasonal
% just to show the individual steps. 

function [vi,v_std,t_yearly,v_yearly] = itslive_interannual_deets(t,v,v_err,varargin)
% itslive_interannual computes the interannual component of velocity variability
% for an itslive time series. 
% 
%% Syntax 
% 
%  vi = itslive_interannual(t,v,err)
%  vi = itslive_interannual(...,'ti',ti)
%  [vi,v_std,t_yearly,v_yearly] = itslive_interannual(...)
% 
%% Description 
% 
% vi = itslive_interannual(t,v,v_err) gives the interannual component of variability
% in v, by assigning the weighted average of velocities for each calendar year to a 
% June 21st posting, then interpolating via pchip to the input times of t. Input times
% t can be Mx1 to represent the mean times of each input pair, or Mx2 for the times
% of each image a la [t1 t2]. Times t must be datenum or datetime format. Weighting
% follows w=1./v_err.^2; 
% 
% vi = itslive_interannual(...,'ti',ti) uses times ti as the posting dates to 
% interpolate to. If dates aren't specified ti = mean times of t. 
% 
% [vi,v_std] = itslive_interannual(...) also provides the unweighted standard
% deviation of interannual variability. 
%
%% Example 
% 
% load byrd_test_data 
% itslive_tsplot(t,v,v_err,'datenum','thresh',50); 
% 
% tm = mean(t,2); % mean times
% vm = itslive_interannual(t,v,v_err); 
% plot(tm,vm,'ro') 
% 
%% Author Info 
% Chad A. Greene wrote this, October 2019. 
% 

%% Error checks: 

narginchk(3,5)
assert(size(t,2)==2,'Error: times t must be Nx2 to specify start and end times for acquisition pairs.')
assert(isequal(size(t,1),size(v,1),size(v_err,1)),'Error: Dimensions of v must correspond to the times t.') 

%% Parse inputs:

user_ti = false;  % by default, use mean image times as posting dates to solve for. 

if nargin>2
   
   tmp = strcmpi(varargin,'ti'); 
   if any(tmp)
      user_ti = true; 
      ti = datenum(varargin{find(tmp)+1}); 
   end
   
end

%% Do mathematics: 

w = 1./v_err.^2; % weights 

% Get center dates: 
tm = mean(datenum(double(t)),2); % This allows input t to be Nx1 or Nx2.
if ~user_ti
   ti = tm; 
end

% Delete NaN data: 
isf = isfinite(v) & isfinite(v_err); 
tm = tm(isf); 
w = w(isf); 
v = v(isf); 

[start_year,~,~] = datevec(min(tm)-182.62); % 182.62 is 365.25/2
[end_year,~,~] = datevec(max(tm)+182.62); 

% Annual postings on June 21 (solstice) for the whole timespan: 
t_yearly = datenum(start_year:end_year,6,21); 
v_yearly = NaN(size(t_yearly)); 

for k = 1:length(t_yearly)
   ind = tm>=(t_yearly(k)-182.62) & tm<=(t_yearly(k)+182.62); 
   if any(ind)
      v_yearly(k) = sum(w(ind).*v(ind))./sum(w(ind)); % weighted mean
      t_yearly(k) = sum(w(ind).*tm(ind))./sum(w(ind)); % rewrites date as the weighted mean date
   end
end

isf = isfinite(v_yearly); 

try
   vi = interp1(t_yearly(isf),v_yearly(isf),ti,'pchip'); % extrapolates by default for pchip
catch 
   vi = NaN(size(ti)); 
end

if nargout>1
   v_std = std(v_yearly(isf)); 
end

end



function [A,ph,var_resid,sigma,v_int,p,S,mu,t_yearly,v_yearly] = itslive_seasonal_deets(t,v,v_err,varargin)
% itslive_seasonal combines itslive_interannual and itslive_sinefit to assess seasonality.
% 
%% Syntax 
% 
%  [A,ph] = itslive_seasonal(t,v,v_err)
%  [A,ph] = itslive_seasonal(...,'poly',polyOrder)
%  [A,ph] = itslive_seasonal(...,'iter',numIterations)
%  [A,ph] = itslive_seasonal(...,'outlier',outlierThresh)
%  [A,ph,var_resid,sigma,v_int,p,S,mu,t_yearly,v_yearly] = itslive_seasonal(...)
% 
%% Description 
%
% [A,ph] = itslive_seasonal(t,v,v_err) detrends the velocity time series v, 
% removes interannual variability with itslive_interannual, and iteratively 
% solves for seasonal sinusoids via itslive_sinefit. 
% 
% [A,ph] = itslive_seasonal(...,'poly',polyOrder) specifies the order of the 
% polynomial to remove from the time series before interannual seasonal variability
% is calculated. Default is the ceiling of the number of years of data in the
% time series divided by four (e.g., linear detrending for less than 4 years of 
% data, quadratic detrending for four to eight years of data, etc). 
% 
% [A,ph] = itslive_seasonal(...,'iter',numIterations) specifies the number 
% of iterations of removing seasonal variability fitting sinusoids before 
% settling on a final answer. Default is 10 iterations. 
% 
% [A,ph] = itslive_seasonal(...,'outlier',outlierThresh) specifies a scalar 
% value which is the number of sigmas beyond the mean to discard. Default 
% outlierThresh is 2, meaning any velocites beyond +/-2 sigmas away from 
% the interannual mean are discarded. That discards about 5% of data. Use Inf
% to discard no data. 
% 
% [A,ph,var_resid,sigma,v_int] = itslive_seasonal(...) also returns the weighted variance
% of residuals of velocities, and formal velocity error sigma of all the measurements
% that were included. Optional v_int is the interannual
% component *after* removing seasonal variability from the internannual component. 
% 
%% Example 
% 
% load byrd_test_data.mat 
% 
% [A,ph] = itslive_seasonal(t,v,v_err); 
% 
% % or something like: 
% 
% [A,ph] = itslive_seasonal(t,v,v_err,'poly',2,'iter',7); 
% 
% % then plot with interesting context: 
% ti = min(t(:)):max(t(:)); 
% vi = itslive_tsinterp(t,v,v_err,ti); 
% 
% figure
% itslive_tsplot(t,v,v_err,'datenum','inliers','thresh',50) 
% hold on
% plot(ti,vi,'r','linewidth',2)
%
%% Author Info 
% Chad A Greene wrote this in December 2019 instead of attending an afternoon
% session at AGU that he really probably should be at. 
%
% See also: itslive_sinefit and itslive_interannual. 

%% Error checks: 

narginchk(3,inf)
assert(size(t,2)==2,'Error: times t must be Nx2 to specify start and end times for acquisition pairs.')
assert(isequal(size(t,1),size(v,1),size(v_err,1)),'Error: Dimensions of v must correspond to the times t.') 

%% Some initial bookkeeping and setting defaults

t = double(t); 
tm = mean(t,2); % mean datenum
dyr = diff(t,1,2)/365.25; % image pair dt in years
polyOrder = ceil((max(tm)-min(tm))/(4*365.25)); % order of polynomial for detrending 
iter = 10; % number of iterations of interannual cycle removal 
outlierThresh = 2; % number of standard deviations away from mean to discard. A value of 2 keeps +/-2 std, or about 95% of data.  

%% Parse inputs: 

if nargin>3
   tmp = strncmpi(varargin,'polynomial',4); 
   if any(tmp)
      polyOrder = varargin{find(tmp)+1}; 
      assert(isscalar(polyOrder) & mod(polyOrder,1)==0,'Error: polynomial order must be a scalar integer value.')
   end
   
   tmp = strncmpi(varargin,'iterations',4); 
   if any(tmp)
      iter = varargin{find(tmp)+1}; 
      assert(isscalar(iter) & mod(iter,1)==0,'Error: Number of iterations must be a scalar integer value.')
   end
   
   tmp = strncmpi(varargin,'outlier',3); 
   if any(tmp)
      outlierThresh = varargin{find(tmp)+1}; 
      assert(isscalar(outlierThresh),'Error: outlier threshold must be a scalar value.')
   end
end

%% Do mathematics 

% Weighted detrend: 
w = 1./v_err.^2; 
[p,S,mu] = polyfitw(tm,v,polyOrder,w); 
v = v - polyval(p,tm,S,mu); 

% First attempt to remove interannual variability: 
v_interannual = itslive_interannual(t,v,v_err); 
v_resid = v - v_interannual; 
   
% Separate the inliers from the outliers: 
good = abs(v_resid)<=outlierThresh*std(v_resid); 

% Set outputs to NaN in case everything fails: 
A = NaN; 
ph = NaN; 
var_resid = NaN; 

if sum(good)>10
   
   % First guess: 
   [A,ph] = itslive_sinefit(t(good,:),v_resid(good),v_err(good)); 
   
   for k = 2:iter % this works even if iter=1
         
         % How much of what each measurment measured is the seasonal component? 
         % Take the sineval of *displacement* (phase adjusted by 91 days and amplitude divided by 2pi) divided by dt 
         vs_tmp = (sineval([A/(2*pi) ph+365.25/4],t(:,2)) - sineval([A/(2*pi) ph+365.25/4],t(:,1)))./dyr; 
         
         % Calculate interannual using only the nonoutlier data, but solve for all measurement times anyway:  
         [v_interannual,~,t_yearly,v_yearly] = itslive_interannual_deets(t(good,:),v(good) - vs_tmp(good),v_err(good),'ti',tm); 
         
         % These residuals should only contain seasonal plus noise: 
         v_resid = v - v_interannual; 
         
         % Redefine outliers:
         good = abs(v_resid)<=outlierThresh*std(v_resid); 
         
         if sum(good)>10
            [A,ph] = itslive_sinefit(t(good,:),v_resid(good),v_err(good));
         else
            A = NaN; 
            ph = NaN; 
         end
         
   end
      
   if nargout>2
      var_resid = var(v_resid(good)-sineval([A ph],tm(good)),w(good)); 
      
      if nargout>3
         
         sigma = sqrt(1./sum(w(good))); 
         
         if nargout>4
            v_int = polyval(p,tm,S,mu) + v_interannual; 
         end
      end
   end
end

end



