function [A_sigma,ph_sigma] = itslive_seasonal_unc(N_pairs,int_rms,A) 
% itslive_seasonal_unc estimates the uncertainty in seasonal amplitude
% and phase for an ITS_LIVE velocity dataset. 
% 
%% Syntax 
% 
% [A_sigma,ph_sigma] = itslive_seasonal_unc(N_pairs,int_rms,A)
%
%% Description
% 
% [A_sigma,ph_sigma] = itslive_seasonal_unc(N_pairs,int_rms) gives the amplitude
% uncertainty A_sigma (m/yr) and phase uncertainty ph_sigma (days) for N_pairs
% number of image pairs with an interannual rms of int_rms. If you can, it's better
% to also enter A as an input, because for small signals, phase uncertainty
% is a strong function of A. 
% 
% [A_sigma,ph_sigma] = itslive_seasonal_unc(N_pairs,int_rms,A) gives the amplitude
% uncertainty A_sigma (m/yr) and phase uncertainty ph_sigma (days) for N_pairs
% number of image pairs with an interannual rms of int_rms and a seasonal
% amplitude A. These uncertainty estimates are from compute_seasonal_uncertainty.m. 
% 
%% Author Info
% Chad A. Greene, April 2020. 

%% Input wrangling: 

int_rms = double(int_rms); 
A = double(A); 
N_pairs = double(N_pairs); 
if isscalar(N_pairs)
   N_pairs = repmat(N_pairs,size(int_rms)); 
end
if isscalar(int_rms) 
   int_rms = repmat(int_rms,size(N_pairs)); 
end

if nargin>2 & isscalar(A) 
   A = repmat(A,size(N_pairs)); 
end

%% Load data: 

D = load('itslive_seasonal_unc.mat'); 


%% Interpolate: 

InterpMethod = 'makima'; 

if nargin==2
    A_sigma = 1.4826*interp2(D.N_pairs,D.interannual_rms,mean(D.A_mad,3)',N_pairs,int_rms,InterpMethod,NaN); 
   ph_sigma = 1.4826*interp2(D.N_pairs,D.interannual_rms,mean(D.ph_mad,3)',N_pairs,int_rms,InterpMethod,NaN); 
else
     A_sigma = 1.4826*interpn(D.N_pairs,D.interannual_rms,D.A,D.A_mad,N_pairs,int_rms,A,InterpMethod,NaN); 
    ph_sigma = 1.4826*interpn(D.N_pairs,D.interannual_rms,D.A,D.ph_mad,N_pairs,int_rms,A,InterpMethod,NaN);   
end

end