
% This script creates a bunch of random synthetic velocity time series
% and samples them with synthetic measurements to determine how accurate/precise
% our method of calculating seasonal variability is. 
% If you test a bunch of parameters with a big number of tests per unique
% combination of independent variables, it might take a while (couple hours or overnight).
% Chad A. Greene, April 2020. 

%% Load data

load image_pair_times.mat
spr = mod(diff(t,1,2),16)==0 & t(:,1)>datenum('april 15, 1999'); % same path row, since launch of Landsat 7
t = t(spr,:); 
v_err = vx_err(spr); 
tm = mean(t,2); % mean times
dyr = diff(t,1,2)/365.25; 

load itslive_counts.mat
load itslive_interannual_std.mat

clear spr vx_err

%%

N_tests_per = 150; % number of random tests per each unique set of indepentent variables. 

N_pairs = round(10.^(linspace(1.5,4,25))); 
interannual_rms = [0 2.1.^(-1:0.5:11)];
A = [0 10.^(linspace(-1,3,30))]; 

ph = 365*rand(N_tests_per,1);

A_mad = NaN(length(N_pairs),length(interannual_rms),length(A)); 
ph_mad = NaN(length(N_pairs),length(interannual_rms),length(A)); 

% Make a continuous daily array of known velocities: 
ti = ((min(t(:,1))-1):(max(t(:,2))+1))'; 

% Filtered random time series (plus an extra years worth on each end) for interannual variability: 
rf = filt1('lp',rand(length(ti)+365*3,N_tests_per)-0.5,'fs',365.25,'Tc',1.5,'order',1); 

% Scale the filtered random numbers to the desired rms (exclude first and last yr bc of filter edge effects) : 
rfA = rms(rf(548:end-548,:)); % amplitude of filtered random signal

vi_interannual_1 = rf(548:end-548,:)./rfA; 

%% Loop through all permutations
% This might take overnight, depending on how you set up the parameters. 
         
for kN = 1:length(N_pairs)
   for kint = 1:length(interannual_rms)
      for kA = 1:length(A)
         
         % Preallocate measured amplitude and phase: 
         Am = NaN(N_tests_per,1); 
         phm = Am; 

         for kT = 1:N_tests_per

            % daily time series of "actual" velocity: 
            vi = vi_interannual_1(:,kT).*interannual_rms(kint) + sineval([A(kA) ph(kT)],ti) + 1000; % (m/yr)

            % daily displacement time series: 
            di = cumtrapz(ti/365.25,vi); % (m) 

            % Randomized measurement: 
            indk = randperm(length(v_err),N_pairs(kN));
            t_tmp = t(indk,:); 
            v_err_tmp = v_err(indk); 
            dyr_tmp = dyr(indk); 

            d_noise = v_err(indk).*dyr(indk).*randn(N_pairs(kN),1); 
            d_measured = diff(interp1(ti,di,t_tmp),1,2) + d_noise; 
            v_measured = d_measured./dyr(indk); 

            [Am(kT),phm(kT)] = itslive_seasonal(t_tmp,v_measured,v_err_tmp); 

         end
         
         % Log the results: 
         A_mad(kN,kint,kA) = median(abs(Am-A(kA)),'omitnan'); 
         ph_mad(kN,kint,kA) = median(abs(phase_diff(ph,phm)),'omitnan'); % sigma is 1.4826*mad

      end
   end
   kN % just a counter to track progress. 
end

readme = 'saved by compute_seasonal_uncertainty.m'; 
%save('itslive_seasonal_unc.mat','A_mad','ph_mad','A','N_pairs','interannual_rms','N_tests_per','readme')
