function fout = scales_jokullhlaups_2018(type)
%computes scales and dimensionless parameters for jokullhlaup stability
%analysis, based on values given in Schoof et al 2014. c_2 = 3.4426e-14
%here; this is the value actually used in Schoof et al, the value c_2 =
%7.11e-24 given in the text. Works as v2 but computes he scale for q in
%such a way as to ensure that the dimensionless opening velocity v_0_star
%is unity

%basic physical constants
fout.rho_i = 910;
fout.rho_w = 1000;
fout.g = 9.8;
fout.L_lat = 3.35e5;
fout.A = 6e-24;
fout.n = 3;

switch type
  case 'large'
    %length and width of system
    fout.L = 50000;
    fout.W = 8000;    
    %slope
    fout.theta_surf = 1000/50000;
    %base slope, default is same as surface
    fout.theta_bed =  0;
    %cavity opening rate
    fout.v_0 = 100/(365*24*3600)*.033;
    %storage capacity
    fout.phi = .01;
    fout.W_store = 10;
    fout.v_p = fout.phi*fout.W_store/(fout.rho_w*fout.g);
    %Lake storage capacity
    fout.V_p = 2000*2000/(fout.rho_w*fout.g);%1000*500/(fout.rho_w*fout.g);
    %water input
    fout.Q_in = .05/24/3600*20e6;
  case 'valley'
    %length and width of system
    fout.L = 5000;
    fout.W = 1000;
    %slope
    fout.theta_surf = 850/5000;
    %base slope, default is same as surface
    fout.theta_bed =  650/5000;
    %cavity opening rate
    fout.v_0 = 30/(365*24*3600)*.033;
    %storage capacity
    fout.phi = .01;
    fout.W_store = 10;
    fout.v_p = fout.phi*fout.W_store/(fout.rho_w*fout.g);
    %Lake storage capacity
    fout.V_p = 10*5/(fout.rho_w*fout.g);
    %water input
    fout.Q_in = .05/24/3600*1000*5000;
end

%computational parameters
fout.n_grid = 500;

%channel aspect ratio
fout.channelratio = 1/16;

%constants in model
fout.c_1 = 1.3455e-9;
fout.c_2 =  3.4426e-24;
fout.c_3 = 4.05e-2;
fout.alpha = 5/4;

%Hydraulic gradient estimate
fout.Psi = fout.rho_i*fout.g*fout.theta_surf + (fout.rho_w-fout.rho_i)*fout.g*fout.theta_bed;

%water input, either as direct estimate or using melt rate and length,
%width
fout.q = fout.v_0/(fout.c_1*fout.Psi);

%Other scales
fout.S = (fout.q/(fout.c_3*fout.Psi^(1/2)))^(1/fout.alpha);
fout.N = (fout.c_1*fout.q*fout.Psi/(fout.c_2*fout.S))^(1/fout.n);
fout.x = fout.N/(fout.Psi);
fout.t = 1/(fout.c_2*fout.N^fout.n);

%dimensionless parameters
fout.v_0_star = fout.v_0/(fout.c_1*fout.q*fout.Psi);
fout.v_p_star = fout.v_p*fout.N*fout.x/(fout.t*fout.q);
fout.V_p_star = fout.V_p*fout.N/(fout.t*fout.q);
fout.v_S = fout.c_1*fout.Psi*fout.x;
fout.L_star = fout.L/fout.x;
fout.Q_in_star = fout.Q_in/fout.q;

%alternative scaling
fout.alt.S = (fout.c_1^(fout.n+1)*fout.c_3*fout.Psi^(3/2+fout.n)*fout.V_p^fout.n/fout.c_2).^(1/(1+fout.n-fout.alpha));
fout.alt.N = fout.alt.S/(fout.c_1*fout.V_p*fout.Psi);
fout.alt.N2 = fout.c_1^(fout.alpha/(fout.n+1-fout.alpha))*fout.c_2^(-1/(fout.n+1-fout.alpha))*fout.c_3^(1/(fout.n+1-fout.alpha))*fout.V_p^((fout.alpha-1)/(fout.n+1-fout.alpha))*fout.Psi^((1+2*fout.alpha)/2/(fout.n+1-fout.alpha));
fout.alt.q = fout.c_3*fout.alt.S^fout.alpha*fout.Psi^(1/2);
fout.alt.t = fout.V_p*fout.alt.N/fout.alt.q;
fout.alt.epsilon = fout.Q_in/fout.alt.q;
fout.alt.delta = fout.v_0*fout.alt.t/fout.alt.S;
fout.alt.nu = fout.alt.N/(fout.Psi*fout.L);
fout.alt.delta2 = fout.v_0*fout.c_1^(-(fout.n+1+fout.n*fout.alpha)/(fout.n+1-fout.alpha))*fout.c_2^(fout.alpha/(fout.n+1-fout.alpha))*fout.c_3^(-(fout.n+1)/(fout.n+1-fout.alpha))*...
    fout.Psi^(-(3+3*fout.n+2*fout.n*fout.alpha)/(2*(fout.n+1-fout.alpha)))*fout.V_p^(-fout.alpha*fout.n/(fout.n+1-fout.alpha));
fout.alt.epsilon2 = fout.Q_in*fout.c_1^(-(fout.n+1)*fout.alpha/(fout.n+1-fout.alpha))*fout.c_2^(fout.alpha/(fout.n+1-fout.alpha))*fout.c_3^(-(fout.n+1)/(fout.n+1-fout.alpha))*...
    fout.Psi^(-(1+fout.n+2*fout.alpha+2*fout.n*fout.alpha)/(2*(fout.n+1-fout.alpha)))*fout.V_p^(-fout.alpha*fout.n/(fout.n+1-fout.alpha));

end