function fout = parallel_periodic_arclength_nullcline(v_0,lmin,ds,scalefactor,scale,direction,npts,type,i_crossing,parameters,srchparams)
%fout = parallel_periodic_arclength_nullcline(v_0,lmin,ds,scalefactor,scale,direction,npts,type,i_crossing,parameters,srchparams)
%Arc length continuation computation of closed orbits using
%parallel_shoot_arclength and parallel_shoot_v3.
%Input variables (all optional) are:
%       v_0:    6-by-one vector consisting of initial guess for initial
%               point [S_R; S_K; N] on a closed  orbit, concatenated with
%               an initial guess for fixed point of dynamical system
%               [S_R_0; S_K_0; N_0]
%       lmin:   first parameter value to be used
%       ds:     Arc length size
%       scalefactor: (diagonal) metric for computation of arc length, takes
%               the form of a 7-by-1 vector [a_1 a_2 a_3 a_4 a_5 a_6 a_7] so that arc
%               length is computed as \sum_i a_i dv_i^2 if scale is
%               'linear', and as  \sum_i a_i d(log(v_i))^2 if scale is
%               'log'
%       scale:  step size computed from change in parameter v_0(4)
%                ('linear', default) or from change in log(parameter)
%                ('log')
%       direction:  initial direction for changes in l, choices +1 and -1
%       npts:   Number of points at which solution should be computed
%       type:   parameter type to be used, choices are  Choices are 'uh', 'Q_tot',
%               'L', 'n_c', 'V_p', defaults to 'Q_tot'
%       i_crossing: sets index of variable that will return to its original
%               value to compute Poincare map. Use 1 for S_R, 2 for S_K, 3
%               for N defaults to 3. Can also be passed in as part of
%               parameters structure.
%       parameters: parameter structure used by parallel_shoot_v5
%       srchparams: search parameters to be passed to Newton solver
%Output variables
%       fout:   output structure containing fields
%           v:  3-by-npts array of initial points  [S_R; S_K; N] lying on closed orbits
%           v_steady:   3-by-npts array of fixed points
%           l:  1-by-npts vector of corresponding parameter values
%           error:  1-by-npts vector of booleans, indicating true if
%               computation did not converge, false otherwise
%           Dv: 2-by-2-by-npts array of Jacobians of Poincare map v_in ->
%               v_f, to compute stability of closed orbit
%           eigmax: maximum absolute value of eigenvalues of Dv, should
%                determine stability of orbit
%           eig_n_c_1: When n_c == 1, the problem of cavity evolution
%                decouples completely from channel evolution; the channel
%                may be stable while the cavity solution is not. For n_c ==
%                1, this output computes the eigenvalue corresponding to a
%                mode in which dS_R/dt is not identically zero
%           t_orbit: orbital period
%           parameters: the parameters structure
%Works as parallel_periodic_arclength_v4 but uses output of parallel_shoot_par_nullcline and
%parallel_shoot_nullcline (uses nullcline to define Poincare map)

%initialize
if nargin == 0 || isempty(v_0)
    v_0 = [1 1 2 1 1 1]';
end
if nargin < 2 || isempty(lmin)
    lmin = 1;
end
if nargin < 3 || isempty(ds)
    ds = 1e-2;
end
if nargin < 4 || isempty(scalefactor)
    scalefactor = [1 1 1 1 1 1 1]';
end
if nargin < 5 || isempty(scale) || (~strcmp(scale,'linear') && ~strcmp(scale,'log'))
    scale = 'linear';
end
if nargin < 6 || isempty(direction)
    direction = +1;
end
if nargin < 7 || isempty(npts)
    npts = 100;
end
if nargin < 8 || isempty(type)
    type = 'Q_tot';
end

%physical parameters
if nargin < 10 || ~isfield(parameters,'L')
    parameters.L = 11.5713;
end
if ~isfield(parameters,'n_c')
    parameters.n_c = 1;
end
if ~isfield(parameters,'uh')
    parameters.uh = 1e-1;   
end
if ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1.95;
end
if ~isfield(parameters,'m_store')
    parameters.m_store = 0;    
end
if ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 1;
end
if ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1000;
end
if ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1000;
end
if ~isfield(parameters,'T')
    parameters.T = 1;
end
if ~isfield(parameters,'nu')
    parameters.nu = 1;
end
%set i_crossing
if nargin > 8 && ~isempty(i_crossing)
    parameters.i_crossing = i_crossing;
elseif ~isfield(parameters,'i_crossing')
    parameters.i_crossing = 3;
end
%set type
parameters.type = type;
%set number of degrees of freedom
parameters.ndegf = 3;
switch parameters.type
    case 'uh'
        parameters.uh = lmin;
    case 'Q_tot'
        parameters.Q_tot = lmin;
    case 'n_c'
        parameters.n_c = lmin;
    case 'L'
        parameters.L = lmin;
    case 'V_p'
        parameters.V_p_0 = lmin;
end 
%solver settings
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'method') || (~strcmp(parameters.solver.method,'15s') && ~strcmp(parameters.solver.method,'45') && ~strcmp(parameters.solver.method,'23t'))
    parameters.solver.method = '15s';
end
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'RelTol')
    parameters.solver.RelTol = 1e-12;
end
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'AbsTol')
    parameters.solver.AbsTol = 1e-12;
end
if ~isfield(parameters,'solver') || ~isfield(parameters.solver,'delta_v_0')
    parameters.solver.delta_v_0 =  100*eps*norm(v_0);
end
if ~isfield(parameters,'t_span')
    parameters.t_span = [0 40];
end
%set verbosity level
if nargin < 11 || ~isfield(srchparams,'verbose')
    srchparams.verbose = 0;
end

%set up higher level parameters structure to pass to 
pars.dv_norm = ds;
pars.scalefactor = scalefactor;
pars.scale = scale;
pars.parameters = parameters;

%initialize output
fout.v = zeros(parameters.ndegf,npts);
fout.v_steady = zeros(parameters.ndegf,npts);
fout.l = zeros(1,npts);
fout.error = zeros(1,npts);
fout.Dv = zeros(parameters.ndegf,parameters.ndegf,npts);
fout.eigmax = zeros(1,npts);
if parameters.n_c == 1 && ~strcmp(type,'n_c')
    fout.eig_n_c_1 = zeros(1,npts);
end
fout.t_orbit = zeros(1,npts);
fout.parameters = parameters;

%compute first point: find steady state for imposed parameter value ---
%improve this to make more general using anonymous functions
vaux = parallel_steady(v_0(parameters.ndegf+1:2*parameters.ndegf),parameters);
%amend initial guess
v_0(parameters.ndegf+1:2*parameters.ndegf) = vaux;

%now compute corresponding closed orbit --- ditto for making more general
[vtemp, errorflag, faux] = Newton_single(@parallel_shoot_nullcline,v_0(1:parameters.ndegf),parameters,srchparams);
fout.v(:,1) = vtemp;
fout.v_steady(:,1) = vaux;
fout.l(1) = lmin;
fout.error(1) = errorflag;

fout.Dv(:,:,1) = faux.DvP;
fout.eigmax(1) = max(real(eig(faux.DvP)));
if parameters.n_c == 1 && ~strcmp(type,'n_c')
    fout.eig_n_c_1 = faux.DvP(1,1);   %Relevant for n_c = 1
end
fout.t_orbit(1) = faux.t_orbit;

%amend initial guess again
v_0(1:parameters.ndegf) = vtemp;

%loop
vtemp = [v_0; lmin];   %change variable vector to include parameter guess

dv = zeros(7,1);
switch scale
    case 'linear'
        dv(7) = ds*direction/scalefactor(7)^(1/2);
    case 'log'
        dv(1:6) = 1;
        dv(7) = exp(direction*ds/scalefactor(7)^(1/2));
end
for ii=2:npts
    ii
    vprev = vtemp;
    switch scale
        case 'linear'
            vtemp = vprev+dv;
        case 'log'
            vtemp = vprev.*dv;
    end
    pars.vprev = vprev;
    %can probably make this more general by putting anonymous funciton
    %handle in pars structure and having parallel_shoot_arclength_v4 the
    %same code (not problem-specific)
    [vtemp, errorflag, faux] =  Newton_single(@parallel_shoot_arclength_nullcline,vtemp,pars,srchparams);
    fout.v(:,ii) = vtemp(1:3);
    fout.v_steady(:,ii) = vtemp(4:6);
    fout.l(ii) = vtemp(7);
    fout.error(ii) = errorflag;
    fout.Dv(:,:,ii) = faux.DvP;
    fout.eigmax(ii) = max(abs(eig(faux.DvP)));
    if parameters.n_c == 1 && ~strcmp(type,'n_c')
        fout.eig_n_c_1 = max(abs(eig(faux.DvP([1;3],[1;3]))));
    end
    fout.t_orbit(ii) = faux.t_orbit;
    switch scale
        case 'linear'
            dv = vtemp-vprev;
        case 'log'
            dv = vtemp./vprev;
    end
end

end
