function fout = parallel_bifurcation_plane_v2(v_in,type1,type2,lmin1,lmax1,lmin2,lmax2,n_type1,n_type2,scale1,scale2,parameters)
%fout = parallel_bifurcation_v2(v_in,type,lmin,lmax,npts,parameters)
%Works as parallel_bifurcation but varies parameters of type1 and type2 over a rectangular
%grid.
%Computes solution branches for steady state effective pressure N in a simple
%conduits-in-parallel drainage system as a function of either total water
%input to the system, or of sliding velocity.
%The problem to be solved takes the form
% c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N = 0
% c_1*Q_K*T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N = 0
% Q_R + (n_c-1)*Q_K = Q_tot
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(beta-2)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% where S_R, S_K and N are the variables solved for, and all other
% quantities are specified as parameters. parallel_bifurcation allows
% either uh or Q_tot to be varied, and solves for S_R, S_K and N using a
% Newton iteration, starting with the initial guess v_in (see below) for
% first point, or solution at previous point for successive solutions along
% solution branch. Also outputs corresponding dispersion relations for
% linearized dynamical system.
%
%Input variables:
%   v_in:       optional initial guess for [S_R; S_K; N], defaults to 
%               guess produced by parallel_input_guess with input option 'channels' if no input variables are specified or v_in is
%               empty
%   type1:      type for parameter 1, choices are 'uh', 'Q_tot', 'L', 'n_c'
%               and 'V_p'
%   type2:      type for parameter 2, same choices as for type1. type2 must
%               not be the same as type1.
%   lmin1:      minimum value of parameter 1
%   lmax1:      maximum value of parameter 1
%   lmin2:      minimum value of parameter 2
%   lmax2:      maximum value of parameter 2
%   n_type1:       number of points for parameter 1
%   n_type2:        number of points for parameter 2
%   scale1:     space points for parameter 1 logarithmically ('log') or linearly
%               ('linear')
%   parameters: parameter strucutre, with fields
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%       V_p_0:  Scale factor for storage capacity
%       uh_bar: Scale factor for uh in calculation of storage capacity
%       m_store:Exponent for dependence of storage capacity on sliding
%               velocity
%
%Output variables:
%   fout:       output structure with fields
%       S_R:    n_pts-by-1 vector with solution S_R at each point on
%               branch
%       S_K:    n_pts-by-1 vector with solution S_K at each point on
%               branch
%       N:      n_pts-by-1 vector with solution N at each point on
%               branch
%       eig_0:    Eigenvalue of linearized differential equation system with
%               no storage capacity, currently commented-out
%       eig_Vp: Eigenvalue of linearized differential equation system with
%               non-zero storage capacity, currently commented-out
%       error:  flag indicating if an error occured in the computation of
%               (S_R,S_K,N)
%       uh:     n_pts-by-1 vector with input parameter uh at each point on
%               branch, output only for type 'uh'
%       type:   input string type
%       Q_tot:  n_pts-by-1 vector with input parameter uQ_tot at each point
%               on branch, output only for type 'Q_tot'
%       parameters: parameters structure used in computations (will contain
%               last uh value used for type 'uh', and last Q_tot value used
%               for type 'Q_tot'); default settings will appear for field
%               not set as part of the input parameters structure
%
%eig and eig_Vp correspond to the linearization around its computed steady
%state of the dynamical system
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K*T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N
% V_p*dN/dt = Q_R + (n_c-1)*Q_K - Q_tot
%where
%V_p = V_p_0*(uh/uh_bar)^m_store


%check input and set defaults as needed
%see below parameters for intial guess

%check input and set defaults as needed; for inital guess v_in, see further below
%parameter changes
if nargin < 2 || isempty(type1) || (~strcmp(type1,'uh') && ~strcmp(type1,'Q_tot') && ~strcmp(type1,'n_c') && ~strcmp(type1,'L') && ~strcmp(type1,'V_p'))
    type1 = 'uh';
end
if nargin < 3 || isempty(type2) || (~strcmp(type2,'uh') && ~strcmp(type2,'Q_tot') && ~strcmp(type2,'n_c') && ~strcmp(type2,'L') && ~strcmp(type2,'V_p'))
    type2 = 'Q_tot';
end
if strcmp(type1,type2), error('two different parameter types must be specified'), end
fout.type1 = type1;
fout.type2 = type2;

%parameter range
if nargin < 4 || isempty(lmin1)
    lmin1 = 1e-2;
end
if nargin < 5 || isempty(lmax1)
    lmax1 = 1e0;
end
if nargin < 6 || isempty(lmin2)
    lmin2 = 1e-2;
end
if nargin < 7 || isempty(lmax2)
    lmax2 = 1e0;
end
%number of parameter steps
if nargin < 8 || isempty(n_type1)
    n_type1 = 100;
end
if nargin < 9 || isempty(n_type2)
    n_type2 = 100;
end
%scales for parameters
if nargin < 10 || (~strcmp(scale1,'linear') && ~strcmp(scale1,'log')) 
    scale1 = 'linear';
end
if nargin < 11 || (~strcmp(scale2,'linear') && ~strcmp(scale2,'log')) 
    scale2 = 'linear';
end
%physical parameters
if nargin < 12 || ~isfield(parameters,'L')
    parameters.L = 10;%1/eps;
end
if nargin < 12 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if nargin < 12 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 12 || ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if nargin < 12 || ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
end
if nargin < 8 || ~isfield(parameters,'m_store')
    parameters.m_store = 0;
end
if nargin < 12 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 12 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 12 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 12 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 12 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 12 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 12 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 12 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 12 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 12 || ~isfield(parameters,'T')
    parameters.T = 1;
end
if nargin < 12 || ~isfield(parameters,'nu')
    parameters.nu = 1;
end

%parameter step size
switch scale1
    case 'linear'
        fout.l1 = linspace(lmin1,lmax1,n_type1);
    case 'log'
        fout.l1 = logspace(log10(lmin1),log10(lmax1),n_type1);
end
switch scale2
    case 'linear'
        fout.l2 = linspace(lmin2,lmax2,n_type2);
    case 'log'
        fout.l2 = logspace(log10(lmin2),log10(lmax2),n_type2);
end

%initial guess
%v_in = [1 1 1]';
if nargin == 0 || isempty(v_in)
    switch type1
        case 'uh'
            parameters.uh = fout.l1(1);
        case 'Q_tot'
            parameters.Q_tot = fout.l1(1);
        case 'n_c'
            parameters.n_c = fout.l1(1);
        case 'L'
            parameters.L = fout.l1(1);
        case 'V_p'
            parameters.V_p_0 = fout.l1(1);
    end    
    switch type2
        case 'uh'
            parameters.uh = fout.l2(1);
        case 'Q_tot'
            parameters.Q_tot = fout.l2(1);
        case 'n_c'
            parameters.n_c = fout.l2(1);
        case 'L'
            parameters.L = fout.l2(1);
        case 'V_p'
            parameters.V_p_0 = fout.l2(1);
    end
    v_in = parallel_initguess(parameters,'channel');
end

fout.S_R = zeros(n_type1,n_type2);
fout.S_K = zeros(n_type1,n_type2);
fout.N = zeros(n_type1,n_type2);
fout.error = zeros(n_type1,n_type2);
fout.eig_0 = zeros(n_type1,n_type2,3);
fout.eig_Vp = zeros(n_type1,n_type2,3);

for ii=1:n_type1
    switch type1
        case 'uh'
            parameters.uh = fout.l1(ii);
        case 'Q_tot'
            parameters.Q_tot = fout.l1(ii);
        case 'n_c'
            parameters.n_c = fout.l1(ii);
        case 'L'
            parameters.L = fout.l1(ii);
        case 'V_p'
            parameters.V_p_0 = fout.l1(ii);
    end
    for jj=1:n_type2
        switch type2
            case 'uh'
                parameters.uh = fout.l2(jj);
            case 'Q_tot'
                parameters.Q_tot = fout.l2(jj);
            case 'n_c'
                parameters.n_c = fout.l2(jj);
            case 'L'
                parameters.L = fout.l2(jj);
            case 'V_p'
                parameters.V_p_0 = fout.l2(jj);
        end
        [v_in,parameters,error_flag] = parallel_steady(v_in,parameters);
        fout.S_R(ii,jj) = v_in(1);
        fout.S_K(ii,jj) = v_in(2);
        fout.N(ii,jj) = v_in(3);
        fout.error(ii,jj) = error_flag;
        fout.eig_0(ii,jj,:) = parallel_spectrum(v_in,parameters,'no_storage');
        fout.eig_Vp(ii,jj,:) = parallel_spectrum(v_in,parameters,'storage');
    end
    v_in(1) = fout.S_R(ii,1);
    v_in(2) = fout.S_K(ii,1);
    v_in(3) = fout.N(ii,1);
end

%finish output
fout.parameters = parameters;
end
