function fout = floodinitialization(parameters,N_c)
%fout = floodinitialization(parameters,N_c)
%Computes the flood initialization, corner, and initial lake drop layer
%asymptotic solutions for delta << epsilon << 1,
%epsilon*delta^(-(alpha-1)*(n+1)/(alpha*n)) ~ O(1). 
%
%The flood initialization layer satisfies
%   epsilon*delta^(-(alpha-1)*(n+1)/(alpha*n))dSinitialize/dtinitialize =
%       Sinitialize^alpha+1-Sinitialize*|Ninitialize|^(n_Glen-1)*Ninitialize
%   dNinitialize/dtinitialize = -1
%with Ninitialize ~ - tinitialize, Sinitialize ~ 1/|tinitialize|^n_Glen as
%tinitialize -> - inf. The solution experiences finite time blow-up at a
%finite Ninitialize = N_c and tinitialize = t_c = -N_c; the code solves for
%N_c and outputs the solution orbit.
%
%The corner layer satisfies
%   dScorner/dtcorner = Scorner^alpha
%   dNcorner/dtcorner =
%   epsilon^(1/alpha)*delta^(-(alpha-1)/(alpha*n))*(-1+Scorner^alpha)
%where epsilon^(1/alpha)*delta^(-(alpha-1)/(alpha*n)) << 1, and Scorner ~
%-1((alpha-1)(-tcorner))^(1/(alpha-1), Ncorner = N_c - tcorner as tcorner
%-> - inf
%
%Lastly, the initial growth layer satisfies
%   dSgrowth/dtgrowth = Sgrowth^alpha
%   dNgrowth/dtgrowth = Sgrowth^alpha
%with Ngrowth -> N_c, Sgrowth ~ -1((alpha-1)(-tgrowth))^(1/(alpha-1) as
%tgrowth -> - inf
%
%The different layers are related to the original, scaled variables S,N,t
%in the system
%   dS/dt = S^alpha*|1-nu*N|^(3/2) + delta - S*|N|^(n_Glen-1)*N
%   dN/dt = epsilon - S^alpha*|1-nu*N|^(-1/2)*(1-nu*N)
%through the following successive rescalings
%   Sinitialize = delta^(-1/alpha)*S
%   Ninitialize = delta^(-(alpha-1)/(alpha*n_Glen))*N
%   tinitialize = delta^(-(alpha-1)/(alpha*n))*(epsilon*(t-t_f)-N_f)
%where N_f is the effective pressure at the end of the flood phase, and t_f
%is the time of the last flood (to leading order, it is immaterial where in
%the flood interval t_f is chosen), and
%   Scorner = delta^(1/alpha)*epsilon^(-1/alpha)*Sinitialize
%   Ncorner = Ninitialize
%   tcorner =
%   delta^((alpha-1)/(alpha*n))*epsilon^(-1/alpha)*(tinitialize-t_c)
%as well as
%   Sgrowth = delta*epsilon^(-1)*Sinitialize
%   Ngrowth = Ninitialize
%   tgrowth = (delta/epsilon)^(-(alpha-1))*(tinitialize-t_c)
%
%In solving for N_c,  the code obviously does not integrate all the way to
%the singularity; instead, it integrates backwards from a large but finite
%value of Sinit and a value of Ninit that is larger than the actual N_c. The
%initial value of S is chosen as follows. Let S^* solve
%   dS^*/dt = 1 + S^* |t|^(n_Glen-1)*t
%with S^* -> 1/|t|^n_Glen as t -> - inf, and  lef Sfin be the value of S^*
%such that S^* |t|^(n_Glen-1)*t < 1 and (S^*)^(alpha-1) > |t|^(n_Glen-1)*t,
%as a measure of when dissipation-driven melting overtakes potential
%viscous creep opening, and neither is dominated by conduit opening due to
%flow over bed roughness. The initial value of Sinit for shooting backward in
%time is set to 10^5*S_fin. The code shoots backwards in time until S =
%S_fin, and also shoots forwards from a very small value S_far = 1e-4*Sfin and
%a corresponding N_far = S_far^(1/n_Glen); again, shooting is done until S
%= S_fin, and the code finds the appropriate Ninit through a Newton solver
%to ensure that the two solutions, obtained by shooting forward and
%backward, have the same value of N = N_fin. Given Ninit, N_c is then
%established by assuming that, for S > Sinit, S is well-approximated by the
%asymptotic formula
%   S ~
%   epsilon^(1/(alpha-1))*delta^(-(n+1)/(alpha*n))/((alpha-1)*(t_c-t))^(1/alpha-1)
%which allows t_c-tinit to be compute from Sinit, and N_c = Ninit -
%(t_c-tinit) to be computed
%
%Input arguments:
%   parameters: optional parameter structure with fields
%       epsilon, delta, n_Glen, alpha:  model parameters as defined above
%           Defaults to epsilon =  0.183001; delta = 0.00689435; n_Glen = 3; alpha
%           = 5/4;
%   N_c:    optional initial guess for the critical effective pressure N_c
%       (more correctly, an initial guess for Ninit as defined above)
%Output argument:
%   fout:   output structure with fields
%       N_c:    the effective pressure at finite-time blow-up
%       deltaN_c:  the correction t_c-tinit to Ninit; this is useful if
%               the output N_c is to be used as an initial guess for a
%               slightly changed set of parameters; it is then better to
%               use N_c+deltaN_c as the initial guess
%       S_initialize:    solution for Sinitialize
%       N_initialize:    corresponding solution for Ninitialize
%       t_initialize:    corresponding values of tinitialize
%       nbreak:     index into S_initialize where the forward and backward
%                   shooting solutions are spliced together
%       S_corner, N_corner, t_corner:   solution of the corner layer (in
%                   terms of Scorner, Ncorner, tcorner as defined above)
%       Sgrowth, Ngrowth, tgrowth:      solution of the growth layer
%       parameters: the parameter structure used, also includes a field
%                   gamma = epsilon*delta^(-(alpha-1)*(n+1)/(alpha*n)) 
%
%Dependencies:  shoot_v1, Newton_single

%parameters if not supplied
if isempty(parameters) || nargin == 0
   parameters.epsilon =0.183001;
   parameters.delta = 0.00689435;
end
if ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end

%initial guess if non is provided
if nargin < 2 || isempty(N_c)
    N_c = 0;
end

%unpack parameters
epsilon = parameters.epsilon;
delta = parameters.delta;
n_Glen = parameters.n_Glen;
alpha = parameters.alpha;

%combination of epsilon and delta
gamma = epsilon*delta^(-(alpha-1)*(n_Glen+1)/(alpha*n_Glen));
parameters.gamma = gamma;

%cut point
breakpointest = false;
tmin = -1e2;
tmax = 1e1;
while ~breakpointest
    [taux,vaux] = ode15s(@(t,v)(gamma^(-1)*(1+v*abs(t)^(n_Glen-1)*t)),[tmin,tmax],1/abs(tmin)^n_Glen);
    Smax = vaux(end);
    if Smax^alpha < Smax*abs(tmax)^n_Glen
        tmax = 5*tmax;
    else
        breakpointest = true;
        vaux(vaux.*abs(taux).^(n_Glen-1).*taux < 1) = NaN;  %ensure that creep dominates
        parameters.S_fin = vaux(abs(vaux.^alpha-vaux.*abs(taux).^(n_Glen-1).*taux)==min(abs(vaux.^alpha-vaux.*abs(taux).^(n_Glen-1).*taux)));
        min(abs(vaux.^alpha-vaux.*abs(taux).^(n_Glen-1).*taux))
        parameters.S_fin
        if isnan(parameters.S_fin), parameters.S_fin = 1; end %to prevent an all-NaN vaux vector from setting S_fin to NaN
        parameters.S_fin = max(parameters.S_fin,1);
        parameters.S_init = 1e5*parameters.S_fin;
        parameters.S_far = 1e-4*parameters.S_fin;
        parameters.N_far = 1/parameters.S_far^(1/n_Glen);
    end
end

%verbose output
srchparams.verbose = 2;

%solve for critical N
N_c = Newton_single(@floodinitializationfunction,N_c,parameters,srchparams);

%correct using asymptotic form of solution
deltaN_c = gamma/((alpha-1)*parameters.S_init^(alpha-1));
N_c = N_c - deltaN_c;

%solve for orbit
options = odeset('RelTol',1e-12,'AbsTol',1e-12,'Events',@(t,v)eventfunction(t,v,0,parameters));
evolvefun = @(t,v)evolvefunction(t,v,0,parameters);

%forward integration
vin = [parameters.S_far; parameters.N_far];
[tout1,vout1] = ode15s(evolvefun,[0 1e6],vin,options);

%backward integration
vin = [(gamma/(alpha-1)/(deltaN_c/2))^(1/(alpha-1)); N_c+deltaN_c/2];
[tout2,vout2] =  ode15s(evolvefun,[0 -100],vin,options);

%corner layer
S_corner = logspace(-2,2,100);
t_corner = -S_corner.^(-(alpha-1))/(alpha-1);
N_corner = N_c + parameters.epsilon^(1/alpha)*parameters.delta^(-(alpha-1)/(alpha*n_Glen))*...
    (-t_corner+1./((alpha-1)*(-t_corner)).^(1/(alpha-1)));

%initial growth layer
N_growth = logspace(log(N_c),5*log(N_c),1000);
S_growth = (N_growth-N_c)/gamma;
t_growth = -gamma*S_growth.^(-(alpha-1))/(alpha-1);

%output
fout.N_c = N_c;
fout.deltaN_c = deltaN_c;
fout.nbreak = length(vout1(:,1));
fout.S_initialize = [vout1(:,1); vout2(end:-1:1,1)];
fout.N_initialize = [vout1(:,2); vout2(end:-1:1,2)];
fout.t_initialize = [tout1; tout2(end:-1:1)];
fout.S_corner = S_corner;
fout.N_corner = N_corner;
fout.t_corner = t_corner;
fout.S_growth = S_growth;
fout.N_growth = N_growth;
fout.t_growth = t_growth;
fout.parameters = parameters;
end

function [fout,Dfout] = floodinitializationfunction(N_init,parameters)
% [fout,Dfout] = floodinitializationfunction(N_init,parameters)
%   The argument N_init that sets the output fout to zero approximates the
%   effective pressure N_c at the finite time blow-up point in the flood
%   initialization model for epsilon*delta^(-(alpha-1)*(n+1)/(alpha*n)) ~
%   O(1). Specifically, the code integrates the ordianry differential
%   equation system
%       dS/dt = gamma^(-1)*(S^alpha+1-S*|N|^(n_Glen-1)*N)
%       dN/dt = -1
%   with
%       gamma = epsilon*delta^(-(alpha-1)*(n_Glen+1)/(alpha*n_Glen));
%   from two locations. One corresponds to matching conditions with the
%   refilling phase,
%       (S,N) = (S_far,1/S_far^(1/n_Glen))
%   with S_far = 1e-6*gamma^(1/(alpha-1)) << 1, and the ode system is
%   integrated forward in time until S = S_fin = gamma^(1/(alpha-1)). The
%   second locations approximates the finite time blow up,
%       (S,N) = (S_init,N_init)
%   with S_init =  1e3*gamma^(1/(alpha-1)) >> 1, and the ode system is
%   integrated backward in time until again S = S_fin. The code then
%   outputs the difference between the value of N obtained by the two
%   integrations, and the derivative of that difference with respect to
%   N_init
%
%Input arguments:
%   N_init: Guess for an initial condition that should set fout to zero at
%           a solution
%   parameters: parameter structure with fields
%           epsilon, delta, n_Glen, alpha: parameters in the ode system
%
%Dependencies: shoot_v1.m


%set up shooting method parameter structure
shootparams.model = parameters;
shootparams.t_span = [0 1e9];
shootparams.functions.evolve = @evolvefunction;
shootparams.functions.events = @eventfunction;
shootparams.solver.method = 'ode15s';

%start and end points
S_init = parameters.S_init;
S_far = parameters.S_far;
N_far = parameters.N_far;

%shoot from left
vleft = shoot_v1([S_far; N_far],shootparams);

%shoot from right
shootparams.t_span = [0 -1e9];
if isfield(parameters,'testflag') && parameters.testflag
    shootparams.flags.test = true;
else
    shootparams.flags.test = false;
end
if shootparams.flags.test
    fout = shoot_v1([S_init; N_init],shootparams);
else
    [vright,Dvright] = shoot_v1([S_init; N_init],shootparams);
    fout = vright(2)-vleft(2);
    Dfout = Dvright(2,2);
end

end

function fout = evolvefunction(t,v,n,params)

%unpack parameters
n_Glen = params.n_Glen;
alpha = params.alpha;
gamma = params.gamma;

%dependent variables
S = v(1); N = v(2);

switch n
    case 0
        fout = zeros(2,1);
        fout(1) = gamma^(-1)*(S^alpha + 1 - S*abs(N)^(n_Glen-1)*N);
        fout(2) = -1;
    case 1
        fout = zeros(2,2);
        fout(1,1) = gamma^(-1)*(alpha*S^(alpha-1)-abs(N)^(n_Glen-1)*N);
        fout(1,2) = gamma^(-1)*(-n_Glen*S*abs(N)^(n_Glen-1));
    case 2
        fout = zeros(2,2,2);
        fout(1,1,1) = gamma^(-1)*alpha*(alpha-1)*S^(alpha-2);
        fout(1,1,2) = gamma^(-1)*(-n_Glen*abs(N)^(n_Glen-1));
        fout(1,2,1) = fout(1,1,2);
        fout(1,2,2) = gamma^(-1)*(-n_Glen*(n_Glen-1)*abs(N)^(n_Glen-3)*N);
end

end

function [fout,isterminal,direction] = eventfunction(t,v,n,params)

S_fin = params.S_fin;

%dependent variables
S = v(1);

switch n
    case 0
        fout = S-S_fin;
    case 1
        fout = zeros(2,1);
        fout(1) = 1;
end

isterminal = true;
direction = 0;

end