function fout = stability_grid_v3(parameters,type,q_vals,V_vals)
%fout = stability_grid_v3(parameters,type,q_vals,V_vals)
%   Computes steady state solutions of the dynamical system defined in the
%   file header for network_stability_v5, posed on a one-dimensional
%   network, with water input q_in to the first node set to the values specified
%   in the input vector q_vals, assuming that the parameters gamma_S
%   vanishes, while n_c = 1 (see file header for network_stability_v5
%   for the definition of the dynamical system and the parameters referred
%   to here). For each entry in q_vals, stability_grid_v2 then sets the
%   storage capacity gamma_store at either the first node to the values 
%   prescribed in V_vals if input argument type is set to 'lake', or to the
%   value in V_vals times node spacing at each node and for each value
%   computes the corresponding spectrum of eigenvalues using
%   network_stability_v5, with the steady state for the prescribed input
%   value q_vals being used as the input variable v_in in
%   network_stability_v5.
%
%   If gamma_store and q_in are set to zero at all but the first node, then
%   this configuration corresponds to have a lake of storage capacity
%   gamma_store with inflow q_in at the upstream end of a one-dimensional
%   drainage system as described in Schoof and others (Oscillatory
%   subglacial drainage in the absence of surface melt, The Cryosphere,
%   2014),
%
%   Input arguments:
%       parameters: parameter structure required by network_stability_v5;
%                   parameters.gamma_store(1) and parameters.q_in(1) are
%                   overwritten by stability_grid_v2. If left empty, the
%                   parameters structure is generater by
%                   parameter_generator_2018.m with input arguments "type"
%                   set to 'large', settings.pointsource set to 'true', and
%                   settings.lake set to true if the input argument 'type'
%                   for stability_grid_v3 is set to 'lake'; see
%                   parameter_generator_2018.m for details
%       type:       string that toggles between concentrated storage in the
%                   first node ('lake') and distributed storage along the
%                   flow path (any other string input)x
%       q_vals:     nq-by-1 or 1-by-nq vector conmaining the values with
%                   which paramters.q_in(1) is overwritten
%                   sequentially, computing a steady state for each. q_vals
%                   is a vector of arbitrary length of these values
%       V_vals:     nv-by-1 or 1-by-nv vector containinf the values with
%                   which parameters.gamma_store(1) is overwritten
%                   sequentially for every value of q_vals. The stability
%                   problem defined by network_stability_v5 is solved for
%                   every combination of values in q_vals and  V_vals,
%                   using the corresponding steady state (which does not
%                   depend on the value of gamma_store and therefore does
%                   not depend on the values prescribed in V_vals) as the
%                   input variable v_in in network_stability_v5.m
%
%   Output argument:
%       fout:       nq-by-1 cell array. The ith cell is a solution
%                   structure corresponding to parameters,q_in(1)
%                   being  overwritten by q_vals(i). The structure has a
%                   single field, "spectrum", that is itself an nv-by-1
%                   cell array. The jth cell in that array is the output of
%                   network_stability_v5 with input argument vin set to the
%                   steady state for parameters.q_in(1) overwritten
%                   by q_vals(i), and with the input structure 'parameters'
%                   of network_stability_v5 given by the input structure
%                   'parameters' to the present code, with
%                   parameters.gamma_store(1) replaced by V_vals(j) and
%                   parameters.q_in(1) repalced by q_vals(i)


if isempty(parameters)
    if strcmp(type,'lake')
        settings.lake = true;
    else
        settings.lake = false;
    end
    settings.pointsource = true;
    type = 'large';
    parameters = parameter_generator_2018(type,settings);
    parameters.eigfun = false;
end

if ~strcmp(type,'lake')
    gamma_store0 = parameters.gamma_store;
end
    
for ii=1:length(q_vals)
    Q = q_vals(ii);
    faux.Q = Q;
    faux.V = V_vals;
    parameters.q_in_0(1) = Q;
    %compute steady state
    steady_state = line_steady_v2(parameters);
    faux.steady_state = steady_state;
    %extract input for newtwork_stability
    if parameters.Kamb_storage
           v_in = [steady_state.S_R; steady_state.S_K; steady_state.N; steady_state.V_K]; 
        else
           v_in = [steady_state.S_R; steady_state.S_K; steady_state.N];
    end
    for jj=1:length(V_vals)
        switch type
            case 'lake'
                parameters.gamma_store(1) = V_vals(jj);
            otherwise
                parameters.gamma_store = gamma_store0*V_vals(jj);
        end
        spectrum = network_stability_v5(v_in,parameters);
        faux.spectrum{jj} = spectrum;
        max(real(spectrum.eigval(isfinite(spectrum.eigval))))
    end
    fout{ii} = faux;
    clear faux
end

end

