function parameters = parameter_generator_2018(type,setting)
%parameters = parameter_generator(type,settings)
%creates a default parameter structure for use with stability_grid_v2,
%creating a one-dimensional network and
%ensuring that n_c = 1 and gamma_S = 0 as assumed by the steady state
%computation in line_steady_v2.m. The parameter structures generated
%correspond to the output by parallel_parameter_generator, and the scales 
%required to convert output back to dimensional variables are included in
%the output argument 'scales' produced by parallel_parameter_generator
%included in the lumped model code suite
%
%Input argument:
%   type:   String that needs to take values 'large', 'valley' or 'valley2',
%           corresponding to different domain lengths and surface slopes
%   settings:   structure with boolean fields 
%                   'pointsource': specifies whether water input q_in is to
%                       be nonzero only at the first node of the
%                       network (true) or distrbiuted equally along the
%                       length of the flow path (false)
%                   'lake': specifies whether storage capacity is to be
%                       nonzero only at first node of the network (true) or
%                       distributed equally along the length of the flow path
%                       (false)
%
%   parameters: parameter structure as required as input to
%               network_stability_v5.m or line_steady_v2

%defaults
if nargin < 2 || ~isfield(setting,'lake')
    setting.lake = true;
end
if ~isfield(setting,'pointsource')
    setting.pointsource = true;
end

%grid
switch type
  case 'large'
    L_0 = 26.9799;
  case 'valley'
    L_0 = 11.0293;
  case 'valley2'
    L_0 = 2*11.0293; 
end

n_x = 200;
dx = L_0/n_x;
[parameters.grid, parameters.L, parameters.Phi_0] = line_grid(n_x,dx);  

%other physical parameters
parameters.n_c = 1*ones(size(parameters.L));
parameters.uh = ones(size(parameters.L));
parameters.T = ones(size(parameters.L));
parameters.nu = ones(size(parameters.L));
parameters.Kamb_storage = false;
parameters.uh_K = ones(size(parameters.Phi_0));
parameters.alpha = 5/4;
parameters.beta = 3/2;
parameters.epsilon = 1e-4;
parameters.n_Glen = 3;
parameters.r = 0.9;
parameters.c_1 = 1;
parameters.c_2 = 1;
parameters.c_3 = 1;
parameters.S_0_K = 2e2;
parameters.S_0_R = 2e2;
parameters.V_0_K = 2e2;
parameters.S_P_R = zeros(size(parameters.L));
parameters.S_P_K = zeros(size(parameters.L));
parameters.epsilon_P_R = zeros(size(parameters.L));
parameters.epsilon_P_K = zeros(size(parameters.L));
parameters.k_leak = 0;
if ~setting.lake
    parameters.gamma_store = ones(size(parameters.Phi_0))*dx/L_0;     %the distributed version
else
    parameters.gamma_store = zeros(size(parameters.Phi_0));
    parameters.gamma_store(1) = 1;                                 %the lake version
end
if  ~setting.pointsource
    parameters.q_in_0 = ones(size(parameters.Phi_0))*dx/L_0;           %the distributed version, gives integrated water input of unity
else
    parameters.q_in_0 = zeros(size(parameters.Phi_0));         %the lake version
    parameters.q_in_0(1) = 1;
end
parameters.gamma_S = zeros(size(parameters.Phi_0));
parameters.N_bdy_nodes = 1e-3*ones(size(parameters.grid.bdy_nodes));
parameters.qscaled = false;

end

function [grid, L, Phi_0]  = line_grid(n_x,dx)
%Generate line of n_x nodes. Spacing between nodes is dx

%define topology and boundary nodes
grid.n_edges = n_x;
grid.n_nodes = n_x+1;
grid.up_node = (1:n_x)';
grid.down_node = (2:n_x+1)';
grid.bdy_nodes = n_x+1;

%define locations of nodes
grid.x = (0:n_x)'*dx;

%edge lengths and potentials
L = abs(grid.x(grid.up_node)-grid.x(grid.down_node));
Phi_0 = -grid.x;

end
