function fout = line_steady_gradient_v2(N,parameters)
%function fout = line_steady_gradient_v2(N,parameters)
%Computes the gradient of the constraint function in line_steady_v2
%Input variables are:
%N:             Guess for effective pressure at current node, see below
%parameters:    Parameter structure with the many of the same fields as
%               would be required to run the network.m function, plus three
%               additional fields (N_plus,node)
%   node        index of node at which N is currently being solved for
%   N_plus      Previously solved-for N at the next node downstream
%   linear      flag that indicates a one-dimensional line network.
%               line_steady will exit with an error if this is not set.
%               The code assumes a network with n_nodes nodes, and
%               n_nodes-1 edges
%   L:          List ((n_nodes-1)-by-one vector) of edge lengths
%   n_c:        List ((n_nodes-1)-by-one vector) of conduits per edge
%   Phi_0:      Reduced hydraulic potential at each node (n_nodes-by-one
%               vector), given by rho_i g s + (rho_w-rho_i) g b, s being
%               local ice surface elevation and b local bed elevation at
%               the node. rho_i and rho_w are ice and water density and g
%               is acceleration due to gravity
%   c_1:        Melting term is c_1*Q*Psi, where Q is discharge and Psi
%               hydraulic gradient along each network edge
%   c_2:        Creep closure is c_2*S*N^n_Glen, where S is conduit
%               cross-section or storage cavity size, N is effective
%               pressure
%   c_3:        Coefficient in Manning or Darcy-Weisbach law, discharge is
%               Q = c_3*S^alpha*Psi^(beta-1)
%   alpha:      Exponent relating conduit cross-section to discharge
%   beta:       Exponent relating Hydraulic gradient to discharge
%   n_Glen:     Glen's law exponent in creep closure relation
%   epsilon:    Regualrizer for the square-root dependence of discharge on
%               hydraulic gradient
%   uh:         List (n_nodes-by-one vector) of 'cavity' opening rates for
%               each edge
%   S_0_R:      cut-off size for cavity opening term for R conduits
%   S_0_K:      cut-off size for cavity opening term for pure cavity K
%               conduits
%   S_P_R:      List ((n_nodes-1)-by-one vector) of percolation cut-offs for R conduits
%   S_P_K:      List ((n_nodes-1)-by-one vector) of percolation cut-offs for K conduits
%   epsilon_P_R: List ((n_nodes-1)-by-one vector) of regularizers for percolation cut-off for R conduits
%   epsilon_P_K: List ((n_nodes-1)-by-one vector) of regularizers for percolation cut-off for K conduits
%   T:          List ((n_nodes-1)-by-one vector) of 'cavity' tortuosities
%               along each edge 
%   nu:         List ((n_nodes-1)-by-one vector) of reduction factors in
%               opening rates from 'cavity' to  'channel' conduits for each
%               edge
%   q_in:       List (n_nodes-by-one vector) of water supply rates at each
%               node
%   Kamb_storage:Flag that sets whether a Kamb-type storage cavity volume
%               needs to be solved for at each node
%   uh_K:       List (n_nodes-by-one vector) of storage cavity opening
%               rates at each node



%unpack inputs
node = parameters.node;     %Current node
N_plus = parameters.N_plus; %Effective pressure at the next node downstream

L = parameters.L(node);     %current edge lengths
n_c = parameters.n_c(node); %number of conduits along current edge
Phi_0 = parameters.Phi_0(node);         %reduced hydraulic potential rho_i s + (rho_w-rho_i) b at current node
Phi_plus = parameters.Phi_0(node+1);    %reduced hydraulic potential at next node downstream
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
epsilon = parameters.epsilon;   %regularization parameter for hydraulic gradient
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh(node);   %cavity opening rate for current network edge
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening
S_0_K  = parameters.S_0_K;  %cut-off size for Kamb cavity opening
S_P_R = parameters.S_P_R(node);   %percolation cut-off for R-conduits
S_P_K = parameters.S_P_K(node);   %percolation cut-off for K-conduits
epsilon_P_R = parameters.epsilon_P_R(node);   %regularizer for R-conduit percolation cut-off
epsilon_P_K = parameters.epsilon_P_K(node);   %regularizer for K-conduit percolation cut-off
T = parameters.T(node);     %Tortuosity for 'cavities' along current network edge
nu = parameters.nu(node);   %Step size ratiosfor 'channel' along current network edge
Q = sum(parameters.q_in(1:node));     %Flux along current edge

%compute hydraulic gradient along each network edge
Psi = (Phi_0 - N - Phi_plus + N_plus)/L;
DPsi = -1/L;
Psi_abs_R = (Psi^2+epsilon^2).^(1/2);
DPsi_abs_R = Psi*DPsi/Psi_abs_R;
Psi_abs_K = (Psi^2/T^2+epsilon^2)^(1/2);
DPsi_abs_K = Psi*DPsi/(T^2*Psi_abs_K);

%compute effective pressure along each network edge
N_edge = (N_plus+N)/2;
DN_edge = 1/2;

%Compute steady cavity size
S_K = uh/(uh/S_0_K + c_2*abs(N_edge)^(n_Glen-1)*N_edge);
DS_K = -S_K/(uh/S_0_K + c_2*abs(N_edge)^(n_Glen-1)*N_edge)*c_2*n_Glen*abs(N_edge)^(n_Glen-1)*DN_edge;

%compute 'cavity' flux along each edge
Q_K = c_3*(((S_K-S_P_K)^2+epsilon_P_K^2)^(1/2)-epsilon_P_K)^alpha*Psi_abs_K^(beta-2)*Psi/T;
DQ_K = alpha*Q_K/(((S_K-S_P_K)^2+epsilon_P_K^2)^(1/2)-epsilon_P_K)*(S_K-S_P_K)/((S_K-S_P_K)^2+epsilon_P_K^2)^(1/2)*DS_K ...
    + (beta-2)*Q_K/Psi_abs_K*DPsi_abs_K + Q_K/Psi;

%compute 'channel' flux along each edge
Q_R = Q - (n_c-1)*Q_K;
DQ_R = -(n_c-1)*DQ_K;

%Compute auxiliary variables Y_R = (c_2*abs(N_edge)^(n_Glen-1)*N_edge +
%nu*uh/S_0_R and X_R = S_R*Y_R assuming steady state S_R:X_R = c_1*Q_R*Psi + nu*uh;
X_R = c_1*Q_R*Psi + nu*uh;
Y_R = c_2*abs(N_edge)^(n_Glen-1)*N_edge +nu*uh/S_0_R;
DX_R = c_1*DQ_R*Psi + c_1*Q_R*DPsi;
DY_R = n_Glen*c_2*abs(N_edge)^(n_Glen-1);

%Output: does this X_R, Y_R correspond to the required flux Q_R?
%Target function fout = Q_R*Y_R^alpha   - c_3*(((X_R - S_P_R*Y_R)^2 + epsilon_P_R^2*Y_R^2)^(1/2)-epsilon_P_R*Y_R)^alpha*Psi_abs_R^(beta-2)*Psi;
fout = DQ_R*Y_R^alpha + alpha*Q_R*Y_R^(alpha-1)*DY_R ...
        - c_3*alpha*(((X_R - S_P_R*Y_R)^2 + epsilon_P_R^2*Y_R^2)^(1/2)-epsilon_P_R*Y_R)^(alpha-1)*Psi_abs_R^(beta-2)*Psi ...
        *(((X_R-S_P_R*Y_R)*(DX_R-S_P_R*DY_R)+2*epsilon_P_R^2*DY_R)/((X_R - S_P_R*Y_R)^2 + epsilon_P_R^2*Y_R^2)^(1/2) - epsilon_P_R*DY_R) ...
        -c_3* (((X_R - S_P_R*Y_R)^2 + epsilon_P_R^2*Y_R^2)^(1/2)-epsilon_P_R*Y_R)^alpha * ((beta-2)*Psi_abs_R^(beta-3)*Psi*DPsi_abs_R + Psi_abs_R^(beta-2)*DPsi);
