function fout = parallel_stability_boundary_v5(v_in,type1,type2,lmin,ds,scalefactor,scale,direction,npts,parameters,srchparams,flags)
%fout = parallel_stability_boundary_v5(v_in,type1,type2,lmin,ds,scalefactor,scale,direction,npts,parameters,srchparams)
%Computes the location of the Hopf bifurcation in the conduits-in-parallel
%system
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R)*(N/N_u)^(-m_slide) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K)*(N/N_u)^(-m_slide) - c_2*S_K*|N|^(n_Glen-1)*N
% -V_p_0*(uh/uh_bar)*dN/dt = Q_tot - Q_R - (n_c-1)*Q_K
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
%For suitable parameter choices, this system undergoes a Hopf bifurcation
%that marks the onset of oscillatory jokulhlaup-style dynamics. This code
%computes combinations of two parameters in the model specified through the
%inputs "type1" and "type2" that correspond to the bifurcation. The initial
%point computed corresponds to a prescribed value of the type1 parameter,
%and the solution from there is computed by arc length continuation. The
%code also outputs the corresponding steady state solution of the dynamical
%system, and the Landau coefficients with respect to changes in the
%parameter Q_tot at the bifurcation. The corresponding Landau equation
% d A / dT + b A + c |A|^2 A = 0
%describes the evolution of the amplitude A of oscillations near the
%bifurcation. The solution of the dynamical system near the bifurcation can
%be approximated by
%[S_R-S_R_0;S_K-S_K_0;N-N_0] ~
%         |Q_tot-Q_tot_0|^{3/2}*A(T)*eigvec*exp(i*omega*t) + complex conj.
%where [S_R_0;S_K_0;N_0] is the steady state solution at the bifurcation,
%eigvec is the normalized eigenvector corresponding to eigenvalue i*omega
%of the linearized dynamical system at the bifurcation, and T =
%(Q_tot-Q_tot_0), where Q_tot is the actual value of Q_tot and Q_tot_0 its
%value at the bifurcation point (all other parameters being the same as at
%the bifurcation). Note that neither the type1 nor the type2 parameter need
%to be set to Q_tot to compute the Landau coefficients as above. A positive
%value of the cubic Landau coefficient signifies a supercritical Hopf
%bifurcation with a stable limit cycle of continuously increasing amplitude
%emanating from the Hopf bifurcation. A negative value of the cubic Landau
%coefficient signifies a subcritical Hopf bifurcation with an unstable
%limit cycle that is not realized in forward integrations with noise. The
%linear Landau coefficient determines whether an increase in Q_tot (i.e.
%Q_tot-Q_tot_0) correspond to the system becoming stable (linear
%coefficient is positive) or unstable (linear coefficient is negative).
%
%Along with the Landau Coefficients, the code also computes the Jacobian of
%the dynamical system at the bifurcation points, its eigenvector eigvec,
%the eigenvalue omega, the corresponding left eigenvector eigvecleft, and
%the third (real) eigenvalue; if that third eigenvalue is positive, the
%steady state is unstable to growth of a non-oscilatory mode, as for
%instance associated with channelization.
%
%v5 also computes the stability boundary for the channlization problem:
%given n_c > 1 conduits in parallel, find the bifurcation at which the
%system with V_p = 0 is unstable to channelization. This is currently
%implemented for nu = T = 1; consider updating in future.
%
%Input variables:
%   v_in:       Initial guess for steady state solution [S_R;S_K;N] at
%               first bifurcation point concatenated with initial guess for
%               type2 parameter
%   type1:      First parameter type to be solved for. Choices are 'uh',
%               'Q_tot', 'n_c', 'L' and 'V_p'. Defaults to 'L'.
%   type2:      Second parameter type to be solved for. Choices are 'uh',
%               'Q_tot', 'n_c', 'L' and 'V_p'. Defaults to 'Q_tot', must
%               not be the same as type1. While parameters of type1 and
%               type2 are varied, all other model parameters are held
%               fixed. The only difference between the type1 and type2
%               parameters in practical terms is that the first point
%               computed has a prescribed value of the type1 aprameter.
%   lmin:       first value to be take by parameter of type1
%   ds:         arc length continuation step size
%   scalefactor: (diagonal) metric for computation of arc length, takes
%               the form of a 2-by-1 vector [a_1 a_2] so that arc
%               length is computed as \sum_i a_i dvl_i^2 if scale is
%               'linear', and as  \sum_i a_i d(log(l_i))^2 if scale is
%               'log', with l_i being the ith parameter (i = 1, 2); l_1 is
%               the type1 parameter and l2 is the type2 parameter
%   scale:  step size computed from change in parameters
%                ('linear', default) or from change in log(parameter)
%                ('log')
%   direction:  initial direction for changes in l, choices +1 and -1
%   npts:       number of points for which bifurcation points are to be
%               found
%   parameters: parameter structure containing all parameters for the
%               dynamical system. Those corresponding to the type1 and
%               type2 parameters are overwritten during the solution
%               process.
%   srchparams: optional parameter structure passed to arc length solver
%               ArcLength_single_symmetric, see documentation for that
%               function
%Output:
%   fout:       Output structure containing the following fields
%       parameters: The parameter structure originally supplied, with any
%               missing fields set to default values. Also contains fields
%               type1 and type2 specifying which parameters were varied and
%               did not take the values specified in this structure
%       uh, Q_tot, n_c, L or V_p:   There will be two fields out of these
%               possible five, corresponding to the type1 and type2
%               parameter values computed at the bifurcation. Each is a
%               npts-by-1 vector of parameter values
%       S_R, S_K, N: npts-by-1 vector of values of S_R, S_K and N at the
%               steady states corredponding to the type1 and type2 parmeter
%               values
%       error_flag: npts-by-1 vector of booleans indicating whether arc
%               length continuation solver did not (true) converge
%       thirdeig: npts-by-1 vector giving the third (real) eigenvalue of
%               the dynamical system at the bifurcation point
%       omega:  npts-by-1 vector of imaginary part of pure imaginary
%               eigenvalues at Hopf bifurcation, takes positive root by
%               convention
%       eigvec: npts-by-3 array of corresponding eigenvector
%       Jacobian: 3-by-3-by-npts array of Jacobians of dynamical system at
%               bifurcation
%       eigvecleft: npts-by-3 array of left eigenvectors
%       Landau_linear: npts-by-1 vector of linear Landau coefficients ('b'
%               in Landau equation above)
%       Landau_cubic:  npts-by-1 vector of cubic Landau coefficients ('c'
%               in Landau equation above)
%       restricted: for n_c = 1 cases, the dynamical system can be reduced
%               by fully decoupling the redundant variable S_K, replacing
%               the equation for dS_K/dt by
%               dS_K/dt = -S_K.
%               For n_c = 1, restricted is a substructure containing all
%               the output for eigvec, eigvecleft, omega, thirdeig, Jacobian,
%               Landau_linear and Landau_cubic as defubed above, but for
%               the reduced dynamical system. It is not clear that there
%               are any circumstances under which this would lead to Landau
%               coefficients with different signs, but is included as a
%               double check.
%       channelize: substructure containing two fields out of uh, Q_tot,
%               n_c, L or V_p (determined by type1 and type2) giving
%               parameters corresponding to the channelization  instability
%               as npts-by-one vectors, and fields S_R, S_K, N
%               corresponding to steady states at that bifurcation


%check input and set defaults as needed; for inital guess v_in, see further below
%parameter changes
if nargin < 2 || isempty(type1) || (~strcmp(type1,'uh') && ~strcmp(type1,'Q_tot') && ~strcmp(type1,'n_c') && ~strcmp(type1,'L') && ~strcmp(type1,'V_p'))
    type1 = 'L';
end
if nargin < 3 || isempty(type2) || (~strcmp(type2,'uh') && ~strcmp(type2,'Q_tot') && ~strcmp(type2,'n_c') && ~strcmp(type2,'L') && ~strcmp(type2,'V_p'))
    type2 = 'Q_tot';
end
%parameter range
if nargin < 4 || isempty(lmin)
    lmin = 1e-3;
end
if nargin < 5 || isempty(ds)
    ds = 1e-2;
end
if nargin < 6 || isempty(scalefactor)
    scalefactor = ones(length(v_in)+1,1);
end
if nargin < 7 || isempty(scale) || (~strcmp(scale,'log') && ~strcmp(scale,'linear'))
    scale = 'linear';
end
if nargin < 8 || isempty(direction)
    direction = +1;
end
%number of parameter steps
if nargin < 9 || isempty(npts)
    npts = 1000;
end
%physical parameters
if nargin < 10 || ~isfield(parameters,'L')
    parameters.L = 10;%1/eps;
end
if nargin < 10 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if ~isfield(parameters,'Q_tot')
    parameters.Q_tot = parameters.n_c;
end
if ~isfield(parameters,'uh_bar')
    parameters.uh_bar = 1;
end
if ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
end
if ~isfield(parameters,'m_store')
    parameters.m_store = 0;
end
if ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if ~isfield(parameters,'T')
    parameters.T = 1;
end
if ~isfield(parameters,'nu')
    parameters.nu = 1;
end
%set number of degrees of freedom
parameters.ndegf = length(v_in)-1;

if nargin < 11
    srchparams.itmax = 10;
    srchparams.toldelta = 1000*sqrt(eps);
    srchparams.verbose = 0;
end

%Hopf bifurcation test set-up
if nargin == 12 && isfield(flags,'Hopf_test') && flags.Hopf_test
    if isfield(flags,'theta')
        theta = flags.theta;
    else
        theta = pi/4;
    end
    %transformation matrix
    if isfield(flags,'R_trans')
        parameters.R_trans = flags.R_trans;
        parameters.R_inv = inv(flags.R_trans);
    else
        parameters.R_trans = [cos(theta) 0 sin(theta); 0 1 0; -sin(theta) 0 cos(theta)];
        parameters.R_inv =  [cos(theta) 0 -sin(theta); 0 1 0; sin(theta) 0 cos(theta)];
    end
    if isfield(flags,'a')
        parameters.a = flags.a;
    else
        parameters.a = pi;
    end
    if isfield(flags,'b')
        parameters.b = flags.b;
    else
        parameters.b = 1;
    end
    if isfield(flags,'c')
        parameters.c = flags.c;
    else
        parameters.c = 1;
    end
    parameters.mu = 0;
    v_in = [0 0 0 0]';
    parameters.ndegf = 3;
    parameters.Hopf_test = true;
    fout.parameters = parameters;
else
    parameters.Hopf_test = false;
end


%Ensure parameters to be solved for are specified as part of parameters
%structure
parameters.type1 = type1;
parameters.type2 = type2;

%derivative test
if nargin == 12 && isfield(flags,'test') && flags.test
    %exclude parameter value from argument
    if isfield(flags,'restrict') && flags.restrict
        parameters.restrict = true;
    else
        parameters.restrict = false;
    end
    v_in = v_in(1:end-1);
    fout.DF_test = Jacobian_test_v2(@F,@DF,rand(parameters.ndegf,1),parameters);
    fout.D2F_test = Jacobian_test_v2(@DF_vec,@D2F_mat,rand(parameters.ndegf,1),parameters);
    fout.D3F_test = Jacobian_test_v2(@D2F_vec,@D3F_mat,rand(parameters.ndegf,1),parameters);
    fout.Dcrit_test = Jacobian_test_single(@parallel_critical,rand(parameters.ndegf+2,1),parameters);
    parameters.v_in = v_in;
    parameters.type = parameters.type1;
    fout.DmuF_test = Jacobian_test_v2(@F_mu,@DmuF_mu,rand(1),parameters);
    fout.D2muF_test = Jacobian_test_v2(@DF_mu_vec,@D2muF_mu_mat,rand(1),parameters);
    [Jacobian thirdeig omega eigvec eigvecleft] = DF(v_in,parameters); 
    fout.eigvec_test = norm((Jacobian-1i*omega*eye(parameters.ndegf))*eigvec)/norm(eigvec); %only works at Hopf bifurcation
    fout.eigvecleft_test = norm(eigvecleft.'*(Jacobian-1i*omega*eye(parameters.ndegf)))/norm(eigvecleft);
    fout.parameters = parameters;
    if ~isfield(flags,'Hopf_test') || ~flags.Hopf_test, return, end
end

if nargin == 12 && isfield(flags,'Hopf_test') && flags.Hopf_test
    if ~isfield(flags,'test') || ~flags.test
        v_in  = v_in(1:end-1);
    end
    [fout.thirdeig fout.eigvec fout.eigvecleft fout.omega fout.Jacobian fout.Landau_linear fout.Landau_cubic] = Hopf(v_in,parameters);
    return
end

%initialize output
if strcmp(type1,type2), error('two different parameter types must be specified'), end
fout.type1 = type1;
switch type1
    case 'uh'
        fout.uh = zeros(npts,1);
        fout.channelize.uh = zeros(npts,1);
    case 'Q_tot'
        fout.Q_tot = zeros(npts,1);
        fout.channelize.Q_tot = zeros(npts,1);
    case 'n_c'
        fout.n_c = zeros(npts,1);
        fout.channelize.n_c = zeros(npts,1);
    case 'L'
        fout.L = zeros(npts,1);
        fout.channelize.L = zeros(npts,1);
    case 'V_p'
        fout.V_p_0 = zeros(npts,1);
        fout.channelize.V_p_0 = zeros(npts,1);
end
fout.type2 = type2;
switch type2
    case 'uh'
        fout.uh = zeros(npts,1);
        fout.channelize.uh = zeros(npts,1);
    case 'Q_tot'
        fout.Q_tot = zeros(npts,1);
        fout.channelize.Q_tot = zeros(npts,1);
    case 'n_c'
        fout.n_c = zeros(npts,1);
        fout.channelize.n_c = zeros(npts,1);
    case 'L'
        fout.L = zeros(npts,1);
        fout.channelize.L = zeros(npts,1);
    case 'V_p'
        fout.V_p_0 = zeros(npts,1);
        fout.channelize.V_p_0 = zeros(npts,1);
end
fout.S_R = zeros(npts,1);
fout.channelize.S_R = zeros(npts,1);
fout.S_K = zeros(npts,1);
fout.channelize.S_K = zeros(npts,1);
fout.N = zeros(npts,1);
fout.channelize.N = zeros(npts,1);
fout.error_flag = false(npts,1);
fout.thirdeig = zeros(npts,1);
fout.eigvec = zeros(npts,parameters.ndegf);
fout.eigvecleft = zeros(npts,parameters.ndegf);
fout.omega = zeros(npts,1);
fout.Jacobian = zeros(parameters.ndegf,parameters.ndegf,npts);
fout.Landau_linear = zeros(npts,1);
fout.Landau_cubic = zeros(npts,1);

if parameters.n_c==1
    fout.restricted.eigvec = zeros(npts,parameters.ndegf);
    fout.restricted.eigvecleft = zeros(npts,parameters.ndegf);
    fout.restricted.thirdeig = zeros(npts,parameters.ndegf);
    fout.restricted.omega = zeros(npts,1);
    fout.restricted.Jacobian = zeros(parameters.ndegf,parameters.ndegf,npts);
    fout.restricted.Landau_linear = zeros(npts,1);
    fout.restricted.Landau_cubic = zeros(npts,1);
end

%set initial guess if needed
if nargin == 0 || isempty(v_in)
    switch parameters.type1
        case 'uh'
            parameters.uh = lmin;
        case 'Q_tot'
            parameters.Q_tot = lmin;
        case 'n_c'
            parameters.n_c = lmin;
        case 'L'
            parameters.L = lmin;
        case 'V_p'
            parameters.V_p_0 = lmin;
    end
    switch type2
        case 'uh'
            l2_init = parameters.uh;
        case 'Q_tot'
            l2_init = parameters.Q_tot;
        case 'n_c'
            l2_init = parameters.n_c;
        case 'L'
            l2_init = parameters.L;
        case 'V_p'
            l2_init = parameters.V_p_0;
    end
    v_in_0 = parallel_initguess(parameters,'channel');
    v_in_0 = parallel_steady(v_in_0,parameters);
    v_in = [v_in_0; l2_init];
end

v_in_aug = [v_in; lmin];
%compute stability boundary for full system 
parameters.restrict = false;

[vtemp, fout.error_flag] = ArcLength_single_symmetric(@parallel_critical,v_in_aug,ds,scalefactor,scale,direction,npts,parameters,srchparams);

fout.S_R(:) = vtemp(1,:);
fout.S_K(:) = vtemp(2,:);
fout.N(:) = vtemp(3,:);
switch type2
        case 'uh'
            fout.uh(:) = vtemp(4,:);
        case 'Q_tot'
            fout.Q_tot(:) = vtemp(4,:);
        case 'n_c'
            fout.n_c(:) = vtemp(4,:);
        case 'L'
            fout.L(:) = vtemp(4,:);
        case 'V_p'
            fout.V_p_0(:) = vtemp(4,:);
end   
switch type1
        case 'uh'
            fout.uh(:) = vtemp(5,:);
        case 'Q_tot'
            fout.Q_tot(:) = vtemp(5,:);
        case 'n_c'
            fout.n_c(:) = vtemp(5,:);
        case 'L'
            fout.L(:) = vtemp(5,:);
        case 'V_p'
            fout.V_p_0(:) = vtemp(5,:);
end   


%now compute channlization stability boundary, this version assumes nu = T
%= 1 and equally-sized S_R and S_K elements
v_in_aug(2) = v_in_aug(1);
[vtemp2, fout.channelize.error_flag] = ArcLength_single_symmetric(@parallel_channelize,v_in_aug,ds,scalefactor,scale,direction,npts,parameters,srchparams);

fout.channelize.S_R(:) = vtemp2(1,:);
fout.channelize.S_K(:) = vtemp2(2,:);
fout.channelize.N(:) = vtemp2(3,:);
switch type2
        case 'uh'
            fout.channelize.uh(:) = vtemp2(4,:);
        case 'Q_tot'
            fout.channelize.Q_tot(:) = vtemp2(4,:);
        case 'n_c'
            fout.channelize.n_c(:) = vtemp2(4,:);
        case 'L'
            fout.channelize.L(:) = vtemp2(4,:);
        case 'V_p'
            fout.channelize.V_p_0(:) = vtemp2(4,:);
end   
switch type1
        case 'uh'
            fout.channelize.uh(:) = vtemp2(5,:);
        case 'Q_tot'
            fout.channelize.Q_tot(:) = vtemp2(5,:);
        case 'n_c'
            fout.channelize.n_c(:) = vtemp2(5,:);
        case 'L'
            fout.channelize.L(:) = vtemp2(5,:);
        case 'V_p'
            fout.channelize.V_p_0(:) = vtemp2(5,:);
end   


%postprocess: compute stability parameters / Landau coefficients
for ii=1:npts
    %set up input
    v_in = [fout.S_R(ii); fout.S_K(ii); fout.N(ii)];
    switch type1
        case 'uh'
            parameters.uh = fout.uh(ii);
        case 'Q_tot'
            parameters.Q_tot = fout.Q_tot(ii);
        case 'n_c'
            parameters.n_c = fout.n_c(ii);
        case 'L'
            parameters.L = fout.L(ii);
        case 'V_p'
            parameters.V_p_0 = fout.V_p_0(ii);
    end   
    switch type2
        case 'uh'
            parameters.uh = fout.uh(ii);
        case 'Q_tot'
            parameters.Q_tot = fout.Q_tot(ii);
        case 'n_c'
            parameters.n_c = fout.n_c(ii);
        case 'L'
            parameters.L = fout.L(ii);
        case 'V_p'
            parameters.V_p_0 = fout.V_p_0(ii);
    end   
    %set parameter type for bifurcation calculations to Q_tot
    parameters.type = 'Q_tot';
    %deal with full dynamical system first
    parameters.restrict = false;
    [fout.thirdeig(ii) fout.eigvec(ii,:) fout.eigvecleft(ii,:) fout.omega(ii,:) fout.Jacobian(:,:,ii) fout.Landau_linear(ii) fout.Landau_cubic(ii)] = Hopf(v_in,parameters);
    if parameters.n_c == 1
        parameters.restrict = true;%now with restricted system
        [fout.restricted.thirdeig(ii) fout.restricted.eigvec(ii,:) fout.restricted.eigvecleft(ii,:) fout.restricted.omega(ii,:) fout.restricted.Jacobian(:,:,ii) fout.restricted.Landau_linear(ii) fout.restricted.Landau_cubic(ii)] = Hopf(v_in,parameters);
    end
end

fout.parameters = parameters;
end


%%%code testing functions (non-problem specific)
function fout = DF_vec(v_in,parameters)
    ftemp = DF(v_in,parameters);
    fout = reshape(ftemp,parameters.ndegf^2,1);
end

function fout = D2F_mat(v_in,parameters)
    ftemp = D2F(v_in,parameters);
    fout = zeros(parameters.ndegf^2,parameters.ndegf);
    for ii = 1:parameters.ndegf
        fout(:,ii)=reshape(ftemp(:,:,ii),parameters.ndegf^2,1);
    end
end

function fout = D2F_vec(v_in,parameters)
    ftemp = D2F(v_in,parameters);
    fout = zeros(parameters.ndegf^2,parameters.ndegf);
    for ii=1:parameters.ndegf
        fout(:,ii)=reshape(ftemp(:,:,ii),parameters.ndegf^2,1);
    end
    fout = reshape(fout,parameters.ndegf^3,1);
end

function fout = D3F_mat (v_in,parameters)
    ftemp = D3F(v_in,parameters);
    ftemp2 = zeros(parameters.ndegf^2,parameters.ndegf,parameters.ndegf);
    fout = zeros(parameters.ndegf^3,parameters.ndegf);
    for jj=1:parameters.ndegf
        for ii=1:parameters.ndegf
            ftemp2(:,ii,jj) = reshape(ftemp(:,:,ii,jj),parameters.ndegf^2,1);
        end
        fout(:,jj)=reshape(ftemp2(:,:,jj),parameters.ndegf^3,1);
    end
end


%%%code testing function (problem specific)
function fout = F_mu(mu_in,parameters)
    v_in = parameters.v_in;
    if parameters.Hopf_test
        parameters.mu = mu_in;
    else
        switch parameters.type
            case 'uh'
                parameters.uh = mu_in;
            case 'Q_tot'
                parameters.Q_tot = mu_in;
            case 'n_c'
                parameters.n_c = mu_in;
            case 'L'
                parameters.L = mu_in;
            case 'V_p'
                parameters.V_p_0 = mu_in;
        end
    end   
    fout = F(v_in,parameters);
end

function fout = DmuF_mu(mu_in,parameters)
    v_in = parameters.v_in;
    if parameters.Hopf_test
        parameters.mu = mu_in;
    else
        switch parameters.type
            case 'uh'
                parameters.uh = mu_in;
            case 'Q_tot'
                parameters.Q_tot = mu_in;
            case 'n_c'
                parameters.n_c = mu_in;
            case 'L'
                parameters.L = mu_in;
            case 'V_p'
                parameters.V_p_0 = mu_in;
        end
    end
    fout = DmuF(v_in,parameters);
end

function fout = DF_mu_vec(mu_in,parameters)
    v_in = parameters.v_in;
    if parameters.Hopf_test
        parameters.mu = mu_in;
    else
        switch parameters.type
            case 'uh'
                parameters.uh = mu_in;
            case 'Q_tot'
                parameters.Q_tot = mu_in;
            case 'n_c'
                parameters.n_c = mu_in;
            case 'L'
                parameters.L = mu_in;
            case 'V_p'
                parameters.V_p_0 = mu_in;
        end
    end 
    ftemp = DF(v_in,parameters);
    fout = reshape(ftemp,parameters.ndegf^2,1);
end

function fout = D2muF_mu_mat(mu_in,parameters)
    v_in = parameters.v_in;
    if parameters.Hopf_test
        parameters.mu = mu_in;
    else
        switch parameters.type
            case 'uh'
                parameters.uh = mu_in;
            case 'Q_tot'
                parameters.Q_tot = mu_in;
            case 'n_c'
                parameters.n_c = mu_in;
            case 'L'
                parameters.L = mu_in;
            case 'V_p'
                parameters.V_p_0 = mu_in;
        end
    end
    fout = D2muF(v_in,parameters);
    fout = reshape(fout,parameters.ndegf^2,1);
end


%%%operational functions, non-problem specific

function [fout, Dfout] = parallel_critical(v_in_0,parameters)
%Re-routes to steady state and marginal stability condition functions

switch parameters.type2
    case 'uh'
        parameters.uh = v_in_0(4);
    case 'Q_tot'
        parameters.Q_tot = v_in_0(4);
    case 'n_c'
        parameters.n_c = v_in_0(4);
    case 'L'
        parameters.L = v_in_0(4);
    case 'V_p'
        parameters.V_p_0 = v_in_0(4);
end
switch parameters.type1
    case 'uh'
        parameters.uh = v_in_0(5);
    case 'Q_tot'
        parameters.Q_tot = v_in_0(5);
    case 'n_c'
        parameters.n_c = v_in_0(5);
    case 'L'
        parameters.L = v_in_0(5);
    case 'V_p'
        parameters.V_p_0 = v_in_0(5);
end

%Degree-of-freedom input
v_in = v_in_0(1:3);

%Compute steady state condition
faux = F(v_in,parameters);
Dfaux = DF(v_in,parameters);

%Compute marginal stability condition
fcrit = crit(v_in,parameters);
Dfcrit = Dcrit(v_in,parameters);

%Compute parameter derivatives
parameters.type = parameters.type1; %decides which parameter is "mu"
Dmu1faux = DmuF(v_in,parameters);
Dmu1fcrit = Dmucrit(v_in,parameters);
parameters.type = parameters.type2; %ditto
Dmu2faux = DmuF(v_in,parameters);
Dmu2fcrit = Dmucrit(v_in,parameters);

fout = [faux; fcrit];
Dfout = [Dfaux Dmu2faux Dmu1faux; Dfcrit Dmu2fcrit Dmu1fcrit];

end

function [fout, Dfout] = parallel_channelize(v_in_0,parameters)
%Re-routes to steady state and marginal stability condition functions

switch parameters.type2
    case 'uh'
        parameters.uh = v_in_0(4);
    case 'Q_tot'
        parameters.Q_tot = v_in_0(4);
    case 'n_c'
        parameters.n_c = v_in_0(4);
    case 'L'
        parameters.L = v_in_0(4);
    case 'V_p'
        parameters.V_p_0 = v_in_0(4);
end
switch parameters.type1
    case 'uh'
        parameters.uh = v_in_0(5);
    case 'Q_tot'
        parameters.Q_tot = v_in_0(5);
    case 'n_c'
        parameters.n_c = v_in_0(5);
    case 'L'
        parameters.L = v_in_0(5);
    case 'V_p'
        parameters.V_p_0 = v_in_0(5);
end

%Degree-of-freedom input
v_in = v_in_0(1:3);

%Compute steady state condition
faux = F(v_in,parameters);
Dfaux = DF(v_in,parameters);

%Fix S_K singularity problem
if parameters.n_c == 1
    faux(2) = 0;
    Dfaux(2,:) = [0 1 0];
end

%Compute marginal stability condition
fcrit = chann(v_in,parameters);
Dfcrit = Dchann(v_in,parameters);

%Compute parameter derivatives
parameters.type = parameters.type1; %decides which parameter is "mu"
Dmu1faux = DmuF(v_in,parameters);
Dmu1fcrit = Dmuchann(v_in,parameters);
parameters.type = parameters.type2; %ditto
Dmu2faux = DmuF(v_in,parameters);
Dmu2fcrit = Dmuchann(v_in,parameters);

fout = [faux; fcrit];
Dfout = [Dfaux Dmu2faux Dmu1faux; Dfcrit Dmu2fcrit Dmu1fcrit];

end

function [thirdeig eigvec eigvecleft omega Jacobian Landau_linear Landau_cubic] = Hopf(v_in,parameters)
%computes third (real) eigenvalue at Hopf bifurcation, as well as the
%linear and cubic coefficients in the Landau equation

%Jacobian matrix, third eigenvalue, imaginary eigenvaue and eigenvectors
[Jacobian thirdeig omega eigvec eigvecleft] = DF(v_in,parameters);

%Derivatives respect to forcing parameter Q_tot (Fmu)
Fmu = DmuF(v_in,parameters);
F2mu = D2muF(v_in,parameters);

%Second and third order arrays: F2(i,j,k) is d^2F_i/dv_jdv_k while
%F2(i,j,k,;) is d^3F_i/dv_jdv_kdv_l
F2 = D2F(v_in,parameters);
F3 = D3F(v_in,parameters);

Landau_linear_temp=zeros(1,3);
for ii=1:3
    Landau_linear_temp(ii) = eigvecleft.'*F2(:,:,ii)*(Jacobian\Fmu);
end
Landau_linear = (Landau_linear_temp*eigvec - eigvecleft.'*F2mu*eigvec)/(eigvecleft.'*eigvec);

%Compute cubic coefficient in Landau equation

%contractions over higher order tensors --- are there more elegant ways to
%do this (generalization of Kronecker product followed by sum?)
F2eigvec = zeros(3,1);
F2eigvec2 = zeros(3,1);
F3eigvec = zeros(3,3);
F2temp = permute(F2,[3 2 1]);
F3temp = permute(F3,[4 3 1 2]);
for ii=1:3
    F2eigvec(ii) = eigvec'*F2temp(:,:,ii)*eigvec;
    F2eigvec2(ii) =  eigvec.'*F2temp(:,:,ii)*eigvec;
    for jj=1:3
        F3eigvec(ii,jj) = eigvec'*F3temp(:,:,ii,jj)*eigvec;
    end
end

Landau_cubic_temp = zeros(3,1);
Landau_cubic_temp2 = zeros(3,1);
for ii=1:3
    Landau_cubic_temp(ii) = eigvecleft.'*F2(:,:,ii)*(Jacobian\F2eigvec);
    Landau_cubic_temp2(ii) = eigvecleft.'*F2(:,:,ii)*((2i*omega*eye(3)-Jacobian)\F2eigvec2)/2;
end

Landau_cubic = (eigvec.'*Landau_cubic_temp - eigvec'*Landau_cubic_temp2 - eigvecleft.'*F3eigvec*eigvec/2)/(eigvecleft.'*eigvec);
end

%%%problem-specific functions

function fout = storefactor(uh,parameters)
fout = uh.^parameters.m_store;
end

function Ds_out = Dstorefactor(uh,parameters)
Ds_out = parameters.m_store*(uh/parameters.uh_bar).^(parameters.m_store-1)/parameters.uh_bar;
end

function F0 = F(v_in,parameters)
%Right-hand side of dynamical system

%test case
if parameters.Hopf_test
    F0temp = zeros(3,1);
    R_trans = parameters.R_trans;
    R_inv = parameters.R_inv;
    mu = parameters.mu;
    a = parameters.a;
    b = parameters.b;
    c = parameters.c;
    vtemp = R_trans*v_in;
    vtemp2 = vtemp(1)^2 + vtemp(3)^2;
    F0temp(1) = -vtemp(3) + mu*vtemp(1) + a*(vtemp(1)^2 + vtemp(3)^2) + c*vtemp(1)*vtemp(3) + b*vtemp2*vtemp(1);
    F0temp(2) = -vtemp(2);
    F0temp(3) = vtemp(1) + mu*vtemp(3) + a*(vtemp(1)^2 + vtemp(3)^2) - c*vtemp(1)*vtemp(3) + b*vtemp2*vtemp(3);
    F0 = R_inv*F0temp;
    return
end

%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi); 

%output
F0 = zeros(3,1);
F0(1) = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
F0(2) = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*abs(N)^(n_Glen-1)*N;
F0(3) =  (Q_R + (n_c-1)*Q_K - Q_tot)/V_p;
if parameters.restrict
    F0(2) = -S_K;
end
end

function [Jacobian thirdeig omega eigvec eigvecleft] = DF(v_in,parameters)
%Jacobian: Jacobian of right-hand side of dynamical system
%thirdeig: Third eigenvalue at Hopf bifurcation (optional)
%omega: Frequency at Hopf bifurcation (optional)
%eigvec: right (ordinary) eigenvector
%eigvecleft: left eigenvector, transposed

%test case
if parameters.Hopf_test
    DFtemp = zeros(3,3);
    R_trans = parameters.R_trans;
    R_inv = parameters.R_inv;
    mu = parameters.mu;
    a = parameters.a;
    b = parameters.b;
    c = parameters.c;
    vtemp = R_trans*v_in;
    vtemp2 = vtemp(1)^2 + vtemp(3)^2;
    DFtemp(1,1) = mu + 2*a*vtemp(1) + c*vtemp(3) + b*vtemp2 + 2*b*vtemp(1)^2;
    DFtemp(1,3) = -1 + 2*a*vtemp(3) +  c*vtemp(1) + 2*b*vtemp(3)*vtemp(1);
    DFtemp(2,2) = -1;
    DFtemp(3,1) = 1 + 2*a*vtemp(1) - c*vtemp(3) + 2*b*vtemp(1)*vtemp(3);
    DFtemp(3,3) = mu + 2*a*vtemp(3) - c*vtemp(1) + b*vtemp2 + 2*b*vtemp(3)^2;
    Jacobian = R_inv*DFtemp*R_trans;
    if nargout > 1
        thirdeig = -1;
        omega = 1;
        eigvectemp = [1; 0; -1i]/sqrt(2);
        eigveclefttemp = conj(eigvectemp);
        eigvec = R_inv*eigvectemp;
        eigvecleft = R_trans.'*eigveclefttemp;
    end
    return
end

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%derivatives in notebook calculation --- signs for dv_RdN, dv_KdN and dQdN
%have been flipped, same as in stability_criterion above, to make them
%consistent with use in the Jacobian computation
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K*Psi/T - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = c_1*(dQ_KdPsi*Psi + Q_K)*dPsidN/T - n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%Jacobian matrix
Jacobian = [dv_RdS_R 0 dv_RdN; 0 dv_KdS_K dv_KdN; dQ_RdS_R/V_p (n_c-1)*dQ_KdS_K/V_p dQdN/V_p];

%optional additional output
if nargout > 1
    %third (real) eigenvalue at Hopf bifurcation
    thirdeig = (dv_RdS_R + dv_KdS_K) + dQdN/V_p;
    %imaginary eigenvalues
    omega = sqrt( -(n_c-1)*dQ_KdS_K*dv_KdN/V_p - dQ_RdS_R*dv_RdN/V_p + dQdN*(dv_RdS_R+dv_KdS_K)/V_p +  dv_RdS_R*dv_KdS_K );
    %corresponding eigenvector (the complex conjugate of eigvec corresponds to
    %the eigenvalue -i*omega
    eigvec = [-dv_RdN/(dv_RdS_R-1i*omega); -dv_KdN/(dv_KdS_K-1i*omega); 1];
    eigvec = eigvec/norm(eigvec);
    eigvecleft =  [-dQ_RdS_R/(V_p*(dv_RdS_R-1i*omega)); -(n_c-1)*dQ_KdS_K/(V_p*(dv_KdS_K-1i*omega)); 1];
    eigvecleft = eigvecleft/norm(eigvecleft);
end

if parameters.restrict
    Jacobian(2,:) = 0;
    Jacobian(:,2) = 0;
    Jacobian(2,2) = -1;
    if nargout > 1
        eigvec(2) = 0;
        eigvec = eigvec/norm(eigvec);
        eigvecleft(2) = 0;
        eigvecleft = eigvecleft/norm(eigvecleft);
        thirdeig = -1;
    end
end

if nargout > 1
    eigtest(Jacobian,omega,eigvec,eigvecleft,thirdeig,parameters)
end
end

function Fmu = DmuF(v_in,parameters)
%Derivative of right-hand side of dynamical system with respect to
%parameter specified in parameters.type field

%test case
if parameters.Hopf_test
    Fmu = zeros(3,1);
    Fmu(1) = v_in(1);
    Fmu(2) = v_in(3);
    return
end

%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%output
Fmu = zeros(3,1);
switch parameters.type
    case 'uh'
        Psi = Psi_0 - N/L;
        Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
        Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi);
        dV_pduh = Dstorefactor(parameters.uh,parameters)*V_p_0;        
        Fmu(1) = nu*(1-S_R/S_0_R);
        Fmu(2) = (1-S_K/S_0_K);
        Fmu(3) = -(Q_R + (n_c-1)*Q_K - Q_tot)/V_p^2*dV_pduh;
    case 'Q_tot'
        Fmu(3) = -1/V_p;
    case 'n_c'
        Psi = Psi_0 - N/L;
        Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi);
        Fmu(3) = Q_K;
    case 'L'
        Psi = Psi_0 - N/L;
        Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
        Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi); 
        dQ_RdPsi = (beta-1)*Q_R/Psi;
        dQ_KdPsi = (beta-1)*Q_K/Psi;
        dPsidL = N/L^2;
        Fmu(1) = c_1*(dQ_RdPsi*Psi + Q_R)*dPsidL;
        Fmu(2) = c_1*(dQ_KdPsi*Psi + Q_K)*dPsidL/T;
        Fmu(3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidL/V_p;
    case 'V_p'
        Psi = Psi_0 - N/L;
        Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
        Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi);
        dV_pdV_p_0 = storefactor(parameters.uh,parameters);
        Fmu(3) = -(Q_R + (n_c-1)*Q_K - Q_tot)/V_p^2*dV_pdV_p_0;
end

if parameters.restrict
    Fmu(2) = 0;
end

end

function F2mu = D2muF(v_in,parameters)
%Derivative of Jacobian with respect toparameter specified in
%parameters.type field

%test case
if parameters.Hopf_test
    D2muFtemp = zeros(3,3);
    R_trans = parameters.R_trans;
    R_inv = parameters.R_inv;
    D2muFtemp(1,1) = 1;
    D2muFtemp(3,3) = 1;
    F2mu = R_inv*D2muFtemp*R_trans;
    return
end

%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
dQdN = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;


%output
F2mu = zeros(3,3);
switch parameters.type
    case 'uh'
        %auxiliary calculation
        dV_pduh = Dstorefactor(parameters.uh,parameters)*V_p_0;        
        %derivatives of notebook parameters --- uh
        d2v_RdS_Rduh =  - nu/S_0_R;
        d2v_KdS_Kduh = - 1/S_0_K;
        %output
        F2mu(1,1) = d2v_RdS_Rduh;
        F2mu(2,2) = d2v_KdS_Kduh;
        F2mu(3,:) = -[dQ_RdS_R (n_c-1)*dQ_KdS_K dQdN]/V_p^2*dV_pduh;
    case 'n_c'
        F2mu(3,2) = dQ_KdS_K/V_p;
    case 'L'
        %other auxiliary calculations
        dPsidL = N/L^2;
        d2PsidNdL = 1/L^2;
        %second derivatives
        d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
        d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
        d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
        d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
        %derivatives of notebook parameters --- L
        d2v_RdS_RdL = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidL- nu*uh/S_0_R;
        d2v_KdS_KdL = c_1*(d2Q_KdS_KdPsi/T*Psi + dQ_KdS_K)*dPsidL;
        d2v_RdNdL = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN*dPsidL + c_1*(dQ_RdPsi*Psi + Q_R)*d2PsidNdL ;
        d2v_KdNdL = c_1*(d2Q_KdPsi2*Psi + 2*dQ_KdPsi)*dPsidN*dPsidL/T + c_1*(dQ_KdPsi/T*Psi + Q_K/T)*d2PsidNdL;
        d2QdNdL = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN*dPsidL +  (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*d2PsidNdL;
        d2Q_RdS_RdL = d2Q_RdS_RdPsi*dPsidL;
        d2Q_KdS_KdL = d2Q_KdS_KdPsi*dPsidL;        
        %output
        F2mu = [d2v_RdS_RdL 0 d2v_RdNdL; 0 d2v_KdS_KdL d2v_KdNdL; d2Q_RdS_RdL/V_p (n_c-1)*d2Q_KdS_KdL/V_p d2QdNdL/V_p];
    case 'V_p'
        %auxiliary calculation
        dV_pdV_p_0 = storefactor(parameters.uh,parameters);      
        %output --- condition for Hopf bifurcation, corrected for flipped signs
        F2mu(3,:) = -[dQ_RdS_R (n_c-1)*dQ_KdS_K dQdN]/V_p^2*dV_pdV_p_0;
end

if parameters.restrict
    F2mu(2,:) = 0;
end

end

function F2 = D2F(v_in,parameters)
%Second derivative array of right-hand side of dynamical system

%test case
if parameters.Hopf_test
    F2 = zeros(3,3,3);
    F2temp = zeros(3,3,3);
    D2Ftemp = zeros(3,3,3);
    R_trans = parameters.R_trans;
    R_inv = parameters.R_inv;
    a = parameters.a;
    b = parameters.b;
    c = parameters.c;
    vtemp = R_trans*v_in;
    D2Ftemp(1,1,1) = 2*a + 6*b*vtemp(1);
    D2Ftemp(1,1,3) = c + 2*b*vtemp(3);
    D2Ftemp(1,3,1) = D2Ftemp(1,1,3);
    D2Ftemp(1,3,3) = 2*a + 2*b*vtemp(1);
    D2Ftemp(3,1,1) = 2*a + 2*b*vtemp(3);
    D2Ftemp(3,1,3) = -c + 2*b*vtemp(1);
    D2Ftemp(3,3,1) = D2Ftemp(3,1,3);
    D2Ftemp(3,3,3) = 2*a + 6*b*vtemp(3);
    %rearrange to do matrix computations
    D2Ftemp = permute(D2Ftemp,[2 3 1]);
    for ii=1:3
        F2temp(ii,:,:) = R_trans.'*D2Ftemp(:,:,ii)*R_trans;
    end
    for ii = 1:3
        F2(:,:,ii) = R_inv*F2temp(:,:,ii);
        %for jj = 1:3
            %F2(ii,:,:) = F2(ii,:,:) + R_inv(ii,jj)*F2temp(jj,:,:);
        %end
    end
    return
end

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
T = parameters.T;           %tortuosities for 'cavities'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%Higher order derivatives
d2Q_RdS_R2 = alpha*(alpha-1)*Q_R/S_R^2;
d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;

d2Q_KdS_K2 = alpha*(alpha-1)*Q_K/S_K^2;
d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;

d2v_cRdS_RdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cRdN2 = c_2*n_Glen*(n_Glen-1)*S_R*abs(N)^(n_Glen-3)*N;

d2v_cKdS_KdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cKdN2 = c_2*n_Glen*(n_Glen-1)*S_K*abs(N)^(n_Glen-3)*N;

%intialize
F2 = zeros(3,3,3);

%Construct second derivitve array
F2(1,1,1) = c_1*d2Q_RdS_R2*Psi;
F2(1,1,3) = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidN - d2v_cRdS_RdN;
F2(1,3,1) = F2(1,1,3);
F2(1,3,3) = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN^2 - d2v_cRdN2;
F2(2,2,2) = c_1*d2Q_KdS_K2*Psi/T;
F2(2,2,3) = c_1*(d2Q_KdS_KdPsi*Psi + dQ_KdS_K)*dPsidN/T - d2v_cKdS_KdN;
F2(2,3,2) = F2(2,2,3);
F2(2,3,3) = c_1*(d2Q_KdPsi2*Psi + 2*dQ_KdPsi)*dPsidN^2/T - d2v_cKdN2;
F2(3,1,1) = d2Q_RdS_R2/V_p;
F2(3,1,3) = d2Q_RdS_RdPsi*dPsidN/V_p;
F2(3,3,1) = F2(3,1,3);
F2(3,2,2) = (n_c-1)*d2Q_KdS_K2/V_p;
F2(3,2,3) = (n_c-1)*d2Q_KdS_KdPsi*dPsidN/V_p;
F2(3,3,2) = F2(3,2,3);
F2(3,3,3) = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN^2/V_p;

if parameters.restrict
    F2(2,:,:) = 0;
    F2(:,2,:) = 0;
    F2(:,:,2) = 0;
end

end

function F3 = D3F(v_in,parameters)
%Third derivatives array of right-hand side of dynamical system

%test case
if parameters.Hopf_test
    F3 = zeros(3,3,3,3);
    F3temp = zeros(3,3,3,3);
    D3Ftemp = zeros(3,3,3,3);
    R_trans = parameters.R_trans;
    R_inv = parameters.R_inv;
    b = parameters.b;
    D3Ftemp(1,1,1,1) = 6*b;
    D3Ftemp(1,1,3,3) = 2*b;
    D3Ftemp(1,3,1,3) = D3Ftemp(1,1,3,3);
    D3Ftemp(1,3,3,1) = D3Ftemp(1,1,3,3);
    D3Ftemp(3,3,1,1) = 2*b;
    D3Ftemp(3,1,3,1) = D3Ftemp(3,3,1,1);
    D3Ftemp(3,1,1,3) = D3Ftemp(3,3,1,1);    
    D3Ftemp(3,3,3,3) = 6*b;
    %rearrange to do matrix computations
    D3Ftemp = permute(D3Ftemp,[3 4 2 1]);
    for ii=1:3
        for jj=1:3
            F3temp(:,:,jj,ii) = R_trans.'*D3Ftemp(:,:,jj,ii)*R_trans;
        end
    end
    F3temp = permute(F3temp,[4 3 1 2]);
    for ii = 1:3
        for jj = 1:3
            F3(:,:,ii,jj) = R_inv*F3temp(:,:,ii,jj)*R_trans;
        end
    end
    return
end

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
T = parameters.T;           %tortuosities for 'cavities'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 

%Higher order derivatives
d2Q_RdS_R2 = alpha*(alpha-1)*Q_R/S_R^2;
d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
d3Q_RdS_R3 = alpha*(alpha-1)*(alpha-2)*Q_R/S_R^3;
d3Q_RdS_R2dPsi = alpha*(alpha-1)*(beta-1)*Q_R/(S_R^2*Psi);
d3Q_RdS_RdPsi2 = alpha*(beta-1)*(beta-2)*Q_R/(S_R*Psi^2);
d3Q_RdPsi3 = (beta-1)*(beta-2)*(beta-3)*Q_R/Psi^3;

d2Q_KdS_K2 = alpha*(alpha-1)*Q_K/S_K^2;
d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
d3Q_KdS_K3 = alpha*(alpha-1)*(alpha-2)*Q_K/S_K^3;
d3Q_KdS_K2dPsi = alpha*(alpha-1)*(beta-1)*Q_K/(S_K^2*Psi);
d3Q_KdS_KdPsi2 = alpha*(beta-1)*(beta-2)*Q_K/(S_K*Psi^2);
d3Q_KdPsi3 = (beta-1)*(beta-2)*(beta-3)*Q_K/Psi^3;

d3v_cRdS_RdN2 = c_2*n_Glen*(n_Glen-1)*abs(N)^(n_Glen-3)*N;
d3v_cRdN3 = c_2*n_Glen*(n_Glen-1)*(n_Glen-2)*S_R*abs(N)^(n_Glen-3);

d3v_cKdS_KdN2 = c_2*n_Glen*(n_Glen-1)*abs(N)^(n_Glen-3)*N;
d3v_cKdN3 = c_2*n_Glen*(n_Glen-1)*(n_Glen-2)*S_K*abs(N)^(n_Glen-3);

%initialize
F3 = zeros(3,3,3,3);

%Third order array
F3(1,1,1,1) = c_1*d3Q_RdS_R3*Psi;
F3(1,1,1,3) = c_1*(d3Q_RdS_R2dPsi*Psi + d2Q_RdS_R2)*dPsidN;
F3(1,1,3,1) = F3(1,1,1,3);
F3(1,3,1,1) = F3(1,1,1,3);
F3(1,1,3,3) = c_1*(d3Q_RdS_RdPsi2*Psi + 2*d2Q_RdS_RdPsi)*dPsidN^2 - d3v_cRdS_RdN2;
F3(1,3,1,3) = F3(1,1,3,3);
F3(1,3,3,1) = F3(1,1,3,3);
F3(1,3,3,3) = c_1*(d3Q_RdPsi3*Psi + 3*d2Q_RdPsi2)*dPsidN^3 - d3v_cRdN3;
F3(2,2,2,2) = c_1*d3Q_KdS_K3*Psi/T;
F3(2,2,2,3) = c_1*(d3Q_KdS_K2dPsi*Psi + d2Q_KdS_K2)*dPsidN/T;
F3(2,2,3,2) = F3(2,2,2,3);
F3(2,3,2,2) = F3(2,2,2,3);
F3(2,2,3,3) = c_1*(d3Q_KdS_KdPsi2*Psi + 2*d2Q_KdS_KdPsi)*dPsidN^2/T - d3v_cKdS_KdN2;
F3(2,3,2,3) = F3(2,2,3,3);
F3(2,3,3,2) = F3(2,2,3,3);
F3(2,3,3,3) = c_1*(d3Q_KdPsi3*Psi + 3*d2Q_KdPsi2)*dPsidN^3/T - d3v_cKdN3;
F3(3,1,1,1) = d3Q_RdS_R3/V_p;
F3(3,1,1,3) = d3Q_RdS_R2dPsi*dPsidN/V_p;
F3(3,1,3,1) = F3(3,1,1,3);
F3(3,3,1,1) = F3(3,1,1,3);
F3(3,1,3,3) = d3Q_RdS_RdPsi2*dPsidN^2/V_p;
F3(3,3,1,3) = F3(3,1,3,3);
F3(3,3,3,1) = F3(3,1,3,3);
F3(3,2,2,2) = (n_c-1)*d3Q_KdS_K3/V_p;
F3(3,2,2,3) = (n_c-1)*d3Q_KdS_K2dPsi*dPsidN/V_p;
F3(3,2,3,2) = F3(3,2,2,3);
F3(3,3,2,2) = F3(3,2,2,3);
F3(3,2,3,3) = (n_c-1)*d3Q_KdS_KdPsi2*dPsidN^2/V_p;
F3(3,3,2,3) = F3(3,2,3,3);
F3(3,3,3,2) = F3(3,2,3,3);
F3(3,3,3,3) = (d3Q_RdPsi3 + (n_c-1)*d3Q_KdPsi3)*dPsidN^3/V_p;

if parameters.restrict
    F3(2,:,:,:) = 0;
    F3(:,2,:,:) = 0;
    F3(:,:,2,:) = 0;
    F3(:,:,:,2) = 0;
end

end

function Fcrit = crit(v_in,parameters)
%Marginal stability criterion for Hopf bifurcation;

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%parameters in notebook calculation -- signs flipped from original notebook
%computation for v_RdN, dv_KdN and dQdN
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN - n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%output --- condition for Hopf bifurcation, corrected for flipped signs
Fcrit = (n_c-1)*dQ_KdS_K*dv_KdN*(dQdN + V_p*dv_KdS_K)...
    + dQ_RdS_R*dv_RdN*(dQdN + V_p*dv_RdS_R)...
    - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
    - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*dv_KdS_K;
end

function DFcrit = Dcrit(v_in,parameters)
%derivative of marginal stability criterion with respect to
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%second derivatives
d2Q_RdS_R2 = alpha*(alpha-1)*Q_R/S_R^2;
d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
d2Q_KdS_K2 = alpha*(alpha-1)*Q_K/S_K^2;
d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
d2v_cRdS_RdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cRdN2 = c_2*n_Glen*(n_Glen-1)*S_R*abs(N)^(n_Glen-3)*N;
d2v_cKdS_KdN = c_2*n_Glen*abs(N)^(n_Glen-1);
d2v_cKdN2 = c_2*n_Glen*(n_Glen-1)*S_K*abs(N)^(n_Glen-3)*N;

%parameters in notebook calculation -- signs flipped from original notebook
%computation for v_RdN, dv_KdN and dQdN
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN - n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%derivatives of notebook parameters
d2v_RdS_R2 = c_1*d2Q_RdS_R2*Psi;
d2v_RdS_RdN = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidN - d2v_cRdS_RdN;
d2v_RdN2 = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN^2 - d2v_cRdN2;
d2v_KdS_K2 = c_1*d2Q_KdS_K2*Psi/T;
d2v_KdS_KdN = c_1*(d2Q_KdS_KdPsi*Psi + dQ_KdS_K)*dPsidN/T - d2v_cKdS_KdN;
d2v_KdN2 = c_1*(d2Q_KdPsi2*Psi + 2*dQ_KdPsi)*dPsidN^2/T - d2v_cKdN2;
d2QdNdS_R = d2Q_RdS_RdPsi*dPsidN;
d2QdNdS_K = (n_c-1)*d2Q_KdS_KdPsi*dPsidN;
d2QdN2 = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN^2;
d2Q_RdNdS_R = d2Q_RdS_RdPsi*dPsidN;
d2Q_KdNdS_K = d2Q_KdS_KdPsi*dPsidN;

%output --- condition for Hopf bifurcation
DFcrit = zeros(1,3);

DFcrit(1) = (n_c-1)*dQ_KdS_K*dv_KdN*d2QdNdS_R...
    + d2Q_RdS_R2*dv_RdN*(dQdN + V_p*dv_RdS_R)...
        + dQ_RdS_R*d2v_RdS_RdN*(dQdN + V_p*dv_RdS_R)...
            + dQ_RdS_R*dv_RdN*(d2QdNdS_R + V_p*d2v_RdS_R2)...
    - d2QdNdS_R*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
        - dQdN*d2v_RdS_R2*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
            - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*d2v_RdS_R2+d2QdNdS_R)...
    - V_p^2*d2v_RdS_R2*dv_RdS_R*dv_KdS_K...
        - V_p^2*dv_RdS_R*d2v_RdS_R2*dv_KdS_K;

DFcrit(2) = (n_c-1)*d2Q_KdS_K2*dv_KdN*(dQdN + V_p*dv_KdS_K)...
                + (n_c-1)*dQ_KdS_K*d2v_KdS_KdN*(dQdN + V_p*dv_KdS_K)...
                    + (n_c-1)*dQ_KdS_K*dv_KdN*(d2QdNdS_K + V_p*d2v_KdS_K2)...
    + dQ_RdS_R*dv_RdN*d2QdNdS_K...
    - d2QdNdS_K*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
        - dQdN*d2v_KdS_K2*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
            - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*d2v_KdS_K2+d2QdNdS_K)...
    - V_p^2*d2v_KdS_K2*dv_RdS_R*dv_KdS_K...
         - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*d2v_KdS_K2;

DFcrit(3) = (n_c-1)*d2Q_KdNdS_K*dv_KdN*(dQdN + V_p*dv_KdS_K)...
        + (n_c-1)*dQ_KdS_K*d2v_KdN2*(dQdN + V_p*dv_KdS_K)...
                + (n_c-1)*dQ_KdS_K*dv_KdN*(d2QdN2 + V_p*d2v_KdS_KdN)...
    + d2Q_RdNdS_R*dv_RdN*(dQdN + V_p*dv_RdS_R)...
        + dQ_RdS_R*d2v_RdN2*(dQdN + V_p*dv_RdS_R)...
            + dQ_RdS_R*dv_RdN*(d2QdN2 + V_p*d2v_RdS_RdN)...
    - d2QdN2*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
        - dQdN*(d2v_RdS_RdN+d2v_KdS_KdN)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
            - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*(d2v_RdS_RdN+d2v_KdS_KdN)+d2QdN2)...
    - V_p^2*(d2v_RdS_RdN+d2v_KdS_KdN)*dv_RdS_R*dv_KdS_K...
        - V_p^2*(dv_RdS_R+dv_KdS_K)*d2v_RdS_RdN*dv_KdS_K...
            - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*d2v_KdS_KdN;
        
 %output --- condition for Hopf bifurcation, corrected for flipped signs
%Fcrit = (n_c-1)*dQ_KdS_K*dv_KdN*(dQdN + V_p*dv_KdS_K)...
%    + dQ_RdS_R*dv_RdN*(dQdN + V_p*dv_RdS_R)...
%    - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
%    - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*dv_KdS_K;       
        
        
end


function DmuFcrit = Dmucrit(v_in,parameters)
%derivative of marginal stability criterion with respect to
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

if strcmp(parameters.type,'Q_tot')
    DmuFcrit = 0;
    return
end

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;
%parameters in notebook calculation -- signs flipped from original notebook
%computation for v_RdN, dv_KdN and dQdN
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN - n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;


switch parameters.type
    case 'uh'
        %auxiliary calculation
        dV_pduh = Dstorefactor(parameters.uh,parameters)*V_p_0;        
        %derivatives of notebook parameters --- uh
        d2v_RdS_Rduh =  - nu/S_0_R;
        d2v_KdS_Kduh = - 1/S_0_K;
        %output
        DmuFcrit = (n_c-1)*dQ_KdS_K*dv_KdN*(dQdN + V_p*d2v_KdS_Kduh)...
            + dQ_RdS_R*dv_RdN*V_p*d2v_RdS_Rduh...
            - dQdN*(d2v_RdS_Rduh+d2v_KdS_Kduh)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
                - dQdN*(dv_RdS_R+dv_KdS_K)*V_p*(d2v_RdS_Rduh+d2v_KdS_Kduh)...
            - V_p^2*(d2v_RdS_Rduh+d2v_KdS_Kduh)*dv_RdS_R*dv_KdS_K...
                - V_p^2*(dv_RdS_R+dv_KdS_K)*d2v_RdS_Rduh*dv_KdS_K...
                    - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*d2v_KdS_Kduh... %now add derivatives appearing through V_p
            +(n_c-1)*dQ_KdS_K*dv_KdN*dV_pduh*dv_KdS_K...
            + dQ_RdS_R*dv_RdN*dV_pduh*dv_RdS_R...
            - dQdN*(dv_RdS_R+dv_KdS_K)*dV_pduh*(dv_RdS_R+dv_KdS_K)...
            - 2*V_p*dV_pduh(dv_RdS_R+dv_KdS_K)*dv_RdS_R*dv_KdS_K;
    case 'n_c'
        DmuFcrit = dQ_KdS_K*dv_KdN*(dQdN + V_p*dv_KdS_K);
    case 'L'
        %other auxiliary calculations
        d2PsidN2 = 0;
        dPsidL = N/L^2;
        d2PsidNdL = 1/L^2;
        %second derivatives
        d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
        d2Q_RdPsi2 = (beta-1)*(beta-2)*Q_R/Psi^2;
        d2Q_KdS_KdPsi = alpha*(beta-1)*Q_K/(S_K*Psi);
        d2Q_KdPsi2 = (beta-1)*(beta-2)*Q_K/Psi^2;
        %derivatives of notebook parameters --- L
        d2Q_RdS_RdL = d2Q_RdS_RdPsi*dPsidL;
        d2Q_KdS_KdL = d2Q_KdS_KdPsi*dPsidL;
        d2v_RdS_RdL = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidL- nu*uh/S_0_R;
        d2v_KdS_KdL = c_1*(d2Q_KdS_KdPsi*Psi + dQ_KdS_K)*dPsidL/T;
        d2v_RdNdL = c_1*(d2Q_RdPsi2*Psi + 2*dQ_RdPsi)*dPsidN*dPsidL + c_1*(dQ_RdPsi*Psi + Q_R)*d2PsidNdL ;
        d2v_KdNdL = c_1*(d2Q_KdPsi2/T*Psi + 2*Q_K/T)*dPsidN + c_1*(dQ_KdPsi/T*Psi + Q_K/T)*d2PsidNdL;
        d2QdNdL = (d2Q_RdPsi2 + (n_c-1)*d2Q_KdPsi2)*dPsidN*dPsidL +  (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*d2PsidNdL;
        %output
        DmuFcrit =  (n_c-1)*d2Q_KdS_KdL*dv_KdN*(dQdN + V_p*dv_KdS_K)...
                + (n_c-1)*dQ_KdS_K*d2v_KdNdL*(dQdN + V_p*dv_KdS_K)...
                    + (n_c-1)*dQ_KdS_K*dv_KdN*(d2QdNdL + V_p*d2v_KdS_KdL)...
            + d2Q_RdS_RdL*dv_RdN*(dQdN + V_p*dv_RdS_R)...
                + dQ_RdS_R*d2v_RdNdL*(dQdN + V_p*dv_RdS_R)...
                    + dQ_RdS_R*dv_RdN*(d2QdNdL + V_p*d2v_RdS_RdL)...
            - d2QdNdL*(dv_RdS_R+dv_KdS_K)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
                - dQdN*(d2v_RdS_RdL+d2v_KdS_KdL)*(V_p*(dv_RdS_R+dv_KdS_K)+dQdN)...
                    - dQdN*(dv_RdS_R+dv_KdS_K)*(V_p*(d2v_RdS_RdL+d2v_KdS_KdL)+d2QdNdL)...
            - V_p^2*(d2v_RdS_RdL+d2v_KdS_KdL)*dv_RdS_R*dv_KdS_K...
                - V_p^2*(dv_RdS_R+dv_KdS_K)*d2v_RdS_RdL*dv_KdS_K...
                    - V_p^2*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*d2v_KdS_KdL;
    case 'V_p'
        %auxiliary calculation
        dV_pdV_p_0 = storefactor(parameters.uh,parameters);      
        %output --- condition for Hopf bifurcation, corrected for flipped signs
        DmuFcrit = (n_c-1)*dQ_KdS_K*dv_KdN*dV_pdV_p_0*dv_KdS_K...
            + dQ_RdS_R*dv_RdN*dV_pdV_p_0*dv_RdS_R...
            - dQdN*(dv_RdS_R+dv_KdS_K)*dV_pdV_p_0*(dv_RdS_R+dv_KdS_K)...
            - 2*V_p*dV_pdV_p_0*(dv_RdS_R+dv_KdS_K)*dv_RdS_R*dv_KdS_K;
end

end


function Fchann = chann(v_in,parameters)
%Marginal stability criterion for channelizing instability; currently
%implemented for nu = T = 1!

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;

%Stability criterion
Fchann = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
end

function DFchann = Dchann(v_in,parameters)
%derivative of marginal stability criterion for channelization with respect
%to v_in
L = parameters.L;           %domain length
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate

%unpack input
S_R = v_in(1);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;

%second derivatives
d2Q_RdS_R2 = alpha*(alpha-1)*Q_R/S_R^2;
d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
d2v_cRdS_RdN = c_2*n_Glen*abs(N)^(n_Glen-1);

%derivatives of notebook parameters
d2v_RdS_R2 = c_1*d2Q_RdS_R2*Psi;
d2v_RdS_RdN = c_1*(d2Q_RdS_RdPsi*Psi + dQ_RdS_R)*dPsidN - d2v_cRdS_RdN;

%output
DFchann = zeros(1,3);
DFchann(1) = d2v_RdS_R2;
DFchann(3) = d2v_RdS_RdN;        
end


function DmuFchann = Dmuchann(v_in,parameters)
%derivative of marginal stability criterion for channelization with respect
%to parameters
L = parameters.L;           %domain length
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
N = v_in(3);

if strcmp(parameters.type,'Q_tot') || strcmp(parameters.type,'n_c') || strcmp(parameters.type,'V_p') 
    DmuFchann = 0;
    return
end

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;


 
switch parameters.type
    case 'uh'
        %output
        DmuFchann = - nu/S_0_R;

    case 'L'
        %other auxiliary calculations
        dPsidL = N/L^2;
        %second derivatives
        d2Q_RdS_RdPsi = alpha*(beta-1)*Q_R/(S_R*Psi);
        %derivatives of notebook parameters --- L
        %output
        DmuFchann = c_1*(d2Q_RdS_RdPsi+Psi + dQ_RdS_R)*dPsidL;
end

end

function eigtest(Jacobian,omega,eigvec,eigvecleft,thirdeig,parameters)
%double check using MATLAB eigenvalue solver

%numerical eigenvalue computations
[EigVecMat, EigValMat] = eig(Jacobian);
[EigVecMatLeft, EigValMatLeft] = eig(Jacobian.');
EigValMat = EigValMat(1+(0:2)*4).';
EigValMatLeft = EigValMatLeft(1+(0:2)*4);
IndHopf = 1:3;
IndHopfLeft = IndHopf(abs(1i*omega-EigValMatLeft) == min(abs(1i*omega-EigValMatLeft)));
IndHopf = IndHopf(abs(1i*omega-EigValMat) == min(abs(1i*omega-EigValMat)));
EigVecHopf = EigVecMat(:,IndHopf);
EigVecHopfLeft = EigVecMatLeft(:,IndHopfLeft);
EigValHopf = EigValMat(IndHopf);
if norm(Jacobian*eigvec-1i*omega*eigvec) > 1e-4
    if parameters.restrict
        disp('testing restricted version')
    end
    disp('Eigenvalue or right eigenvector computed incorrectly')
    disp(Jacobian*eigvec)
    disp(1i*omega*eigvec)
end
if norm(Jacobian.'*eigvecleft-1i*omega*eigvecleft) > 1e-4
    disp('Left eigenvector computed incorrectly')
    disp(Jacobian.'*eigvecleft)
    disp(1i*omega*eigvecleft)    
end
if min(norm(abs(eigvec./EigVecHopf)-1),norm(abs(conj(eigvec)./EigVecHopf)-1)) > 1e-4 || min(norm(abs(eigvecleft./EigVecHopfLeft)-1),norm(abs(conj(eigvecleft)./EigVecHopfLeft)-1)) > 1e-4
    if parameters.restrict
        disp('testing restricted version')
    end
    disp('Computation of eigenvectors and eigenvalues or of Jacobian may contain error')
    disp('Eigenvalues from analytical criterion and numerically computed from Jacobian:')
    disp(1i*omega)
    disp(EigValHopf)
    disp('Corresponding eigenvectors:')
    disp(eigvec)
    disp(EigVecHopf)
    disp('Left eigenvectors:')
    disp(eigvecleft)
    disp(EigVecHopfLeft)    
end
IndThird = 1:3;
IndThird = IndThird(abs(thirdeig-EigValMat) == min(abs(thirdeig-EigValMat)));
EigVecThird = EigVecMat(:,IndThird);
EigValThird = EigValMat(IndThird);
if min(norm(abs(eigvec./EigVecHopf)-1),norm(abs(conj(eigvec)./EigVecHopf)-1)) > 1e-4
    if parameters.restrict
        disp('testing restricted version')
    end
    disp('Computation of eigenvectors and eigenvalues or of Jacobian may contain error')
    disp('Eigenvalues from analytical criterion and numerically computed from Jacobian:')
    disp(thirdeig)
    disp(EigValThird)
    disp('Corresponding numerically computed eigenvector:')
    disp(EigVecThird)
end
end