function fout = parallel_spectrum(v_in,parameters,type,flags)
%fout = parallel_spectrum(v_in,parameters,type)
%parallel_spectrum computes eigenvalue spectrum for steady states of the
%general conduits-in-parallel problem
%Computes linearized version of right-hand side of
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N
% V_p*dN/dt = Q_R + (n_c-1)*Q_K - Q_tot
%where
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
%as Jacobian matrix Jac, then finds generalized eigenvalues lambda
%satisfying
% det(Jac - lambda*diag(1 1 V_p)) = 0
%V_p is computed as
% V_p = V_p_0*(uh/uh_bar)^m_store
%
%Input variables:
%   v_in:       3-by-1 vector containing values for [S_R; S_K; N]
%   parameters: parameter structure with fields
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%       V_p_0   Scale factor for storage capacity V_p
%       m_store exponent for computing storage capacity V_p in
%               velocity-dependent form as V_p_0*uh^m_store
%   type:       String, toggles whether V_p is set to zero ('no_storage')
%               or not ('storage'); if neither of these two is specified,
%               code defaults to 'no_storage'
%Output variables
%   fout:       Row vector giving eigenvalues computed by MATLAB function
%               'eig', with any infinite eigenvalues (as will be computed
%               if V_p = 0 or type is set to 'no_storage') removed.

%supply missing flag if needed
if nargin < 4 || ~isfield(flags,'test')
    flags.test = false;
end

%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%check if type is set
if nargin < 3
    type = 'no_storage';
end

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%compute Jacobian
Jac= zeros(3,3);
Jac(1,1) = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
Jac(1,3) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
Jac(2,2) = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
Jac(2,3) = (c_1*dQ_KdPsi/T*Psi + c_1*Q_K/T)*dPsidN-  n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
Jac(3,1) =  dQ_RdS_R;
Jac(3,2) = (n_c-1)*dQ_KdS_K;
Jac(3,3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%compute weight matrix
M = zeros(3,3);
M(1,1) = 1;
M(2,2) = 1;
switch type
    case 'no_storage'
        M(3,3) = 0;
    case 'storage'
        M(3,3) = parameters.V_p_0*storefactor(uh,parameters);
    otherwise warning('Incorrect storage type is set, defaulting to no_storage')
        M(3,3) = 0;
end
fout = eig(Jac,M);
fout = fout.';

%optional elimination of infinite eigenvalues for generalized eigenvalue
%problems V_p = 0)
%if ~strcmp(type,'storage')
%    fout = fout(isfinite(fout));
%end

%test alternative forms
if flags.test
    ftemp = fout;
    clear fout
    fout.numerical_eigs = ftemp;
    switch type
        case 'no_storage'
            parameters.V_p_0 = 0;
            fout.analytical_test = eig_test(ftemp,v_in,parameters);
            fout.analytical_eigs = nostorage_sol(v_in,parameters);
        case 'storage'
            fout.analytical_test = eig_test(ftemp,v_in,parameters);
        otherwise
            warning('Incorrect storage type is set, defaulting to no_storage but using alternative computation of eigenvalues')
            parameters.V_p_0 = 0;
            fout.analytical_test = eig_test(ftemp,v_in,parameters);
            fout.analytical_eigs = nostorage_sol_alt(v_in,parameters);
    end
end

end

function fout = storefactor(uh,parameters)
fout = (uh/parameters.uh_bar).^parameters.m_store;
end

function fout = eig_test(eigvals,v_in,parameters)
%Characteristic polynomial

%unpack parameters; same conventions as in parallel_steady; only V_p_0 is
%new here
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
V_p_0 = parameters.V_p_0;       %Storage capacity parameter
V_p = V_p_0*storefactor(uh,parameters);    %Allow for storage capacity to depend on uh

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%parameters in notebook calculation
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = -c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN + n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = -c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN + n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = -(dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%output --- evaluation of characteristic polynomial, should be 0 at
%eigenvalue
fout = V_p*eigvals.^3 +...
    (dQdN-V_p*(dv_RdS_R+dv_KdS_K))*eigvals.^2 +...
    ((n_c-1)*dQ_KdS_K*dv_KdN + dQ_RdS_R*dv_RdN - dQdN*(dv_RdS_R+dv_KdS_K) + V_p*dv_RdS_R*dv_KdS_K)*eigvals + ...
    dv_RdS_R*dv_KdS_K*dQdN - (n_c-1)*dv_RdS_R*dQ_KdS_K*dv_KdN - dv_KdS_K*dQ_RdS_R*dv_RdN;

end

function fout = nostorage_sol(v_in,parameters)
%Analytically computed eigenvalues for the case V_p = 0

%unpack parameters; same conventions as in parallel_steady
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%parameters in notebook calculation
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = -c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN + n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = -c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN + n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = -(dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

%two terms in solution
a = dv_RdS_R + dv_KdS_K - (dv_RdN*dQ_RdS_R + (n_c-1)*dv_KdN*dQ_KdS_K)/dQdN;
b = ((dv_RdS_R + dv_KdS_K - (dv_RdN*dQ_RdS_R + (n_c-1)*dv_KdN*dQ_KdS_K)/dQdN)^2 - ...
    4*(dv_RdS_R*dv_KdS_K - (dv_KdS_K*dv_RdN*dQ_RdS_R + (n_c-1)*dv_RdS_R*dv_KdN*dQ_KdS_K)/dQdN))^(1/2);

%output
fout = zeros(1,2);
fout(1) = (a+b)/2;
fout(2) = (a-b)/2;

end

function fout = nostorage_sol_alt(v_in,parameters)
%Alternative to no_storage, constructs same 2-by-2 Jacobian but computes
%eigenvalues numerically

%unpack parameters; same conventions as in parallel_steady
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%parameters in notebook calculation
dv_RdS_R = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
dv_KdS_K = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
dv_RdN = -c_1*(dQ_RdPsi*Psi + Q_R)*dPsidN + n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
dv_KdN = -c_1*(dQ_KdPsi/T*Psi + Q_K/T)*dPsidN + n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
dQdN = -(dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;

Jac = zeros(2);
Jac(1,1) = dv_RdS_R - dQ_RdS_R*dv_RdN/dQdN;
Jac(1,2) = -(n_c-1)*dQ_KdS_K*dv_RdN/dQdN;
Jac(2,1) = - dQ_RdS_R*dv_KdN/dQdN;
Jac(2,2) = dv_KdS_K - (n_c-1)*dQ_KdS_K*dv_KdN/dQdN;

fout = eig(Jac)';

end