function [parameters, scales] = parallel_parameter_generator(type)
%[parameters, scales] = parallel_parameter_generator(type)
%Generates parameter structure for use in "parallel" code suite for three
%basic scenarios - a large glacier with a 1-in-50 surface slope, zero bed slope and a
%length of 50 km, with a lake measuring 1000 by 500 m and a water input of
%50 m^3 per second into the lake, or a steeper valley glacier with a
%surface slope of .85-in-5 and a bed slope of .65-in-5, a storage reservoir
%measuring 10 by 5 metres and a water throughput of 3 m^3 per second, or
%the same valley glacier with its length doubled to 10km. The
%standard parameters are c_1 = 1.3455e-9; c_2 =  3.4426e-24; c_3 = 4.05e-2;
%alpha = 5/4; n = 3; as per Rada and Schoof (2018). The basal obstacle
%height is assumed to be .033 m, with the large glacier moving at 100 m per
%year and the valley glacier at 30 m per year. In both cases, we assume a
%single conduit with n_c = 1.
%
%For the valley glacier, only a single scaled set of parameters is
%developed, based on steady flow and balancing all three terms in the
%conduit evolution equation. For the large glacier case, an alternative
%scaling based on the flood stage of the jokulhlaup cycle is also
%available.
%
%Input argument:
%   type:   string defining the glacier case. 'large' and 'valley' give the
%           standard balancing-of-opening-and-closing-rates scales, leaving
%           Q_in, V_p and L as dimensionless parameters, as described in
%           accompanying paper (Schoof, An analysis of instabilities and
%           limit cycles in glacier-dammed reservoirs, 2020)
%           'large_alt' uses the alternative scaling  that sets the opening
%           rate uh due to sliding over bed roughness, the filling rate
%           Q_in and the length L as free parameters, but sets V_p to unity
%   parameters: the corresponding parameter structure for use in the
%           "parallel" code suite
%   scales: structure containing scales for the problem: the variables and
%           parameters in the problem computed with the parameter structure
%           "parameters" are dimensionless, and the scales structure allows
%           easy conversion to dimensional variables. The fields of primary
%           importance in doing so are
%               t:  time scale (in seconds)
%               S:  conduit cross-section scale (in m^2)
%               N:  effective pressure scale (in Pa)
%               x:  intrinsic flow path scale (multiply parameters.L by
%                   scales.x to get dimensional flow path length in metres)
%               q:  flux scale (multiply by dimensionless Q_tot to get
%                   actual input flux; multiply by S^alpha*Psi^(beta-1) to
%                   get actual discharge in channel
%               V:  storage capacity scale, multiply by dimensionless V_p_0
%                   get storage capacity in m^3/Pa
%           other fields are
%               rho_i:  density of ice
%               rho_w:  density of water
%               g:      acceleration due to gravity
%               n:      Glen's law exponent
%               c_1,c_2,c_3:    coefficients in drainage model (see
%                       documentaiton for parallel_evolve_v3.m)
%               alpha:  exponent in Darcy-Weisbach law
%               Psi:    geometrical background hydraulic gradient
%               L:      actual flow path length
%               v_0:    opening rate due to ice flow
%               V_p:    dimensional storage capacity parameter
%                       corresponding to V_p_0 output in parameters
%                       structure. Dimensional storage parameter can be
%                       obtained by multiplying V_p_0 in parameters
%                       structure (for instance, when changed) by scales.V
%               Q_in:   nominal water input to lake; Q_tot value in
%                       parameters structure does not exactly correspond to
%                       this; multiply Q_tot used in parameters structure
%                       by scales.q
%               XXX_star:   dimensionless parameters corresponding to
%                       dimensional value of v_0, V_p, Q_in. v_S_star is a
%                       parameter describing the role of water dissipation
%                       and of storage in the conduit on mass conservation;
%                       not used for this work
%               alt:    alternative scaling parameters corresponding to a
%                       flood stage balance, rendering the system (with n_c
%                       = 1) in the form
%                           dS/dt = S^alpha*Psi + delta - S|N|^{n-1}N
%                           V_p*dN/dt = Q - epsilon
%                           Q = S^alpha*|Psi|^(beta-2)*Psi
%                           Psi = 1- nu*N
%                       alt has subfields S, N, Q, t that give the scales
%                       corresponding to the dimensionless variables in
%                       this formulation of the model (obtained by setting
%                       V_p_0 = 1 as well as uh = delta, Q_tot = epsilon, L
%                       = nu^(-1) in the parameters structure), and the
%                       dimensionless parameters delta, esilon and nu.
%                       These are based on the dimensional flux Q_in,
%                       dimensional storage capacity V_p and dimensional
%                       flow path length specified under the corresponding
%                       fields in "scales" (not scales.alt!). Note that
%                       Q_in may not be the same as the dimensional flux
%                       the is implied by the Q_tot field in parameters,
%                       and epsilon must be adjusted accordingly


if nargin == 0 | (~strcmp(type,'large')&~strcmp(type,'large_alt')&~strcmp(type,'valley')&~strcmp(type,'valley2'))
    type = 'large';
    disp('large glacier standard scaling selected')
end

%common parameters
parameters.n_c = 1;
parameters.c_1 = 1;
parameters.c_2 = 1;
parameters.c_3 = 1;
parameters.Psi_0 = 1;
parameters.uh = 1;
parameters.nu = 1;
parameters.T = 1;
parameters.alpha = 5/4;
parameters.beta = 3/2;
parameters.n_Glen = 3;
parameters.m_store = 0;
parameters.uh_bar = 1;
parameters.v_S = 0;
parameters.S_0_K = 1/eps;
parameters.S_0_R = 1/eps;
switch type
    case 'large'
        parameters.L = 26.9799;
        parameters.V_p_0 = 2*4*4.8086;
        parameters.Q_tot = 22.9337;
    case 'large_alt'
        parameters.L = 1/0.0516;
        parameters.V_p_0 = 1;
        parameters.Q_tot = 0.1581;
        parameters.uh = 0.0069;
    case 'valley'
        parameters.L = 11.0293;
        parameters.V_p_0 = 0.3667;
        parameters.Q_tot = 209.6794;
    case 'valley2'
        parameters.L = 2*11.0293;
        parameters.V_p_0 = 0.3667;
        parameters.Q_tot = 2*209.6794;
end

%scales
switch type
  case 'large'   
    %length
    scales.L = 50000;
    %slope
    scales.theta_surf = 1000/50000;
    %base slope, default is same as surface
    scales.theta_bed =  0;
    %cavity opening rate
    scales.v_0 = 100/(365*24*3600)*.033;
    %Lake storage capacity
    scales.V_p = 2000*2000/(1000*9.8);
    %water input
    scales.Q_in = 10;%.05/24/3600*20e6;
  case 'valley'
    %length
    scales.L = 5000;
    %slope
    scales.theta_surf = 850/5000;
    %base slope, default is same as surface
    scales.theta_bed =  650/5000;
    %cavity opening rate
    scales.v_0 = 30/(365*24*3600)*.033;
    %Lake storage capacity
    scales.V_p = 10*5/(1000*9.8);
    %water input
    scales.Q_in = 3;%.05/24/3600*1000*5000;
  case 'valley2'
    %length
    scales.L = 10000;
    %slope
    scales.theta_surf = 850/5000;
    %base slope, default is same as surface
    scales.theta_bed =  650/5000;
    %cavity opening rate
    scales.v_0 = 30/(365*24*3600)*.033;
    %Lake storage capacity
    scales.V_p = 10*5/(1000*9.8);
    %water input
    scales.Q_in = 3;%.05/24/3600*1000*5000;
end


%constants in model

%basic physical constants
scales.rho_i = 910;
scales.rho_w = 1000;
scales.g = 9.8;
scales.n = 3;
scales.c_1 = 1.3455e-9;
scales.c_2 =  3.4426e-24;
scales.c_3 = 4.05e-2;
scales.alpha = 5/4;

%Hydraulic gradient estimate
scales.Psi = scales.rho_i*scales.g*scales.theta_surf + (scales.rho_w-scales.rho_i)*scales.g*scales.theta_bed;

%water input, either as direct estimate or using melt rate and length,
%width
scales.q = scales.v_0/(scales.c_1*scales.Psi);

%Other scales
scales.S = (scales.q/(scales.c_3*scales.Psi^(1/2)))^(1/scales.alpha);
scales.N = (scales.c_1*scales.q*scales.Psi/(scales.c_2*scales.S))^(1/scales.n);
scales.x = scales.N/(scales.Psi);
scales.t = 1/(scales.c_2*scales.N^scales.n);
scales.V = scales.t*scales.q/scales.N;

%dimensionless parameters
scales.v_0_star = scales.v_0/(scales.c_1*scales.q*scales.Psi);
scales.V_p_star = scales.V_p*scales.N/(scales.t*scales.q);
scales.v_S = scales.c_1*scales.Psi*scales.x;
scales.L_star = scales.L/scales.x;
scales.Q_in_star = scales.Q_in/scales.q;

%alternative scaling
scales.alt.S = (scales.c_1^(scales.n+1)*scales.c_3*scales.Psi^(3/2+scales.n)*scales.V_p^scales.n/scales.c_2).^(1/(1+scales.n-scales.alpha));
scales.alt.N = scales.alt.S/(scales.c_1*scales.V_p*scales.Psi);
scales.alt.q = scales.c_3*scales.alt.S^scales.alpha*scales.Psi^(1/2);
scales.alt.t = scales.V_p*scales.alt.N/scales.alt.q;
scales.alt.epsilon = scales.Q_in/scales.alt.q;
scales.alt.delta = scales.v_0*scales.alt.t/scales.alt.S;
scales.alt.nu = scales.alt.N/(scales.Psi*scales.L);
scales.alt.delta2 = scales.v_0*scales.c_1^(-(scales.n+1+scales.n*scales.alpha)/(scales.n+1-scales.alpha))*scales.c_2^(scales.alpha/(scales.n+1-scales.alpha))*scales.c_3^(-(scales.n+1)/(scales.n+1-scales.alpha))*...
    scales.Psi^(-(3+3*scales.n+2*scales.n*scales.alpha)/(2*(scales.n+1-scales.alpha)))*scales.V_p^(-scales.alpha*scales.n/(scales.n+1-scales.alpha));
scales.alt.epsilon2 = scales.Q_in*scales.c_1^(-(scales.n+1)*scales.alpha/(scales.n+1-scales.alpha))*scales.c_2^(scales.alpha/(scales.n+1-scales.alpha))*scales.c_3^(-(scales.n+1)/(scales.n+1-scales.alpha))*...
    scales.Psi^(-(1+scales.n+2*scales.alpha+2*scales.n*scales.alpha)/(2*(scales.n+1-scales.alpha)))*scales.V_p^(-scales.alpha*scales.n/(scales.n+1-scales.alpha));

%scales.alt.S = (scales.c_1*scales.c_3^(scales.n+1)*scales.Psi^(3/2+scales.n)*scales.V_p^scales.n/scales.c_2).^(1/(1+scales.n-scales.alpha));
%scales.alt.N = scales.alt.S/(scales.c_3*scales.V_p*scales.Psi);
%scales.alt.q = scales.c_1*scales.alt.S^scales.alpha*scales.Psi^(1/2);
%scales.alt.t = scales.V_p*scales.alt.N/scales.alt.q;
%scales.alt.epsilon = scales.Q_in/scales.alt.q;
%scales.alt.delta = scales.v_0*scales.alt.t/scales.alt.S;
%scales.alt.nu = scales.alt.N/(scales.Psi*scales.L);

%solver options
parameters.solver.methods = '15s';
parameters.solver.RelTol = 1e-12;
parameters.solver.AbsTol = 1e-12;
parameters.t_span = [0 500];
end
