function v_out = parallel_initguess(parameters,type)
%v_out = parallel_initguess(parameters,type)
%Computes an asymptotically-valid approximation for the steady state
%solution of the channels-in-parallel problem
% c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N = 0
% c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N = 0
% Q_R + (n_c-1)*Q_K = Q_tot
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% S_R, S_K and N are the variables solved for, all other quantities are
% specified as parameters.
%
%Two different approximations can be computed. One, identified by setting
%the input argument type to 'channel', assumes that S_R is a pure channel,
%satisfying
% c_1*Q_R*Psi_0 - c_2*S_R*|N|^(n_Glen-1)*N = 0
%with Q_R = c_3*S_R^alpha*Psi_0^(beta-2) given by replacing Psi by Psi_0,
%while S_K is a pure cavity, satisfying
% uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N = 0
%with Q_K = c_3*S_K^alpha*T^(1-beta)*Psi_0^(beta-2);
%while the other, with type set to 'cavity' assumes that S_R and S_K are
%pure cavities, satisfying
% nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N = 0
% uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N = 0
%with Psi in the computation of Q_R and Q_K replaced by Psi_0 as in
%the'channel' case
%
%input arguments:
%   parameters: the parameter structure requried by parallel_steady with
%       fields
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%   type:   string argument taking values 'channel' or 'cavity', identifies
%           wich type of approximation is computed
%
%Output argument:
%   v_out:  3-by-1 vector [S_R;S_K;N]

if nargin < 2
    type = 'channel';
end

%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
Q_tot = parameters.Q_tot;   %total discharge through system


switch type
    case 'channel'
        S_R = (Q_tot/c_3/Psi_0^(beta-1))^(1/alpha);
        N = (c_1*Q_tot*Psi_0/c_2/S_R)^(1/n_Glen);
        S_K = uh/(uh/S_0_K+c_2*N^n_Glen);
    case 'cavities'
        S_K = (Q_tot*T^(beta-1)/n_c/c_3/Psi_0^(beta-1))^(1/alpha);
        N = (uh*(1-S_K/S_0_K)/c_2/S_K)^(1/n_Glen);
        S_R = nu*S_K;
    otherwise error('Incorrect type specified in initial guess computation')
end
if N > Psi_0*L
    N = Psi_0*L;
    S_K = uh/(uh/S_0_K+c_2*N^n_Glen);
    S_R = nu*uh/(nu*uh/S_0_K+c_2*N^n_Glen);
    N = L*(Psi_0-(Q_tot/(n_c-1)/c_3/S_K^alpha)^(1/(beta-1)));
    if N < 0, warning('Negative effective pressure computed'), end
end
v_out = [S_R; S_K; N];

end

