function fout = parallel_evolve_v2(v_init,t_span,parameters,solver)
%fout = parallel_evolve_v2(v_init,t_span,parameters,solver)
%Solves time-dependent conduits-in parallel problem, consisting of
% dS_R/dt = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*|N|^(n_Glen-1)*N
% dS_K/dt = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*|N|^(n_Glen-1)*N
% -V_p_0*dN/dt = Q_tot - Q_R - (n_c-1)*Q_K
%with
% Q_R = c_3*S_R^alpha*Psi^(beta-2);
% Q_K = c_3*S_K^alpha*T^(1-beta)*Psi^(beta-2);
% Psi = Psi_0 - N/L;
% S_R, S_K and N are the variables solved for, all other quantities are
% specified as parameters.
%Currently does not have the "(N/N_u)^(-m_slide)" factor implemented,
%corresponding to m_slide = 0;
%
%Input variables are
%   v_init:     3-by-1 vector containing initial condition for [S_R; S_K; N]
%   t_span:     2-by-1 vector specifying initial and final time for
%               integration
%   parameters: optional parameter structure with fields
%       Q_tot   total discharge through system
%       L       length of domain
%       n_c     number of conduits in parallel
%       Psi_0   background hydraulic gradient, used in computation of
%               hydraulic gradient as Psi = Psi_0 - N/L
%       c_1     constant that relates dissipation rate to melt rate as
%               c_1*Q*Psi
%       c_2     constant in computation of closure rate
%               c_2*S*|N|^(n_Glen-1)*N
%       c_3     friction parameter in compute of water discharge as
%               c_3*S^alpha*|Psi|^(beta-2)*Psi
%       alpha   S-exponent in friction law for water discharge
%       beta    Psi-exponent in friction law for water discharge
%       n_Glen  Glen's law exponent
%       uh      Cavity opening rate factor due to sliding over bed roughness
%       S_0_R   Cut-off size for opening of cavities due to sliding
%       S_0_K   As S_0_R but for S_K
%       T       Tortuosity of K-conduits
%       nu      Reduction factor for sliding opening rate for R-conduits
%       V_p_0:  Scale factor for storage capacity
%
%   solver: optional structure with fields
%       methods: string that specifies the matlab ode solver to be used,
%               allows '15s', '23t'  and '45'. Defaults to 15s, and forces
%               15s if V_p_0 = 1 (in which case the system is a DAE system)
%               and '45' or no option is selected (23t can also solve DAEs
%               and is used when specified even if V_p_0 = 0)
%       RelTol: specifies 'RelTol' relative error tolerance passed to
%               matlab ode solver
%       AbsTol: specifies 'AbsTol' absolute error tolerance passed to
%               matlab ode solever
%
%Output struture fout contains the following fields
%   t_sol:      time stamps for solutions
%   v_sol:      length(t_sol)-by-3 array, with each row representing a
%               vector [S_R S_K N] computed at the corresponding entry in
%               t_sol
%   parameters: parameter structure used in calculation, with defaults set
%               for any fields not set in the input variable parameters 
%   solver:     ode solver settings used


%set initial guess if needed
if nargin == 0 || isempty(v_init)
    v_init = [1; 1; 1];
end

%set tyime span if needed
if nargin < 2 || isempty(t_span)
    t_span = [0 10];
end

%set parameters if needed
if nargin < 3 || ~isfield(parameters,'Q_tot')
    parameters.Q_tot = 1;
end
if nargin < 3 || ~isfield(parameters,'L')
    parameters.L = 1/eps;
end
if nargin < 3 || ~isfield(parameters,'n_c')
    parameters.n_c = 10;
end
if nargin < 3 || ~isfield(parameters,'Psi_0')
    parameters.Psi_0 = 1;
end
if nargin < 3 || ~isfield(parameters,'c_1')
    parameters.c_1 = 1;
end
if nargin < 3 || ~isfield(parameters,'c_2')
    parameters.c_2 = 1;
end
if nargin < 3 || ~isfield(parameters,'c_3')
    parameters.c_3 = 1;
end
if nargin < 3 || ~isfield(parameters,'alpha')
    parameters.alpha = 5/4;
end
if nargin < 3 || ~isfield(parameters,'beta')
    parameters.beta = 3/2;
end
if nargin < 3 || ~isfield(parameters,'n_Glen')
    parameters.n_Glen = 3;
end
if nargin < 3 || ~isfield(parameters,'uh')
    parameters.uh = 1;
end
if nargin < 3 || ~isfield(parameters,'S_0_R')
    parameters.S_0_R = 1/eps;
end
if nargin < 3 || ~isfield(parameters,'S_0_K')
    parameters.S_0_K = 1/eps;
end
if nargin < 3 || ~isfield(parameters,'T')
    parameters.T = 1;
end
if nargin < 3 || ~isfield(parameters,'nu')
    parameters.nu = 1;
end
if nargin < 3 || ~isfield(parameters,'V_p_0')
    parameters.V_p_0 = 1;
end

%solver settings
if nargin < 4 || ~isfield(solver,'method') ||  (~strcmp(solver.method,'15s') && ~strcmp(solver.method,'45') && ~strcmp(solver.method,'23t'))
    solver.method = '15s';
end
if nargin < 4 || ~isfield(solver,'RelTol')
    solver.RelTol = 1e-7;
end
if nargin < 4 || ~isfield(solver,'AbsTol')
    solver.AbsTol = 1e-9;
end

%set mass matrix and Jacobian
options = odeset('Mass',@parallel_mass,'MStateDep','none','Jacobian',@parallel_Jac,'RelTol',solver.RelTol,'AbsTol',solver.AbsTol);

%solve, use DAE-compatible solver ode15s as default if V_p is zero, else
%use supplied solver specification
if parameters.V_p_0 == 0 || strcmp(solver.method,'15s') || strcmp(solver.method,'23t')
    if strcmp(solver.method,'23t')
        [t_sol, v_sol] = ode23t(@parallel_step,t_span,v_init,options);
    else    
        [t_sol, v_sol] = ode15s(@parallel_step,t_span,v_init,options);
    end
else
    [t_sol, v_sol] = ode45(@parallel_step,t_span,v_init,options);
end

fout.t_sol = t_sol;
fout.v_sol = v_sol;
fout.parameters = parameters;
fout.solver = solver;

function v_out = parallel_step(t,v_in)
%UNTITLED2 Summary of this function goes here
%   Detailed explanation goes here

%unpack parameters
Q_tot = parameters.Q_tot;   %total discharge
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'

%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*(Psi); 

%output
v_out= zeros(3,1);
v_out(1) = c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R) - c_2*S_R*abs(N)^(n_Glen-1)*N;
v_out(2) = c_1*Q_K/T*Psi + uh*(1-S_K/S_0_K) - c_2*S_K*abs(N)^(n_Glen-1)*N;
v_out(3) =  Q_R + (n_c-1)*Q_K - Q_tot;
end

function J_out = parallel_Jac(t,v_in)
%Jacobian of parallel_Step with regard to input argument v_in

%unpack parameters
L = parameters.L;           %domain length
n_c = parameters.n_c;       %number of conduits in parallel
Psi_0 = parameters.Psi_0;   %reduced potential gradient 
c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to s*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %cavity opening rate
S_0_R = parameters.S_0_R;   %cut-off size for cavity opening for  'channel' element
S_0_K = parameters.S_0_K;   %cut-off size for cavity opening for 'cavity' elements
T = parameters.T;           %tortuosities for 'cavities'
nu = parameters.nu;         %step size ratio for 'channel'
%unpack input
S_R = v_in(1);
S_K = v_in(2);
N = v_in(3);

%auxiliary computations
Psi = Psi_0 - N/L;
dPsidN = -1/L;
Q_R = c_3*S_R^alpha*abs(Psi)^(beta-2)*Psi;
dQ_RdS_R = alpha*Q_R/S_R;
dQ_RdPsi = (beta-1)*Q_R/Psi;
Q_K = c_3*S_K^alpha*T^(1-beta)*abs(Psi)^(beta-2)*Psi; 
dQ_KdS_K = alpha*Q_K/S_K;
dQ_KdPsi = (beta-1)*Q_K/Psi;

%output
J_out= zeros(3,3);
J_out(1,1) = c_1*dQ_RdS_R*Psi - nu*uh/S_0_R - c_2*abs(N)^(n_Glen-1)*N;
J_out(1,3) = (c_1*dQ_RdPsi*Psi+c_1*Q_R)*dPsidN - n_Glen*c_2*S_R*abs(N)^(n_Glen-1);
J_out(2,2) = c_1*dQ_KdS_K/T*Psi - uh/S_0_K - c_2*abs(N)^(n_Glen-1)*N;
J_out(2,3) = (c_1*dQ_KdPsi/T*Psi + c_1*Q_K/T)*dPsidN-  n_Glen*c_2*S_K*abs(N)^(n_Glen-1);
J_out(3,1) =  dQ_RdS_R;
J_out(3,2) = (n_c-1)*dQ_KdS_K;
J_out(3,3) = (dQ_RdPsi + (n_c-1)*dQ_KdPsi)*dPsidN;
end

function M_out = parallel_mass(t,v_in)
V_p = parameters.V_p_0;
M_out = eye(3);
M_out(3,3) = V_p*M_out(3,3);
end

end