function fout = flood(parameters)
%fout = flood(parameters)
%Solution of the leading-order flood stage problem
%   dS/dt = S^alpha*|1-nu*N|^(3/2) - S*|N|^(n_Glen-1)*N
%   dN/dt = - S^alpha*|1-nu*N|^(-1/2)*(1-nu*N)
%with S ~  1/((alpha-1)*(-t))^(1/(alpha-1)) -> 0, N ~ S as t -> -inf
%
%Input argument
%   parameters: optional parameter structure with fields
%       n_Glen, alpha, nu. Defaults to n_Glen = 3, alpha = 5/4, nu = 1/26.9799;
%Output argument
%   fout:   solution structure, with fields
%       S,N,t:  solution vectors for S and N and corresponding time stamps
%       N_f:    the unique far-field value of lim_(t->inf) N that
%               determines the length of the recharge phase
%       parameters: parameter structure

if nargin == 0
    parameters.alpha = 5/4;
    parameters.beta = 3/2;
    parameters.n_Glen = 3;
    parameters.nu = 1/26.9799;
end

%parameters
alpha = parameters.alpha;

%initial conditions
tmin=-1e4;tmax=1e4;
Smin = 1/((alpha-1)*(-tmin))^(1/(alpha-1));
Nmin = Smin;
vin = [Smin;Nmin];

%setup
options = odeset('RelTol',1e-12,'AbsTol',1e-12);
evolvefun = @(t,v)evolvefunction(t,v,parameters);

%solve for orbit
[tout,vout] = ode45(evolvefun,[tmin,tmax],vin,options);

fout.S = vout(:,1);
fout.N = vout(:,2);
fout.t = tout;
fout.N_f = fout.N(end);
fout.parameters = parameters;
end


function fout = evolvefunction(t,v,params)

%unpack parameters
n_Glen = params.n_Glen;
alpha = params.alpha;
beta = params.beta;
nu = params.nu;

%dependent variables
S = v(1); N = v(2);

fout = zeros(2,1);
fout(1) = S^alpha*abs(1-nu*N)^beta - S*abs(N)^(n_Glen-1)*N;
fout(2) = S^alpha*abs(1-nu*N)^(beta-2)*(1-nu*N);

end