function [xout, error_flag, faux] = ArcLength_single_symmetric(f,x,ds,scalefactor,scale,direction,npts,parameters,srchparams)
%[xout, error_flag, faux] = ArcLength_single_symmetric(f,x,ds,scalefactor,scale,direction,npts,parameters,srchparams)
%Arc length continuation method for finding zeros for function f(x) where
%l is part of the set of parameters admitted by the function. In ArcLength_single_symmetric, a call to f outputs both f(x)
%and the gradient function Df(x) with respect to x. ArcLength_single
%computes derivatives with respect to l by divided differences, making it
%easier to apply where f and Df are known.
%
%   Input arguments:
%   f:      function handle of function whose roots are to be found, along
%           with its derivative with respect to x
%           If ArcLength_single requires one or two output arguments
%           f must take the form
%               [fout, Dfout] = f(x,parmaeters)
%           where Dfout is the Jacobian of f with respect to x
%           f must also produce the single output fout if called in the
%           form
%               fout = f(x,parameters)
%           If ArcLength_single requires three output arguments, f must take the form
%               [fout, Dfout, faux] = f(x,parameters)
%           but still produce the single output fout if called in the form
%               fout = f(x,parameters)
%           The Newton iteration will solve for fout=0; faux can be
%           additional outputs computed within f that may be used
%           elsewhere. 
%   x:      initial guess for x; must have correct dimensions for an input
%           argument of f. f must be formulated so that x is an n-by-1
%           column vector, and the output of f must be a 1-by-n vector fout
%           and a n-by-n+1 matrix Dfout
%   ds:     prescribed arc length step, default is 0.1
%   scalefactor: (diagonal) metric for computation of arc length, takes the
%           form of a n-by-1 vector [a_1 a_2 ....] so that arc length is
%           computed as \sum_i a_i dv_i^2 if scale is 'linear', and as
%           \sum_i a_i d(log(v_i))^2 if scale is 'log', with v_i being the
%           concatenated vector [x; l], and n being the length of the
%           vector x. Default is the identity metric ones(length(x),1)
%   scale: sets whether step size computed from change in parameters
%          ('linear', default) or from change in log(parameter) ('log').
%          default is 'linear'
%   direction:  initial direction for changes in last parameter, choices +1 and -1,
%           default is +1
%   npts:       number of solution points to be found
%   parameters: optional parameter structure to be passed to f,
%   srchparams: optional structure containing search parameters for Newton
%               iteration. This can contain the fields:
%               itmax:  maximum number of iterations
%               toldelta:   error bound on x; last Newton step must be <=
%                           0.5*toldelta
%               tolgrad:    error bound on df = 0; 0.5*norm(df,2) <=
%                           tolgrad to terminate iteration
%               verbose:    0 produces no runtime output
%                           1 displays how many steps in the continuation
%                           have been taken
%                           2 displays also step size and current value of
%                           function f
%                           3 displays all of the above and also displays
%                           current iterate
%               delta:      step size for finite difference computation of
%                           derivatives
%
%   The algorithm used is Newton's method as a default; intend to upgrade
%   to line search algorithm
%
%   Output:
%   xout:       n-by-npts array roots of f=0
%   error_flag: npts-by-one vector of booleans, returns logical one if
%               convergence to prescribed tolerance has not been achieved
%   faux:       npts-by-one cell array of optional third outputs of
%               function f, if auxiliary calculations within f generate
%               additional output
%
%Works as ArcLength_single_v2 but does not make an intrinsic distinction
%between input x and parameter l; also requires f to output a Jacobian
%matrix of dimensions (n-1)-by-n where n = length(x)
%
%Tested; suggest making computation of initial point with subfunction
%"redirect" work such that auxiliary function computations can be done for
%that point
%
%Christian Schoof, 20 May 2014
%File header corrected 8 November 2015
%Output of auxiliary argument corrected 19 February 2016

    %set up Newton iteration and finite differencing parameters
    if nargin < 9
        srchparams.itmax = 25;
        srchparams.tolF = sqrt(eps);
        srchparams.toldelta = sqrt(eps);
        srchparams.verbose = 0;         
    else
        if ~isfield(srchparams,'itmax')
            srchparams.itmax = 25;
        end
        if ~isfield(srchparams,'tolF')
            srchparams.tolF = sqrt(eps);
        end
        if ~isfield(srchparams,'toldelta')
            srchparams.toldelta = sqrt(eps);
        end
        if ~isfield(srchparams,'verbose')
            srchparams.verbose = 0;
        end
    end
    
    itmax = srchparams.itmax;
    tolF = srchparams.tolF;
    toldelta = srchparams.toldelta;
    verbose = srchparams.verbose;
    
    %Deal with missing input parameters
    if nargin < 3 || isempty(parameters)
        parameters.noparameters = true;
    end
    
    %%initialize arc length parameters
    if nargin < 4 || isempty(ds)
        ds = 1e-2;
    end
    if nargin < 5 || isempty(scalefactor)
        scalefactor = ones(size(x));
    end
    if nargin < 6 || isempty(scale)
        scale = 'linear';
    end
    if nargin < 7 || isempty(direction)
        direction = +1;
    end
    
    %intialize output
    xout = zeros(length(x),npts);
    if nargout > 2
        error_flag = false(npts,1);
    end
    if nargout > 3
        faux = cell(npts,1);
    end
    
    %compute first point
    if isfield(parameters,'noparameters') && parameters.noparameters
        falt = @(x_in,parameters) redirect(x_in,x(end),parameters,f,1);
        Dfalt = @(x_in,parameters) redirect(x_in,x(end),parameters,f,2);
        [xtemp, errortemp] = Newton(falt,Dfalt,x(1:end-1),parameters,srchparams);
    else
        falt = @(x_in,parameters) redirect(x_in,x(end),parameters,f,1);
        Dfalt = @(x_in,paramters) redirect(x_in,x(end),parameters,f,2);
        [xtemp, errortemp] = Newton(falt,Dfalt,x(1:end-1),parameters,srchparams);
    end
    xout(:,1) = [xtemp;x(end)];
    if nargout > 1
        error_flag(1) = errortemp;
    end
    if nargout > 2
        [fjunk Dfjunk faux{1}] = f(xout(:,1),parameters);
    end
    
    %set up arc length continuation
    v = [xtemp;x(end)];  %initial point
    dv = zeros(length(x),1);
    %first increment
    switch scale
        case 'linear'
            dv(end) = ds*direction/scalefactor(end)^(1/2);
        case 'log'
            dv(1:end-1) = 1;
            dv(end) = exp(direction*ds/scalefactor(end)^(1/2));
    end
    for ii=2:npts
        if srchparams.verbose, disp(strcat('Computing point #',num2str(ii))), end
        %increment initial guess based on previous increment
        vprev = v;
        switch scale
            case 'linear'
                v = vprev+dv;
            case 'log'
                v = vprev.*dv;
        end
        %initialize Newton iteration
        errortemp = false;
        F = zeros(size(v));
        DF = zeros(length(v));
        if isfield(parameters,'noparameters') && parameters.noparameters
            if nargout < 3
                [F(1:end-1), DFtemp] = f(v);
            else
                [F(1:end-1), DFtemp, fauxtemp] = f(v);
            end            
        else
            if nargout < 3
                [F(1:end-1), DFtemp] = f(v,parameters);
            else
                [F(1:end-1), DFtemp, fauxtemp] = f(v,parameters);
            end
        end
        DF(1:end-1,:) = DFtemp;
        switch scale
            case 'linear'
                F(end) = norm((v-vprev).*(scalefactor.^(1/2))) - ds;
                DF(end,:) = (v-vprev).*scalefactor./norm((v-vprev).*(scalefactor.^(1/2)));
            case 'log'
                F(end) = norm((log(v)-log(vprev)).*(scalefactor.^(1/2))) - ds;
                DF(end,:) = (log(v)-log(vprev)).*scalefactor./v./norm((log(v)-log(vprev)).*(scalefactor.^(1/2)));
        end        
        if verbose > 1, disp(strcat('Initial norm of f =',num2str(norm(F,2)))), end
        Dv = toldelta + eps;
        iter = 1;
        %Newton iteration
        while iter < itmax && (norm(F) >= tolF || norm(Dv) >= toldelta)
            if verbose == 3 && length(v) <= 5, disp(strcat('Jacobian matrix DF = ',num2str(DF))), end
            Dv = -DF\F;
            v = v + Dv;
            if norm(Dv) < toldelta, break, end
            if isfield(parameters,'noparameters') && parameters.noparameters
                if nargout < 3
                    [F(1:end-1), DFtemp] = f(v);
                else
                    [F(1:end-1), DFtemp, fauxtemp] = f(v);
                end                
            else
                if nargout < 3
                    [F(1:end-1), DFtemp] = f(v,parameters);
                else
                    [F(1:end-1), DFtemp, fauxtemp] = f(v,parameters);
                end
            end
            DF(1:end-1,:) = DFtemp;
            switch scale
                case 'linear'
                    F(end) = norm((v-vprev).*(scalefactor.^(1/2))) - ds;
                    DF(end,:) = (v-vprev).*scalefactor./norm((v-vprev).*(scalefactor.^(1/2)));
                case 'log'
                    F(end) = norm((log(v)-log(vprev)).*(scalefactor.^(1/2))) - ds;
                    DF(end,:) = (log(v)-log(vprev)).*scalefactor./v./norm((log(v)-log(vprev)).*(scalefactor.^(1/2)));
            end 
            if verbose > 1, disp(strcat('Updated norm of f =',num2str(norm(F,2)))), disp(strcat('Size of step Dv =',num2str(norm(Dv,2)))),end
            iter = iter + 1;
            if verbose == 3, disp('Current iterate ='), disp(v), end
        end
        %output warning, set error_flag if convergence not achieved
        if iter >= itmax, warning('Newton did not converge'), if nargout >= 2, errortemp = true; end, end
        xout(:,ii) = v;
        if nargout > 1
            error_flag(ii) = errortemp;
            if nargout > 2
                faux{ii} = fauxtemp;
            end
        end
        %compute incement to produce initial guess for next point
        if ii<npts
            switch scale
                case 'linear'
                    dv = v-vprev;
                case 'log'
                    dv = v./vprev;
            end
        end
    end
end

function fout = redirect(x,xend,parameters,f,argout)
%output for f that makes sure Jacobian Dfout is not overdimensioned
if isfield(parameters,'noparameters') && parameters.noparameters
    [ftemp Dftemp] = f([x;xend]);
else
    [ftemp Dftemp] = f([x;xend],parameters);
end
switch argout
    case 1
        fout = ftemp;
    case 2
        fout = Dftemp(:,1:end-1);
    otherwise
        error('cannot choose output argument > 2')
end
end 