function fout = network_stability_v5(v_in,parameters)
%fout = network_stability_v5(fin,parameters)
%Computes spectrum of the dynamical system that results from linearizing 
%network-type drainage system described in Rada and Schoof (The Cryopshere,
%2018), or alternatively, from linearizing the discretized form of the
%one-dimensional continuum drainage system in Schoof and others (The 
%Cryopshere, 2014) or Schoof (2020).
%
%Let S_R be the cross-sectional area of a generic drainage element defined
%on the edges of a network as in Schoof (Ice sheet acceleration driven by
%melt supply variability, Nature, 2010), evolving as (equation #1)
% dS_R / dt - ( c_1*Q_R*Psi + nu*uh*(1-S_R/S_0_R -
%           c_2*S_R*|N_edge|^(n_Glen-1)*N_edge ) = 0
%where N_edge is the mean of effective pressure N at the neighbouring
%nodes, and the discharge Q_R along the edge from its upstream end at index
%"up_node" to its downstream end at index "down_node" is
% Q_R = c_3*heaviside(S_R-S_P_R)*((S_R-S_P_R).^2+epsilon_P_R^2)^(1/2)-epsilon_P_R)^alpha*|Psi_abs|^(beta-2)*Psi
% and
% Psi = (N(up_node) - N(down_node) + Phi_0(down_node) - Phi_0(up_node))/L,
% Psi_abs = sqrt(Psi*2+epsilon^2)
%are the (signed) hydraulic gradient and its regularized magnitude along
%the edge.
%Let S_K be the cross-sectional area of a drainage element along the same
%edge that experiences no enlargement due to dissipation-driven melting.
%S_K evolves according to (equation #2)
% dS_K / dt - ( uh*(1-S_K/S_0_K) - c_2*S_K*|N_edge|^(n_Glen-1)*N_edge ) = 0
% Discharge in the S_K conduits is given by
% Q_K = c_3*heaviside(S_K-S_P_K)*((S_K-S_P_K).^2+epsilon_P_K^2)^(1/2)-epsilon_P_K)^alpha*|Psi_abs_K|^(beta-2)*Psi_K
%where Psi_K = Psi/T and Psi_abs_K = sqrt(Psi_K^2+epsilon^2). T is the
%tortuosity of the S_K conduit.
%S_R and S_K are defined on the same network edges. In addition, we permit a leakage flux as
%Q_leak = k_leak*Psi
%The total discharge along an edge is
%Q = Q_R + (n_c-1)*Q_K+Q_leak
%assuming n_c-1 cavity-type conduits and one "potential channel" conduit
%
%Conservation of mass at each node gives (constraint #3)
% d(storage at node) / dt + 1/2* gamma_S * sum (L*(dS_R/dt+(n_c-1)*T*dS_K/dt)) -  ( q_in +
% sum Q_in - sum Q_out - sum L*melt) = 0
%where Q_in is  discharges along an edge into the node, and Q_out is
%discharge along an edge out of the node, and the sum over dS_R/dt nd
%S_K/dt is taken over all edges into or out of the node. "melt" is given by
% 1/2*gamma_S*r*L*Q*Psi
% and the sum is taken over all edges into or out of a node; r is the
% ice-to-water density ratio.
%Storage at a node has an 'elastic' componennt gamma_store*(N-N_0), where
%N_0 is a local constant
%
%If model flag named Kamb_storage is set tu true, there is also storage in 
%a cavity of size V_K(t). This cavity evolves according to (constraint #4)
%dV_K / dt - ( uh_K*(1-V_K/V_0_K) - c_3*V_K*|N|^(n_Glen-1)*N ) = 0
%
%The dynamical system can be written in the form
% M*dv/dt = F(v)
%where v is a concatenated vector of values of S_R and S_K on the network
%edges, N and potentially V_K (if Kamb_storage is set) at network nodes. M
%is a mass matrix, and F corresponds to the right-hand side of the
%evolution equations above. There will generally be ghost nodes at the edge
%of the domain on which Dirichlet conditions on N are set; these ghost node
%values do not form part of the vector v.
%
%The time evolution of this dynamical system can be modelled using the
%network_precolation_v2 code included in the supplementary material of Rada
%and Schoof (Channelized, distributed, and disconnected: subglacial
%drainage under a valley glacier in the Yukon, The Cryosphere, 2018). The
%present code simply solves for the eigenvalues and eigenvectors of the
%linearized system
%lambda*M*v' = J(v_in)*v'
%where J is the Jacobian of of the right-hand side F evaluated at some
%state vector v_in
%
%Input variables are:
%v_in:          Concatenated column vectors for S_K, S_R, N and
%               (optionally) V_K
%parameters:    Parameter structure with the following fields
%   grid:       substructure with fields
%               n_nodes: number of nodes in network
%               n_edges: number of network edges
%               up_node: list of upstream nodes for each edge
%               down_node: list of downstream nodes for each edge
%                   (up_node and down_node together give the data expected
%                   in a connectivity array)
%               x,y:    notional coordinates of grid nodes; these are not
%               used to compute edge lengths, however (see L below)
%               bdy_nodes: list of nodes on the domain boundary, where
%               Dirichlet conditions are to be applied
%   dt:         Size of time step
%   L:          List (n_edges-by-one vector) of edge lengths
%   n_c:        List (n_edges-by-one vector) of conduits per edge
%   Phi_0:      Reduced hydraulic potential at each node (n_nodes-by-one
%               vector), given by rho_i g s + (rho_w-rho_i) g b, s being
%               local ice surface elevation and b local bed elevation at
%               the node. rho_i and rho_w are ice and water density and g
%               is acceleration due to gravity
%   N_bdy_nodes:Effective pressure at boundary nodes (column vector with
%               same dimensions as bdy_nodes)
%   c_1:        Melting term is c_1*Q*Psi, where Q is discharge and Psi
%               hydraulic gradient along each network edge
%   c_2:        Creep closure is c_2*S*N^n_Glen, where S is conduit
%               cross-section or storage cavity size, N is effective
%               pressure
%   c_3:        Coefficient in Manning or Darcy-Weisbach law, discharge is
%               Q = c_3*S^alpha*Psi^(beta-1)
%   alpha:      Exponent relating conduit cross-section to discharge
%   beta:       Exponent relating Hydraulic gradient to discharge
%   n_Glen:     Glen's law exponent in creep closure relation
%   epsilon:    Regualrizer for the square-root dependence of discharge on
%               hydraulic gradient
%   uh:         List (n_edges-by-one vector) of 'cavity' opening rates for
%               each edge
%   S_0_R:      cut-off size for cavity opening term for R conduits
%   S_0_K:      cut-off size for cavity opening term for pure cavity K
%               conduits
%   V_0_K:      cut-off size for cavity opening term for storage cavity K
%               terms
%   S_P_R:      percolation cut-off size for R conduits
%   S_P_K:      percolation cut-off size for K conduits
%   epsilon_P_R: regularizer for percolation cut-off for R conduits
%   epsilon_P_K: regularizer for precolation cut-off for K conduits
%   T:          List (n_edges-by-one vector) of 'cavity' tortuosities along
%               each edge 
%   nu:         List (n_edges-by-one vector) of reduction factors in
%               opening rates from 'cavity' to  'channel' conduits for each
%               edge
%   gamma_store:List (n_nodes-by-one vector) of 'compressibiities' of
%               storage system at each node
%   gamma_S:    List(n_nodes-y-one-vector) of parameters defining contribution of storage in conduits to mass balance at each node
%   r:          ice to water density ratio
%   k_leak:     leakage permeability
%   q_in:       List (n_nodes-by-one vector) of water supply rates at each
%               node
%   Kamb_storage:Flag that sets whether a Kamb-type storage cavity volume
%               needs to be solved for at each node
%   uh_K:       List (n_nodes-by-one vector) of storage cavity opening
%               rates at each node
%
%The vectors S_R and S_K in v_in have dimensions n_edges-by-one, N and
%(optionally) V_K are n_nodes-by-one, and v_in = [S_R; S_K; N; V_K] if
%Kamb_storage is true, v_in = [S_R; S_K,N] otherwise. S_R and S_K are
%defined on network edges, N and V_K are defined at nodes.                
%
%Output:
%fout:  output structure containing fields
%   parameters:         the parameter structure used (see input variables)
%   S_R, S_K, N, V_K:   vectors containing the values S_R,
%                       S_K,N,V_K (if Kamb_storage is used) specified by
%                       the input vector v_in
%   mass_matrix:        the mass matrix M of the dynamical system
%   Jacobian:           the Jacobian J of the dynamical system
%   eigval:             a vector containing the eigenvalues of the problem
%   eigvec:             a matrix containing the eigenvectors of the
%                       problem, output only if parameters contains a field
%                       'eigfun' set to 'true'

%extract data
n_nodes = parameters.grid.n_nodes;   %number of nodes
n_edges = parameters.grid.n_edges;   %number of edges
bdy_nodes = parameters.grid.bdy_nodes;

S_R = v_in(1:n_edges);
S_K = v_in(n_edges+1:2*n_edges);
N = v_in(2*n_edges+1:2*n_edges+n_nodes);
N(bdy_nodes) = parameters.N_bdy_nodes;
if parameters.Kamb_storage
    V_K = v_in(2*n_edges+n_nodes+1:2*n_edges+2*n_nodes);  %optional Kamb-style storage cavity at node
end

%Set up water supply rate
parameters.q_in = qfun(parameters,0);


%Set up output
fout.parameters = parameters;
fout.S_R = S_R;
fout.S_K = S_K;
fout.N = N;
if parameters.Kamb_storage
    fout.V_K = V_K;
end

%compute eigenvalues
[Jac, mass] = network_Jacobian(v_in,parameters);
if isfield(parameters,'eigfun') && parameters.eigfun
    [fout.eigvec, fout.eigval] = eig(full(Jac),full(mass));
    fout.Jacobian = Jac;
    fout.mass_matrix = mass;
else
    fout.eigval = eig(full(Jac),full(mass));
end
end

function qout = qfun(parameters,t)
%compute water supply
if parameters.qscaled
    qout = parameters.q_in_0*parameters.q_0;
else
    qout = parameters.q_in_0;
end
end


function [foutJac foutMass] = network_Jacobian(v_in,parameters)
%Jacobian of right-hand side of dynamical system and mass matrix

%unpack parameters
n_nodes = parameters.grid.n_nodes;   %number of nodes
n_edges = parameters.grid.n_edges;   %number of edges
up_node = parameters.grid.up_node;   %list (n_edges-by-1 vector) of 'upstream' node for each edge
down_node = parameters.grid.down_node;   %list (n_edges-by-1 vector) of 'downstream' nodes for each edge
bdy_nodes = parameters.grid.bdy_nodes;   %list (n_bdy_nodes-by-1 vector) of nodes at domain boundary

L = parameters.L;           %list (n_edges-by-1 vector) of edge lengths
n_c = parameters.n_c;       %list (n_edges-by-1 vector) of number of conduits along edge
Phi_0 = parameters.Phi_0;   %reduced hydraulic potential rho_i s + (rho_w-rho_i) b at each node (n_nodes-by-1 vector) 
N_bdy_nodes = parameters.N_bdy_nodes;   %Dirichlet conditions (n_bdy_nodes-by-1 vector) for effective pressure at domain boundary

c_1 = parameters.c_1;       %relates opening rate to Q*Psi
c_2 = parameters.c_2;       %relates closure rate to S*N*n
c_3 = parameters.c_3;       %relates discharge Q to S^alpha*Psi^beta
alpha = parameters.alpha;   %exponent in dependence of discharge Q on cross-section S
beta = parameters.beta;     %exponent in dependence of discharge Q on hydraulic gradient Psi, same convention as in Schoof et al 2012 / Hewitt et al 2012 so 'frozen-time' problenm for N only becomes beta-Laplacian (i.e. p-Laplacian with p=beta)
epsilon = parameters.epsilon;   %regularization parameter for hydraulic gradient
n_Glen = parameters.n_Glen; %Glen's law exponent
uh = parameters.uh;         %list (n_edges-by-1 vector) of cavity opening rates for each network edge
S_0_R = parameters.S_0_R;       %cut-off size for cavity opening
S_0_K  = parameters.S_0_K;  %cut-off size for Kamb cavity opening
V_0_K  = parameters.V_0_K;  %cut-off size for storage cavity opening
S_P_R = parameters.S_P_R;   %percolation cut-off size for R conduit
S_P_K = parameters.S_P_K;   %percolation cut-off size for K conduit
epsilon_P_R = parameters.epsilon_P_R;   %percolation cut-off regularizer for R-conduits;
epsilon_P_K = parameters.epsilon_P_K;   %percolation cut-off regularizer for K-conduits;
T = parameters.T;           %list (n_edges-by-1 vector) of tortuosities for 'cavities' along each network edge
nu = parameters.nu;         %list (n_edges-by-1 vector) of step size ratios for 'channel' along each network edge
gamma_S = parameters.gamma_S;   %contribution of conduit evolution to mass balance
gamma_store = parameters.gamma_store;   %list (n_nodes-by-1 vector) of 'ealstic' water storage capacities at each node (i.e. storage is affine in N with coefficient gamma_store)
k_leak = parameters.k_leak; %scalar leakage permeability
r = parameters.r;           %ice-to-water density ratio

Kamb_storage = parameters.Kamb_storage;     %Boolean that determines whether Kamb-style storage cavities are included at each node
if Kamb_storage
    uh_K = parameters.uh_K; %list (n_nodes-bv-1 vector) of cavity opening rates for storage cavities at each node
end

%unpack input variable v_in
if (Kamb_storage && length(v_in) ~= 2*(n_edges+n_nodes)) || (~Kamb_storage && length(v_in) ~= 2*n_edges+n_nodes), error('input size incompatible with network geometry'), end
                                                        %check input size
S_R = v_in(1:n_edges);
S_K = v_in(n_edges+1:2*n_edges);
N = v_in(2*n_edges+1:2*n_edges+n_nodes);
N(bdy_nodes) = N_bdy_nodes;
if Kamb_storage
    V_K = v_in(2*n_edges+n_nodes+1:2*n_edges+2*n_nodes);  %optional Kamb-style storage cavity at node
end

%compute hydraulic gradient and derivatives with respect to N's along each network edge
Psi = (Phi_0(up_node) - N(up_node) - Phi_0(down_node) + N(down_node))./L;
dPsidN_up = -1./L;
dPsidN_down = 1./L;

%same for regularized absolute values of hydraulic gradient
Psi_abs_R = (Psi.^2+epsilon.^2).^(1/2);
Psi_abs_K = (Psi.^2./T.^2+epsilon.^2).^(1/2);
dPsi_abs_RdPsi = Psi./Psi_abs_R;
dPsi_abs_KdPsi = Psi./(T.^2.*Psi_abs_K);

%compute effective pressure and derivatives with respect to N's along each network edge
N_edge = (N(up_node)+N(down_node))/2;
dN_edgedN_up = 1/2;
dN_edgedN_down = 1/2;

%compute 'channel' flux and derivatives along each edge
Q_R = c_3*(((S_R-S_P_R).^2+epsilon_P_R.^2).^(1/2)-epsilon_P_R).^alpha.*Psi_abs_R.^(beta-2).*Psi;
Q_R(S_R<S_P_R)=0;
%Q_R = c_3*S_R.^alpha.*abs(Psi).^(beta-2).*Psi;
dQ_RdS_R = alpha*c_3*(((S_R-S_P_R).^2+epsilon_P_R.^2).^(1/2)-epsilon_P_R).^(alpha-1).*(S_R-S_P_R)./((S_R-S_P_R).^2+epsilon_P_R.^2).^(1/2)   ...
    .*Psi_abs_R.^(beta-2).*Psi;
dQ_RdS_R(S_R<S_P_R)=0;
dQ_RdPsi = (beta-2).*Q_R.*dPsi_abs_RdPsi./Psi_abs_R ...
    + c_3*(((S_R-S_P_R).^2+epsilon_P_R.^2).^(1/2)-epsilon_P_R).^alpha.*Psi_abs_R.^(beta-2);
dQ_RdPsi(S_R<S_P_R)=0;

%compute 'cavity' flux and derivatives along each edge
Q_K = c_3*(((S_K-S_P_K).^2+epsilon_P_K.^2).^(1/2)-epsilon_P_K).^alpha.*Psi_abs_K.^(beta-2).*Psi./T;
Q_K(S_K<S_P_K)=0;
%Q_K = c_3*S_K.^alpha.*abs(Psi./T).^(beta-2).*Psi./T;
dQ_KdS_K = alpha*c_3*(((S_K-S_P_K).^2+epsilon_P_K.^2).^(1/2)-epsilon_P_K).^(alpha-1).*(S_K-S_P_K)./((S_K-S_P_K).^2+epsilon_P_K.^2).^(1/2)   ...
    .*Psi_abs_K.^(beta-2).*Psi./T;
dQ_KdS_K(S_K<S_P_K)=0;
dQ_KdPsi = (beta-2).*Q_K.*dPsi_abs_KdPsi./Psi_abs_K ...
    + c_3*(((S_K-S_P_K).^2+epsilon_P_K.^2).^(1/2)-epsilon_P_K).^alpha.*Psi_abs_K.^(beta-2)./T;
dQ_KdPsi(S_K<S_P_K)=0;

%compute channel opening rate derivatives
dfout1dS_R = spdiags(-c_1*dQ_RdS_R.*Psi + nu.*uh./S_0_R + c_2*abs(N_edge).^(n_Glen-1).*N_edge,0,n_edges,n_edges);
dfout1dN = sparse(1:n_edges,up_node,-c_1*dQ_RdPsi.*Psi.*dPsidN_up-c_1*Q_R.*dPsidN_up+c_2*n_Glen*S_R.*abs(N_edge).^(n_Glen-1).*dN_edgedN_up,n_edges,n_nodes) ...
    +sparse(1:n_edges,down_node,-c_1*dQ_RdPsi.*Psi.*dPsidN_down-c_1*Q_R.*dPsidN_down+c_2*n_Glen*S_R.*abs(N_edge).^(n_Glen-1).*dN_edgedN_down,n_edges,n_nodes);
%mass matrix
dfout1MassdS_R = -speye(n_edges);
%correct for Dirichlet conditions
dfout1dN(:,bdy_nodes) = 0;

%compute cavity opening rate derivatives
dfout2dS_K = spdiags(uh./S_0_K + c_2*abs(N_edge).^(n_Glen-1).*N_edge,0,n_edges,n_edges);
dfout2dN = sparse(1:n_edges,up_node,c_2*n_Glen*S_K.*abs(N_edge).^(n_Glen-1).*dN_edgedN_up,n_edges,n_nodes) ...
    +sparse(1:n_edges,down_node,c_2*n_Glen*S_K.*abs(N_edge).^(n_Glen-1).*dN_edgedN_down,n_edges,n_nodes);
%mass matrix
dfout2MassdS_K = -speye(n_edges);
%correct for Dirichlet conditions
dfout2dN(:,bdy_nodes) = 0;

%compute mass conservation derivatives
dfout3dN =  sparse(up_node,up_node,(dQ_RdPsi+(n_c-1).*dQ_KdPsi+k_leak).*dPsidN_up,n_nodes,n_nodes) ...
    +  sparse(up_node,down_node,(dQ_RdPsi+(n_c-1).*dQ_KdPsi+k_leak).*dPsidN_down,n_nodes,n_nodes) ...
    - sparse(down_node,up_node, (dQ_RdPsi+(n_c-1).*dQ_KdPsi+k_leak).*dPsidN_up,n_nodes,n_nodes) ...
    - sparse(down_node,down_node,(dQ_RdPsi+(n_c-1).*dQ_KdPsi+k_leak).*dPsidN_down,n_nodes,n_nodes)...
    - sparse(up_node,up_node,r/2*gamma_S(up_node).*L.*(dQ_RdPsi.*Psi+Q_R).*dPsidN_up,n_nodes,n_nodes)...
    - sparse(up_node,down_node,r/2*gamma_S(up_node).*L.*(dQ_RdPsi.*Psi+Q_R).*dPsidN_down,n_nodes,n_nodes)...
    - sparse(down_node,up_node,r/2*gamma_S(down_node).*L.*(dQ_RdPsi.*Psi+Q_R).*dPsidN_up,n_nodes,n_nodes)...
    - sparse(down_node,down_node,r/2*gamma_S(down_node).*L.*(dQ_RdPsi.*Psi+Q_R).*dPsidN_down,n_nodes,n_nodes);
dfout3dS_R = sparse(up_node,1:n_edges,dQ_RdS_R,n_nodes,n_edges) - sparse(down_node,1:n_edges,dQ_RdS_R,n_nodes,n_edges) ...
    - sparse(up_node,1:n_edges,r/2*gamma_S(up_node).*L.*dQ_RdS_R.*Psi,n_nodes,n_edges) - sparse(down_node,1:n_edges,r/2*gamma_S(down_node).*L.*dQ_RdS_R.*Psi,n_nodes,n_edges);
dfout3dS_K = sparse(up_node,1:n_edges,(n_c-1).*dQ_KdS_K,n_nodes,n_edges) - sparse(down_node,1:n_edges,(n_c-1).*dQ_KdS_K,n_nodes,n_edges);
%mass matrix
dfout3MassdN = spdiags(gamma_store,0,n_nodes,n_nodes);
dfout3MassdS_R = -1/2*(sparse(up_node,1:n_edges,L.*gamma_S(up_node),n_nodes,n_edges) + sparse(down_node,1:n_edges,L.*gamma_S(down_node),n_nodes,n_edges));
dfout3MassdS_K = -1/2*(sparse(up_node,1:n_edges,L.*T.*(n_c-1).*gamma_S(up_node),n_nodes,n_edges) + sparse(down_node,1:n_edges,L.*T.*(n_c-1).*gamma_S(down_node),n_nodes,n_edges));
%correct for Dirichlet conditions
dfout3dN(:,bdy_nodes) = 0;
dfout3dN(bdy_nodes,:) = 0;
dfout3dN = dfout3dN + sparse(bdy_nodes,bdy_nodes,ones(length(bdy_nodes),1),n_nodes,n_nodes);
dfout3dS_R(bdy_nodes,:) = 0;
dfout3dS_K(bdy_nodes,:) = 0;
%correct mass matrix boundary nodes
dfout3MassdN(:,bdy_nodes) = 0;
dfout3MassdN(bdy_nodes,:) = 0;
dfout3MassdS_R(bdy_nodes,:) = 0;
dfout3MassdS_K(bdy_nodes,:) = 0;

%correct for Kamb storage
if Kamb_storage
    dfout4dV_K =spdiags(uh_K./V_0_K + c_3*abs(N).^(n_Glen-1).*N,0,n_nodes,n_nodes);
    dfout4dN = spdiags(c_3*n_Glen*V_K.*abs(N).^(n_Glen-1),0,n_nodes,n_nodes);
    %mass matrix
    dfout3MassdV_K = -speye(n_nodes);
    dfout4MassdV_K = -speye(n_nodes); 
    %correct for Dirichlet conditions
    dfout4dN(:,bdy_nodes) = 0;
    %correct mass matrix
    dfout3MassdV_K(n_bdy_nodes,:) = 0; 
end

%Assemble final Jacobian and weight matrices
if ~Kamb_storage
    foutJac = [dfout1dS_R sparse(n_edges,n_edges) dfout1dN; ...
        sparse(n_edges,n_edges) dfout2dS_K dfout2dN; ...
        dfout3dS_R dfout3dS_K dfout3dN];
    foutMass = [dfout1MassdS_R sparse(n_edges,n_edges) sparse(n_edges,n_nodes); ...
        sparse(n_edges,n_edges) dfout2MassdS_K sparse(n_edges,n_nodes); ...
        dfout3MassdS_R dfout3MassdS_K dfout3MassdN];
else
    foutJac = [dfout1dS_R sparse(n_edges,n_edges) dfout1dN sparse(n_edges,n_nodes); ...
        sparse(n_edges,n_edges) dfout2dS_K dfout2dN sparse(n_edges,n_nodes); ...
        dfout3dS_R dfout3dS_K dfout3dN sparse(n_nodes,n_nodes);...
        sparse(n_nodes,2*n_edges) dfout4dN dfout4dV_K];
    foutMass = [dfout1MassdS_R sparse(n_edges,n_edges) sparse(n_edges,n_nodes) sparse(n_edges,n_nodes); ...
        sparse(n_edges,n_edges) dfout2MassdS_K sparse(n_edges,n_nodes) sparse(n_edges,n_nodes); ...
         dfout3MassdS_R dfout3MassdS_K dfout3MassdN dfout3MassdV_K; ...
        sparse(n_nodes,2*n_edges+n_nodes) dfout4MassdV_K];
end

end