def dataset():
    import netCDF4 as nc4

    sheba_dir = '/data/local/hadax/SHEBA/'
    filename = 'surf_obs.nc'
    ffile = sheba_dir + filename
    ncid = nc4.Dataset(ffile)

    return ncid
    

def convert_Jdd_to_datetime(Jdd):
    import datetime as dt
    import numpy as np
    
    tiny = 1.e-4
    
    ref_dt = dt.date(1997,1,1).toordinal()
    
    integer_part = np.floor(Jdd)
    fractional_part = Jdd - integer_part
    
    actual_day_ordinals = integer_part + ref_dt
    
    actual_days = [dt.date.fromordinal(int(do)) for do in actual_day_ordinals]
    
    hours = np.floor((fractional_part + tiny) * 24.)
    
    actual_dts = [dt.datetime(ddate.year,ddate.month,ddate.day,int(hr)) for (hr,ddate) in zip(hours,actual_days)]
    
    return actual_dts


def mask(array):
    import numpy.ma as ma
    missing_logical = (array == 9999.) | (array == 999.)
    out_array = ma.masked_array(array,mask=missing_logical)
    return out_array


def bin_by_month(array,dts):
    import numpy as np
    import numpy.ma as ma
    
    month_array = np.array([dt.month for dt in dts])
    year_array  = np.array([dt.year for dt in dts])
    both_array = zip(year_array,month_array)
    both_nparray = np.array(both_array)
    both_set = set(both_array)
    nset = len(both_set)
    both_list = list(both_set)
    both_list.sort()
    
    return_values = ma.zeros(nset)
    for (i,(year,month)) in enumerate(both_list):
        idate = np.where((both_nparray[:,0] == year) * (both_nparray[:,1] == month))
	
        value = ma.mean(array[idate])

        return_values[i] = value
	
    return return_values


def mid_month_times():
    import numpy as np
    import datetime as dt

    months = ((np.arange(13) + 9) % 12) + 1
    years  = ((np.arange(13) + 9) / 12) + 1997

    return_dts = [dt.date(yr,mon,15) for (yr,mon) in zip(years,months)]
    return return_dts


def plot(ncid,varname):
    import matplotlib.pyplot as plt

    Jdd = ncid.variables['Jdd'][:]
    datetimes = convert_Jdd_to_datetime(Jdd)
    mmts = mid_month_times()
    
    timeseries = ncid.variables[varname][:]
    masked_timeseries = mask(timeseries)
    
    masked_timeseries_month = bin_by_month(masked_timeseries,datetimes)
    
    fig = plt.figure(figsize = (20,8))
    plt.plot(datetimes,masked_timeseries,color='#00FFFF')
    plt.plot(mmts,masked_timeseries_month,color='#000080')
    fig.autofmt_xdate()
    plt.show()


def record_seb():
    import iris
    import datetime as dt
    import numpy as np

    ref_date = dt.date(1978,9,1)
    ref_date_units = ref_date.strftime('days since %Y-%m-%d')

    outdir = '/data/cr1/hadax/SHEBA/SFC_over_ice/'
    tcode = 'Nov1997.Sep1998'
    rcode = 'SHEBA.actual_track'
    ncid = dataset()
    datetimes = convert_Jdd_to_datetime(ncid.variables['Jdd'][:])
    mSWd_mask = bin_by_month(mask(ncid.variables['SWd'][:]),datetimes)
    mSWu_mask = bin_by_month(mask(ncid.variables['SWu'][:]),datetimes)
    mLWd_mask = bin_by_month(mask(ncid.variables['LWd'][:]),datetimes)
    mLWu_mask = bin_by_month(mask(ncid.variables['LWu'][:]),datetimes)
    mhs_mask = bin_by_month(mask(ncid.variables['hs'][:]),datetimes)
    mhl_mask = bin_by_month(mask(ncid.variables['hl'][:]),datetimes)
    mmts = mid_month_times()
    month_points = np.array([sheba_date.month for sheba_date in mmts])

    mSWn_mask = mSWd_mask - mSWu_mask
    mLWn_mask = mLWd_mask - mLWu_mask
    
    mSWu_mask = 0. - mSWu_mask
    mLWu_mask = 0. - mLWu_mask

    indices = ((np.arange(12) + 2) % 12) + 1

    cubelist = iris.cube.CubeList([])
    cubelist.append(iris.cube.Cube(mSWd_mask[indices],\
        long_name='SW_down_over_ice'))
    cubelist.append(iris.cube.Cube(mSWu_mask[indices],\
        long_name='SW_up_over_ice'))
    cubelist.append(iris.cube.Cube(mSWn_mask[indices],\
        long_name='SW_net_over_ice'))
    cubelist.append(iris.cube.Cube(mLWd_mask[indices],\
        long_name='LW_down_over_ice'))
    cubelist.append(iris.cube.Cube(mLWu_mask[indices],\
        long_name='LW_up_over_ice'))
    cubelist.append(iris.cube.Cube(mLWn_mask[indices],\
        long_name='LW_net_over_ice'))
    cubelist.append(iris.cube.Cube(mhs_mask[indices],\
        long_name='Sensible_heat_flux_over_ice'))
    cubelist.append(iris.cube.Cube(mhl_mask[indices],\
        long_name='Latent_heat_flux_over_ice'))

    month_coord = iris.coords.DimCoord(month_points[indices],\
        'time',units=ref_date_units,long_name='month')
    for cube in cubelist:
        cube.add_dim_coord(month_coord,0)    
        outfile = outdir+cube.long_name+'.'+tcode+'.'+rcode+'.nc'
        cube.units = 'W/m^2'
        iris.save(cube,outfile)
    
